import React from 'react';
import { PropertyFilters as FilterType } from '../types';
import { availableCities } from '../utils/mockData';
import { Search, Filter, Home, MapPin } from 'lucide-react';

interface PropertyFiltersProps {
  filters: FilterType;
  onFiltersChange: (filters: FilterType) => void;
}

const PropertyFilters: React.FC<PropertyFiltersProps> = ({ filters, onFiltersChange }) => {
  const handleFilterChange = (key: keyof FilterType, value: string) => {
    onFiltersChange({
      ...filters,
      [key]: value
    });
  };

  return (
    <div className="flex flex-wrap gap-3 items-center">
      {/* Search */}
      <div className="relative">
        <input
          type="text"
          placeholder="Search properties..."
          value={filters.search}
          onChange={(e) => handleFilterChange('search', e.target.value)}
          className="w-64 pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
        />
        <Search className="absolute left-3 top-2.5 w-5 h-5 text-gray-400" />
      </div>

      {/* Status Filter */}
      <div className="relative">
        <Filter className="absolute left-3 top-2.5 w-4 h-4 text-gray-400 pointer-events-none" />
        <select
          value={filters.status}
          onChange={(e) => handleFilterChange('status', e.target.value)}
          className="pl-10 pr-8 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent appearance-none bg-white"
        >
          <option value="all">All Status</option>
          <option value="active">Active</option>
          <option value="sold">Sold</option>
          <option value="rented">Rented</option>
        </select>
      </div>

      {/* Listing Intent Filter */}
      <div className="relative">
        <Home className="absolute left-3 top-2.5 w-4 h-4 text-gray-400 pointer-events-none" />
        <select
          value={filters.listingIntent}
          onChange={(e) => handleFilterChange('listingIntent', e.target.value)}
          className="pl-10 pr-8 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent appearance-none bg-white"
        >
          <option value="all">All Types</option>
          <option value="sale">For Sale</option>
          <option value="rent">For Rent</option>
          <option value="urgent-sale">Urgent Sale</option>
        </select>
      </div>

      {/* City Filter */}
      <div className="relative">
        <MapPin className="absolute left-3 top-2.5 w-4 h-4 text-gray-400 pointer-events-none" />
        <select
          value={filters.city}
          onChange={(e) => handleFilterChange('city', e.target.value)}
          className="pl-10 pr-8 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent appearance-none bg-white"
        >
          <option value="">All Cities</option>
          {availableCities.map(city => (
            <option key={city} value={city}>{city}</option>
          ))}
        </select>
      </div>
    </div>
  );
};

export default PropertyFilters;