import React from "react";
import { Card, CardContent } from "./ui/card";
import { Badge } from "./ui/badge";
import { Button } from "./ui/button";
import {
  MapPin,
  Bed,
  Bath,
  Square,
  Eye,
  Edit,
  Pause,
  Trash2,
  Play,
} from "lucide-react";
import { useAzureBlobMedia } from "../../../../hooks/useAzureBlobMedia";
import { useLegalHealth } from "../../../../hooks/useLegalHealth";
import { LegalHealthStatus } from "../../../../components/legal/LegalHealthStatus";
import type { Property as PropertyRow } from "../types";

interface SellerPropertyCardProps {
  property: PropertyRow;
  onEdit: (property: PropertyRow) => void;
  onPause: (property: PropertyRow) => void;
  onDelete: (property: PropertyRow) => void;
  onView: (property: PropertyRow) => void;
  onListForSale?: (property: PropertyRow) => void;
}

const SellerPropertyCard: React.FC<SellerPropertyCardProps> = ({
  property,
  onEdit,
  onPause,
  onDelete,
  onView,
  onListForSale,
}) => {
  console.log("🆕 NEW SELLER PROPERTY CARD RENDERED 🆕", property.id);

  // Azure Blob Storage integration
  const { images: azureImages, loading: mediaLoading } = useAzureBlobMedia(
    property.id.toString(),
  );

  // Legal health integration
  const {
    legalHealth,
    loading: legalLoading,
    error: legalError,
  } = useLegalHealth(property.id);

  // Debug logging
  console.log("Legal Health Debug:", {
    propertyId: property.id,
    legalHealth,
    legalLoading,
    legalError,
  });

  const getImageUrl = () => {
    if (azureImages && azureImages.length > 0) {
      return azureImages[0].url; // Use first Azure image
    }
    return "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=250";
  };

  const formatPrice = (price: number) => {
    if (!price) return "Price on request";
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(1)} Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(0)} L`;
    return `₹${price.toLocaleString()}`;
  };

  const getStatusColor = (status: string, isasset?: boolean) => {
    if (isasset) {
      // For assets that are listed for sale
      if (property.listing_status === "listed" && status === "active") {
        return "bg-blue-100 text-blue-800"; // Blue for listed assets
      }
      if (property.listing_status === "listed" && status === "pending") {
        return "bg-yellow-100 text-yellow-800"; // Yellow for pending approval
      }
      if (status === "approved" && property.listing_status !== "listed") {
        return "bg-green-100 text-green-800"; // Green for approved but not listed
      }
      if (status === "pending") return "bg-yellow-100 text-yellow-800";
      if (status === "sold") return "bg-purple-100 text-purple-800";
      return "bg-gray-100 text-gray-800";
    }
    if (status === "pending") return "bg-yellow-100 text-yellow-800";
    switch (status) {
      case "active":
        return "bg-green-100 text-green-800";
      case "sold":
        return "bg-purple-100 text-purple-800";
      case "rented":
        return "bg-indigo-100 text-indigo-800";
      case "under_review":
        return "bg-blue-100 text-blue-800";
      case "approved":
        return "bg-green-100 text-green-800";
      case "inactive":
        return "bg-orange-100 text-orange-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getStatusLabel = (status: string, isasset?: boolean) => {
    if (isasset) {
      // For assets, check listing_status to determine if they are listed for sale
      if (property.listing_status === "listed" && status === "active") {
        return "Listed for Sale";
      }
      if (property.listing_status === "listed" && status === "pending") {
        return "Pending Approval";
      }
      if (status === "approved" && property.listing_status !== "listed") {
        return "Not Listed";
      }
      if (status === "pending") return "Pending Approval"; // This should not be visible in manage properties
      if (status === "sold") return "Sold";
      return "Asset";
    }
    if (status === "pending") return "Pending Approval";
    switch (status) {
      case "active":
        return "Active";
      case "sold":
        return "Sold";
      case "rented":
        return "Rented";
      case "under_review":
        return "Under Review";
      case "approved":
        return "Approved";
      case "inactive":
        return "Inactive";
      default:
        return status;
    }
  };

  // More robust asset detection
  const isasset =
    property.isasset === true ||
    (property.listing_status === "not_listed" &&
      property.status === "approved");

  console.log("🔍 ASSET DETECTION DEBUG:", {
    propertyId: property.id,
    "property.isasset": property.isasset,
    "typeof isasset": typeof property.isasset,
    isasset_final: isasset,
    status: property.status,
    listing_status: property.listing_status,
  });

  const isNotListed =
    isasset &&
    property.status === "approved" &&
    property.listing_status !== "listed";

  // Debug logging for ALL properties
  console.log("🔍 PROPERTY DEBUG:", {
    propertyId: property.id,
    isasset: property.isasset,
    status: property.status,
    listing_status: property.listing_status,
    isasset_check: isasset,
    isNotListed_check: isNotListed,
    onListForSale_exists: !!onListForSale,
    shouldShowButton: isNotListed && !!onListForSale,
    note: "Assets with pending status should not be visible in manage properties",
  });

  return (
    <Card
      className="overflow-hidden hover:shadow-lg transition-shadow cursor-pointer border-2 border-blue-200"
      onClick={() => onView(property)}
    >
      <div className="relative">
        <img
          src={getImageUrl()}
          alt={property.title}
          className="w-full h-48 object-cover"
          onError={(e) => {
            e.currentTarget.src =
              "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=250";
          }}
        />
        <div className="absolute top-3 left-3">
          <Badge className={getStatusColor(property.status, isasset)}>
            {getStatusLabel(property.status, isasset)}
          </Badge>
        </div>
        <div className="absolute top-3 right-3 flex gap-1">
          <Button
            variant="outline"
            size="sm"
            onClick={(e) => {
              e.stopPropagation();
              onEdit(property);
            }}
            className="bg-white/90 hover:bg-white"
          >
            <Edit className="h-3 w-3" />
          </Button>
          <Button
            variant="outline"
            size="sm"
            onClick={(e) => {
              e.stopPropagation();
              onPause(property);
            }}
            className="bg-white/90 hover:bg-white"
          >
            {property.status === "inactive" ? (
              <Play className="h-3 w-3" />
            ) : (
              <Pause className="h-3 w-3" />
            )}
          </Button>
          <Button
            variant="outline"
            size="sm"
            onClick={(e) => {
              e.stopPropagation();
              onDelete(property);
            }}
            className="bg-white/90 hover:bg-white"
          >
            <Trash2 className="h-3 w-3 text-red-500" />
          </Button>
        </div>
      </div>

      <CardContent className="p-4">
        <div className="space-y-3">
          <div>
            <h3 className="font-semibold text-lg text-gray-900 line-clamp-1">
              {property.title}
            </h3>
            <div className="flex items-center text-gray-600 text-sm mt-1">
              <MapPin className="w-4 h-4 mr-1" />
              <span className="line-clamp-1">
                {property.city}, {property.state}
              </span>
            </div>
          </div>

          <div className="flex items-center justify-between text-sm text-gray-600">
            <div className="flex items-center space-x-4">
              <div className="flex items-center">
                <Bed className="w-4 h-4 mr-1" />
                <span>{property.bhk || "N/A"}</span>
              </div>
              <div className="flex items-center">
                <Bath className="w-4 h-4 mr-1" />
                <span>{property.bathrooms || "N/A"}</span>
              </div>
              <div className="flex items-center">
                <Square className="w-4 h-4 mr-1" />
                <span>{property.area} sqft</span>
              </div>
            </div>
          </div>

          <div className="flex items-center justify-between">
            <div className="text-xl font-bold text-blue-600">
              {isasset ? "Asset" : formatPrice(property.price)}
            </div>
            <div className="flex items-center text-gray-500 text-sm">
              <Eye className="w-4 h-4 mr-1" />
              <span>{property.views || 0} views</span>
            </div>
          </div>

          {/* Legal Health Indicator - Always Show */}
          <div className="mt-2">
            {legalHealth ? (
              <div className="flex items-center gap-2">
                <LegalHealthStatus
                  legalHealth={legalHealth}
                  showDetails={false}
                  userRole="seller"
                />
                {legalHealth.completion_percentage < 100 && (
                  <button
                    onClick={(e) => {
                      e.stopPropagation();
                      window.location.href = `/seller/dashboard/legal-documents?propertyId=${property.id}`;
                    }}
                    className="px-4 py-2 bg-[#0056D2] text-white text-sm font-bold rounded-lg hover:bg-[#004BB8] transition-colors shadow-lg border-2 border-[#0056D2]"
                  >
                    📄 Upload Docs
                  </button>
                )}
              </div>
            ) : legalLoading ? (
              <div className="inline-flex items-center gap-1 px-2 py-1 rounded-full text-xs border bg-gray-50 text-gray-600 border-gray-200">
                <div className="w-3 h-3 border border-gray-400 border-t-transparent rounded-full animate-spin"></div>
                <span>Loading legal status...</span>
              </div>
            ) : (
              <div className="flex items-center gap-2">
                <div className="inline-flex items-center gap-1 px-2 py-1 rounded-full text-xs border bg-yellow-50 text-yellow-800 border-yellow-200">
                  <span>⚠️ Legal: No data</span>
                </div>
                <button
                  onClick={(e) => {
                    e.stopPropagation();
                    window.location.href = `/seller/dashboard/legal-documents?propertyId=${property.id}`;
                  }}
                  className="px-4 py-2 bg-[#0056D2] text-white text-sm font-bold rounded-lg hover:bg-[#004BB8] transition-colors shadow-lg border-2 border-[#0056D2]"
                >
                  📄 Upload Docs
                </button>
              </div>
            )}
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

export default SellerPropertyCard;
