import React from 'react';
import { Property } from '../types';
import { MapPin } from 'lucide-react';
import { useAzureBlobMedia } from '../../../../hooks/useAzureBlobMedia';

interface WebPropertyCardProps {
  property: Property;
  onEdit: (id: string) => void;
  onDelete: (id: string) => void;
  onView: (id: string) => void;
}

const WebPropertyCard: React.FC<WebPropertyCardProps> = ({
  property,
  onEdit,
  onDelete,
  onView
}) => {
  // Azure Blob media integration
  const {
    images: azureImages,
  } = useAzureBlobMedia(property.id.toString());

  // Process images using same logic as PropertyDetail
  const getPropertyImage = () => {
    // Prioritize Azure Blob images if available
    if (azureImages && azureImages.length > 0) {
      return azureImages[0].url;
    }
    
    // Fallback to legacy images
    const processImageUrl = (url: string) => {
      if (!url) return null;
      if (url.startsWith('http')) return url;
      return `http://localhost:8090${url.startsWith('/') ? url : '/' + url}`;
    };
    
    // Check various image fields
    if ((property as any).photo) {
      return processImageUrl((property as any).photo);
    }
    
    if (property.images && property.images.length > 0) {
      const firstImg = property.images[0];
      if (typeof firstImg === 'string') {
        return processImageUrl(firstImg);
      } else if (firstImg && typeof firstImg === 'object' && 'url' in firstImg) {
        return processImageUrl((firstImg as any).url);
      }
    }
    
    if ((property as any).primary_image) {
      return processImageUrl((property as any).primary_image);
    }
    
    return "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=200";
  };

  const propertyImage = getPropertyImage();
  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'bg-green-500';
      case 'inactive': return 'bg-gray-500';
      case 'under_review': return 'bg-amber-500';
      default: return 'bg-green-500';
    }
  };

  return (
    <div className="bg-white rounded-lg border border-gray-200 hover:border-primary/30 hover:shadow-lg transition-all duration-200 overflow-hidden">
      {/* Image Section */}
      <div className="relative h-40 overflow-hidden">
        <img
          alt={property.title}
          className="w-full h-full object-cover"
          src={propertyImage}
          onError={(e) => {
            console.error('Web property card image failed to load:', propertyImage);
            e.currentTarget.src = "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=200";
          }}
        />
        <div className={`absolute top-2 left-2 ${getStatusColor(property.status)} text-white text-xs font-medium px-2 py-1 rounded`}>
          {property.status === 'active' ? 'For Sale' : property.status}
        </div>
      </div>

      {/* Content Section */}
      <div className="p-4">
        <h3 className="text-base font-semibold text-gray-900 mb-1 line-clamp-1">{property.title}</h3>
        
        <div className="flex items-center text-gray-500 text-sm mb-2">
          <MapPin className="w-3 h-3 mr-1" />
          <span className="line-clamp-1">{property.city}, {property.state}</span>
        </div>
        
        <div className="text-lg font-bold text-primary mb-3">
          ₹{(property.price / 100000).toFixed(1)}L
          {property.listingIntent === 'rent' && <span className="text-sm font-normal"> / month</span>}
        </div>
        
        {/* Property Details */}
        <div className="flex items-center justify-between text-xs text-gray-500 mb-3 pb-3 border-b border-gray-100">
          {property.bhk && (
            <span>{property.bhk}</span>
          )}
          <span>{property.area} sqft</span>
          <span>{new Date(property.createdAt).toLocaleDateString('en-GB')}</span>
        </div>
        
        {/* Action Buttons */}
        <div className="flex gap-2">
          <button 
            onClick={() => onView(property.id)}
            className="flex-1 bg-primary text-white py-2 px-3 rounded text-sm font-medium hover:bg-primary/90 transition-colors"
          >
            View Details
          </button>
          <button 
            onClick={() => onDelete(property.id)}
            className="px-3 py-2 border border-red-200 text-red-600 rounded text-sm font-medium hover:bg-red-50 transition-colors"
          >
            Delete
          </button>
        </div>
      </div>
    </div>
  );
};

export default WebPropertyCard;