import { useState, useRef, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { Bell, User, Settings, LogOut, Eye, X } from "lucide-react";
import { getUserNotifications, markNotificationAsRead, Notification } from "../../../../../api/notifications";
import { RoleSwitcher } from '../../../../RoleSwitcher';
import { getCampaignsByUserId } from "../../../../../api/campaigns";
import { getCurrentUserUuid } from "../../../../../utils/userUtils";

function SellerNotificationBell({ userId }: { userId: string }) {
  const [isOpen, setIsOpen] = useState(false);
  const [notifications, setNotifications] = useState<Notification[]>([]);
  const [unreadCount, setUnreadCount] = useState(0);
  const [isLoading, setIsLoading] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const navigate = useNavigate();

  const fetchNotifications = async () => {
    try {
      setIsLoading(true);
      const data = await getUserNotifications(userId);

      let mergedNotifications = [...(data.notifications || [])];
      let extraUnread = 0;

      const currentUserUuid = getCurrentUserUuid();
      if (currentUserUuid) {
        try {
          const campaignResponse = await getCampaignsByUserId(currentUserUuid, { status: 'approved' });
          const approvedCampaigns = campaignResponse?.campaigns || [];
          const approvedCount = approvedCampaigns.length;

          const lastSeenKey = `campaignApprovedLastSeen:${currentUserUuid}`;
          const lastSeenCount = Number(localStorage.getItem(lastSeenKey) || '0');
          const isRead = approvedCount <= lastSeenCount;

          if (approvedCount > 0) {
            mergedNotifications.unshift({
              id: `campaign-approved-${currentUserUuid}`,
              type: 'admin_action',
              title: 'Campaigns Approved',
              message: `You have ${approvedCount} approved campaign${approvedCount > 1 ? 's' : ''}. Start tracking performance now.`,
              time: new Date().toISOString(),
              read: isRead,
              data: { count: approvedCount, status: 'approved' }
            });
          }

          if (!isRead) {
            extraUnread = 1;
          }
        } catch (error) {
          console.error('Error fetching campaign approvals:', error);
        }
      }

      setNotifications(mergedNotifications);
      setUnreadCount((data.unread_count || 0) + extraUnread);
    } catch (error) {
      console.error('Error fetching notifications:', error);
    } finally {
      setIsLoading(false);
    }
  };

  useEffect(() => {
    fetchNotifications();
    const interval = setInterval(fetchNotifications, 60000);
    return () => clearInterval(interval);
  }, [userId]);

  useEffect(() => {
    if (isOpen) fetchNotifications();
  }, [isOpen]);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setIsOpen(false);
      }
    };
    if (isOpen) {
      document.addEventListener('mousedown', handleClickOutside);
      return () => document.removeEventListener('mousedown', handleClickOutside);
    }
  }, [isOpen]);

  return (
    <div className="relative" ref={dropdownRef}>
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="relative inline-flex items-center justify-center w-10 h-10 p-1 rounded-full hover:bg-gray-100 transition"
      >
        <Bell className="w-5 h-5 text-gray-700" />
        {unreadCount > 0 && (
          <span className="absolute top-0 right-0 flex items-center justify-center h-5 min-w-[20px] px-1.5 text-[8px] font-semibold text-white bg-red-600 rounded-full border-2 border-white">
            {unreadCount > 99 ? '99+' : unreadCount}
          </span>
        )}
      </button>

      {isOpen && (
        <div className="absolute right-0 mt-2 w-96 bg-white rounded-2xl shadow-xl border border-gray-100 z-50">
          <div className="px-6 py-4 bg-gradient-to-r from-blue-50 to-indigo-50 border-b border-gray-100">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-3">
                <div className="w-8 h-8 bg-blue-500 rounded-lg flex items-center justify-center">
                  <Bell className="w-4 h-4 text-white" />
                </div>
                <div>
                  <h3 className="text-lg font-semibold text-gray-900">Notifications</h3>
                  <p className="text-xs text-gray-600">
                    {unreadCount === 0 ? "All caught up!" : `${unreadCount} unread`}
                  </p>
                </div>
              </div>
              <button
                onClick={() => setIsOpen(false)}
                className="p-2 text-gray-400 hover:text-gray-600 hover:bg-white/50 rounded-lg transition-all"
              >
                <X className="w-4 h-4" />
              </button>
            </div>
          </div>

          <div className="max-h-80 overflow-y-auto">
            {isLoading ? (
              <div className="p-8 flex justify-center">
                <div className="animate-spin rounded-full h-6 w-6 border-2 border-blue-500 border-t-transparent"></div>
              </div>
            ) : notifications.length === 0 ? (
              <div className="p-8 text-center">
                <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-3">
                  <Bell className="w-6 h-6 text-gray-400" />
                </div>
                <p className="text-sm text-gray-500">No notifications yet</p>
              </div>
            ) : (
              <div className="divide-y divide-gray-50">
                {notifications.map((notification) => {
                  let timeStr = 'Just now';
                  try {
                    if (notification.time) {
                      const date = new Date(notification.time);
                      if (!isNaN(date.getTime())) {
                        timeStr = new Intl.DateTimeFormat('en-US', {
                          month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit'
                        }).format(date);
                      }
                    }
                  } catch (e) {
                    timeStr = 'Just now';
                  }

                  return (
                    <div 
                      key={notification.id} 
                      className="p-4 hover:bg-gray-50 transition-colors cursor-pointer" 
                      onClick={async () => {
                        try {
                          if (notification.id.startsWith('campaign-approved-')) {
                            const currentUserUuid = getCurrentUserUuid();
                            if (currentUserUuid && notification.data?.count) {
                              localStorage.setItem(
                                `campaignApprovedLastSeen:${currentUserUuid}`,
                                String(notification.data.count)
                              );
                            }
                          }
                          if (!notification.read) {
                            await markNotificationAsRead(notification.id);
                            setNotifications(prev => 
                              prev.map(n => n.id === notification.id ? { ...n, read: true } : n)
                            );
                            setUnreadCount(prev => Math.max(0, prev - 1));
                          }
                          if (notification.data?.negotiation_id) {
                            navigate(`/seller/dashboard/negotiations`);
                            setIsOpen(false);
                          }
                        } catch (error) {
                          console.error('Error handling notification:', error);
                        }
                      }}
                    >
                      <div className="flex items-start justify-between gap-3">
                        <div className="flex-1 min-w-0">
                          <div className="flex items-center gap-2 mb-1">
                            <h4 className="text-sm font-semibold text-gray-900">
                              {notification.title || 'Notification'}
                            </h4>
                            {!notification.read && (
                              <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
                            )}
                          </div>
                          <p className="text-sm text-gray-600 mb-2">
                            {notification.message || 'No message'}
                          </p>
                          <span className="text-xs text-gray-400">{timeStr}</span>
                        </div>
                      </div>
                    </div>
                  );
                })}
              </div>
            )}
          </div>
        </div>
      )}
    </div>
  );
}

export function SellerDashboardHeader() {
  const [notifications] = useState([
    { id: 1, message: "New property inquiry received", time: "2 min ago" },
    { id: 2, message: "Your property valuation is ready", time: "1 hour ago" },
    { id: 3, message: "Property listing approved", time: "3 hours ago" },
    { id: 4, message: "New message from buyer", time: "5 hours ago" },
    { id: 5, message: "Document verification complete", time: "1 day ago" },
  ]);

  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const [isNotificationOpen, setIsNotificationOpen] = useState(false);
  const [showAllNotifications, setShowAllNotifications] = useState(false);
  const navigate = useNavigate();

  const toggleDropdown = () => {
    setIsDropdownOpen(!isDropdownOpen);
  };

  const handleLogout = () => {
    localStorage.clear();
    sessionStorage.clear();
    window.location.replace('/');
  };

  const goToSettings = () => {
    navigate("/seller/dashboard/settings");
  };

  const toggleNotifications = () => {
    setIsNotificationOpen(!isNotificationOpen);
    setShowAllNotifications(false);
  };

  const visibleNotifications = showAllNotifications ? notifications : notifications.slice(0, 3);
  const hasMoreNotifications = notifications.length > 3;

  const getUserEmail = () => {
    // Use original email if switching dashboards, otherwise use current
    const originalEmail = sessionStorage.getItem('original_email');
    if (originalEmail) return originalEmail;
    return localStorage.getItem("sellerEmail") || "User";
  };

  return (
    <header className="bg-white shadow-sm border-b px-8 py-4">
      <div className="flex items-center justify-between">
        <h1 className="text-xl font-semibold text-gray-900">Welcome back, {getUserEmail()}!</h1>
        <div className="flex items-center gap-4">
          <RoleSwitcher currentRole="seller" />
          
          <SellerNotificationBell userId={(() => {
            const sellerId = localStorage.getItem('sellerId') || localStorage.getItem('seller_id');
            const sellerEmail = localStorage.getItem('sellerEmail');
            return sellerId || sellerEmail || 'seller';
          })()} />
          <button
            onClick={handleLogout}
            className="text-gray-600 hover:text-gray-800"
          >
            Logout
          </button>
        </div>
      </div>
    </header>
  );
}
