import {
  Building,
  Users,
  BarChart3,
  ChevronDown,
  ChevronRight,
  ChevronLeft,
  Home,
  FileText,
  TrendingUp,
  DollarSign,
  Settings,
  Upload,
  Calculator,
  Target,
  Globe,
  Megaphone,
  MessageSquare,
  FileSignature,
  Shield,
  Activity,
  Monitor,
  Briefcase,
  Zap,
  CheckCircle,
} from "lucide-react";

import { Link, useLocation } from "wouter";
import { cn } from "../../lib/utils";
import { useState } from "react";
const nalLogo = "/src/assets/logo.jpg";

type NavigationItem = {
  name: string;
  href?: string;
  icon: any;
  children?: NavigationItem[];
};

const navigation: NavigationItem[] = [
  { name: "Dashboard", href: "/", icon: Home },
  { name: "Manage Property", href: "/properties", icon: Building },
  { name: "Asset Management", href: "/assets", icon: Briefcase },
  { name: "Campaigns", href: "/campaigns", icon: Megaphone },
  { name: "Negotiations", href: "/negotiations", icon: MessageSquare },
  { name: "Booking & Scheduling", href: "/booking-scheduling", icon: Users },
  { name: "Leads & Inquiries", href: "/leads", icon: Users },
  { name: "Reports", href: "/reports", icon: FileText },
  { name: "Property Analytics", href: "/property-analytics", icon: Activity },
  { name: "Auction / Bidding", href: "/auction-bidding", icon: BarChart3 },
  { name: "Marketing Tools", href: "/marketing-tools", icon: TrendingUp },
  { name: "Valuation Tools", href: "/valuation", icon: Building },
  { name: "Price Prediction", href: "/price-prediction", icon: Calculator },
  { name: "Price Intelligence", href: "/price-intelligence", icon: Target },
  { name: "Market Analysis", href: "/market-intelligence", icon: TrendingUp },
  { name: "Dynamic Pricing", href: "/dynamic-pricing", icon: Zap },
  { name: "NRI Assistance", href: "/nri-assistance", icon: Globe },
  { name: "System Health & Monitoring", href: "/system-health", icon: Monitor },
  { name: "Financials & Transactions", href: "/financials", icon: DollarSign },
  { name: "Contracts & eSign", href: "/contracts", icon: FileSignature },
  { name: "Documents", href: "/documents", icon: FileText },
  { name: "Legal Documents", href: "/legal-documents", icon: Shield },
  { name: "Legal Health Reports", href: "/legal-health-reports", icon: CheckCircle },
  { name: "Digital Vault", href: "/digital-vault", icon: Upload },
  { name: "Settings", href: "/settings", icon: Settings },
];

interface SidebarProps {
  isCollapsed: boolean;
  setIsCollapsed: (collapsed: boolean) => void;
}

export function Sidebar({ isCollapsed, setIsCollapsed }: SidebarProps) {
  const [location] = useLocation();
  const [expandedSections, setExpandedSections] = useState<Set<string>>(
    new Set(["Booking & Scheduling", "Marketing Tools"]),
  );

  const toggleSection = (sectionName: string) => {
    const newExpanded = new Set(expandedSections);
    if (newExpanded.has(sectionName)) {
      newExpanded.delete(sectionName);
    } else {
      newExpanded.add(sectionName);
    }
    setExpandedSections(newExpanded);
  };

  const renderNavigationItem = (item: NavigationItem) => {
    const isExpanded = expandedSections.has(item.name);
    const Icon = item.icon;

    if (item.children) {
      return (
        <li key={item.name}>
          <button
            onClick={() => toggleSection(item.name)}
            data-testid={`nav-section-${item.name
              .toLowerCase()
              .replace(/\s+/g, "-")}`}
            className={cn(
              "w-full flex items-center gap-3 px-3 py-3 rounded-lg text-sm font-medium transition-colors text-left",
              "text-gray-700 hover:bg-gray-100 hover:text-[#0056D2]",
            )}
          >
            <Icon className={`w-5 h-5 ${isCollapsed ? "mx-auto" : "mr-3"}`} />
            {!isCollapsed && (
              <span className="whitespace-nowrap">{item.name}</span>
            )}
            {!isCollapsed &&
              (isExpanded ? (
                <ChevronDown className="w-4 h-4 ml-auto" />
              ) : (
                <ChevronRight className="w-4 h-4 ml-auto" />
              ))}
          </button>
          {!isCollapsed && isExpanded && (
            <ul className="mt-2 space-y-1 ml-6">
              {item.children.map((child) => {
                const isActive = location === child.href;
                const ChildIcon = child.icon;

                return (
                  <li key={child.name}>
                    <Link
                      href={child.href!}
                      data-testid={`nav-link-${child.name
                        .toLowerCase()
                        .replace(/\s+/g, "-")}`}
                      className={cn(
                        "flex items-center gap-2 px-3 py-2 rounded-md text-sm transition-colors relative",
                        isActive
                          ? "bg-[#0056D2] text-white shadow-sm"
                          : "text-gray-700 hover:bg-gray-100 hover:text-[#0056D2]",
                      )}
                    >
                      <ChildIcon className="w-4 h-4 mr-2" />
                      {!isCollapsed && child.name}
                      {isActive && !isCollapsed && (
                        <div className="absolute right-0 top-0 bottom-0 w-1 bg-[#00BFA6] rounded-l"></div>
                      )}
                    </Link>
                  </li>
                );
              })}
            </ul>
          )}
        </li>
      );
    }

    // Single item without children
    const isActive = location === item.href;
    return (
      <li key={item.name}>
        <Link
          href={item.href!}
          data-testid={`nav-link-${item.name
            .toLowerCase()
            .replace(/\s+/g, "-")}`}
          className={cn(
            "flex items-center gap-3 px-3 py-3 rounded-lg text-sm font-medium transition-colors relative",
            isActive
              ? "bg-[#0056D2] text-white shadow-sm"
              : "text-gray-700 hover:bg-gray-100 hover:text-[#0056D2]",
          )}
        >
          <Icon className={`w-5 h-5 ${isCollapsed ? "mx-auto" : "mr-3"}`} />
          {!isCollapsed && (
            <span className="whitespace-nowrap">{item.name}</span>
          )}
          {isActive && !isCollapsed && (
            <div className="absolute right-0 top-0 bottom-0 w-1 bg-[#00BFA6] rounded-l"></div>
          )}
        </Link>
      </li>
    );
  };

  return (
    <aside
      className={`bg-white shadow-sm border-r fixed left-0 top-0 h-screen overflow-y-auto z-[60] transition-all duration-300 ${
        isCollapsed ? "w-20" : "w-64"
      }`}
    >
      <div className="p-4">
        <div
          className="flex items-center gap-3 cursor-pointer hover:bg-gray-50 rounded-lg p-2 transition-colors"
          onClick={() => (window.location.href = "/")}
        >
          {!isCollapsed ? (
            <>
              <img
                src={nalLogo}
                alt="NAL Logo"
                className="w-12 h-12 rounded-lg object-cover"
              />
              <div>
                <h1 className="text-lg font-bold text-gray-900">
                  New Age Land
                </h1>
                <p className="text-xs text-gray-600">Seller Panel</p>
              </div>
            </>
          ) : (
            <img
              src={nalLogo}
              alt="NAL Logo"
              className="w-12 h-12 rounded-lg object-cover mx-auto"
            />
          )}
        </div>
      </div>

      <nav
        className="px-2 space-y-1 pb-6 overflow-y-auto"
        style={{
          maxHeight: "calc(100vh - 80px)",
          scrollbarWidth: "none",
          msOverflowStyle: "none",
        }}
      >
        <ul className="space-y-1">
          {navigation.map((item, index) => {
            const isMarketingItem = item.name === "Marketing Tools";
            if (isMarketingItem) {
              const Icon = item.icon;
              const isActive = location === item.href;
              return (
                <li key={item.name} className="flex items-center">
                  <Link
                    href={item.href!}
                    className={cn(
                      "flex-1 flex items-center gap-3 px-3 py-3 rounded-lg text-sm font-medium transition-colors relative",
                      isActive
                        ? "bg-[#0056D2] text-white shadow-sm"
                        : "text-gray-700 hover:bg-gray-100 hover:text-[#0056D2]",
                    )}
                  >
                    <Icon
                      className={`w-5 h-5 ${isCollapsed ? "mx-auto" : "mr-3"}`}
                    />
                    {!isCollapsed && (
                      <span className="whitespace-nowrap">{item.name}</span>
                    )}
                    {isActive && !isCollapsed && (
                      <div className="absolute right-0 top-0 bottom-0 w-1 bg-[#00BFA6] rounded-l"></div>
                    )}
                  </Link>
                  <button
                    onClick={() => setIsCollapsed(!isCollapsed)}
                    className="ml-2 w-8 h-8 bg-[#0056D2] hover:bg-[#0041A3] rounded-full flex items-center justify-center transition-colors shadow-sm"
                  >
                    {isCollapsed ? (
                      <ChevronRight className="w-4 h-4 text-white" />
                    ) : (
                      <ChevronLeft className="w-4 h-4 text-white" />
                    )}
                  </button>
                </li>
              );
            }
            return renderNavigationItem(item);
          })}
        </ul>
      </nav>
    </aside>
  );
}
