import React, { createContext, useContext, useState, useEffect, ReactNode } from 'react';
import type { Property } from '../types';

interface PromotionContextType {
  promotedProperties: Set<string>;
  promoteProperty: (propertyId: string) => void;
  unpromoteProperty: (propertyId: string) => void;
  isPropertyPromoted: (propertyId: string) => boolean;
  updatePropertyPromotionStatus: (property: Property) => Property;
}

const PromotionContext = createContext<PromotionContextType | undefined>(undefined);

export const usePromotion = () => {
  const context = useContext(PromotionContext);
  if (!context) {
    throw new Error('usePromotion must be used within a PromotionProvider');
  }
  return context;
};

interface PromotionProviderProps {
  children: ReactNode;
}

export const PromotionProvider: React.FC<PromotionProviderProps> = ({ children }) => {
  const [promotedProperties, setPromotedProperties] = useState<Set<string>>(new Set());

  // Load promoted properties from localStorage on mount
  useEffect(() => {
    const stored = localStorage.getItem('promotedProperties');
    if (stored) {
      try {
        const parsed = JSON.parse(stored);
        setPromotedProperties(new Set(parsed));
      } catch (error) {
        console.error('Error loading promoted properties:', error);
      }
    }
  }, []);

  // Save to localStorage whenever promotedProperties changes
  useEffect(() => {
    localStorage.setItem('promotedProperties', JSON.stringify(Array.from(promotedProperties)));
  }, [promotedProperties]);

  const promoteProperty = (propertyId: string) => {
    setPromotedProperties(prev => new Set([...prev, propertyId]));
  };

  const unpromoteProperty = (propertyId: string) => {
    setPromotedProperties(prev => {
      const newSet = new Set(prev);
      newSet.delete(propertyId);
      return newSet;
    });
  };

  const isPropertyPromoted = (propertyId: string) => {
    return promotedProperties.has(propertyId);
  };

  const updatePropertyPromotionStatus = (property: Property): Property => {
    const isPromoted = promotedProperties.has(property.id);
    return {
      ...property,
      isPromoted,
      promotionStatus: isPromoted ? 'active' : undefined,
      promotionType: isPromoted ? 'featured' : undefined
    };
  };

  const value: PromotionContextType = {
    promotedProperties,
    promoteProperty,
    unpromoteProperty,
    isPropertyPromoted,
    updatePropertyPromotionStatus
  };

  return (
    <PromotionContext.Provider value={value}>
      {children}
    </PromotionContext.Provider>
  );
};