// Minimal API client just for the seller dashboard
export type Coordinates = { lat: number; lng: number };

export interface CreatePayload {
  title: string;
  type: 'apartment'|'house'|'villa'|'commercial';
  sellerType?: 'owner'|'agent'|'company';
  bhk?: string;
  area: number;
  price: number;
  listing_intent: 'sale'|'rent'|'urgent-sale';
  description: string;
  biddingEnabled?: boolean;
  address: string;
  city: string;
  state: string;
  pincode: string;
  landmark?: string;
  coordinates?: Coordinates;
  amenities?: string[];
  customAmenities?: string[];
  virtualTour?: string;
  termsAccepted?: boolean;

  // files
  images?: File[];
  videos?: File[];
  imageCategories?: Record<string, File[]>;
  documents?: { type: string; file: File }[];
}

export interface PropertyRow {
  id: string;
  title: string;
  type: string;
  bhk?: string;
  address: string;
  city: string;
  state: string;
  pincode: string;
  landmark?: string;
  area: number;
  price: number;
  listingIntent: string;
  status: string;
  amenities: string[];
  customAmenities: string[];
  virtualTour?: string;
  sellerType?: string;
  biddingEnabled?: boolean;
  coordinates?: Coordinates;
  views: number;
  inquiries: number;
  createdAt: string;
  updatedAt: string;
}

const API_BASE = (import.meta as any).env?.VITE_API_BASE
  || (window as any).__API_BASE
  || 'http://localhost/realical-php-api/api';

// map backend snake_case to frontend camelCase
function mapRow(row: any): PropertyRow {
  const amenities = row.amenities ? JSON.parse(row.amenities) : [];
  const customAmenities = row.custom_amenities ? JSON.parse(row.custom_amenities) : [];
  const lat = row.latitude != null ? parseFloat(row.latitude) : null;
  const lng = row.longitude != null ? parseFloat(row.longitude) : null;

  return {
    id: String(row.id),
    title: row.title,
    type: row.type,
    bhk: row.bhk ?? undefined,
    address: row.address,
    city: row.city,
    state: row.state,
    pincode: row.pincode,
    landmark: row.landmark ?? undefined,
    area: Number(row.area),
    price: Number(row.price),
    listingIntent: row.listing_intent,
    status: row.status,
    amenities,
    customAmenities,
    virtualTour: row.virtual_tour ?? undefined,
    sellerType: row.seller_type ?? undefined,
    biddingEnabled: !!row.bidding_enabled,
    coordinates: (lat!=null && lng!=null) ? { lat, lng } : undefined,
    views: Number(row.views ?? 0),
    inquiries: Number(row.inquiries ?? 0),
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export const api = {
  async create(payload: CreatePayload): Promise<{ id:number; documents?: {doc_id:string; type:string; name:string}[] }> {
    const fd = new FormData();
    fd.append('title', payload.title);
    fd.append('type', payload.type);
    if (payload.sellerType) fd.append('sellerType', payload.sellerType);
    if (payload.bhk) fd.append('bhk', payload.bhk);
    fd.append('area', String(payload.area));
    fd.append('price', String(payload.price));
    fd.append('listing_intent', payload.listing_intent);
    fd.append('description', payload.description);
    if (payload.biddingEnabled !== undefined) fd.append('biddingEnabled', String(payload.biddingEnabled));
    fd.append('address', payload.address);
    fd.append('city', payload.city);
    fd.append('state', payload.state);
    fd.append('pincode', payload.pincode);
    if (payload.landmark) fd.append('landmark', payload.landmark);
    if (payload.virtualTour) fd.append('virtualTour', payload.virtualTour);
    if (payload.termsAccepted !== undefined) fd.append('termsAccepted', String(payload.termsAccepted));

    if (payload.amenities) fd.append('amenities', JSON.stringify(payload.amenities));
    if (payload.customAmenities) fd.append('customAmenities', JSON.stringify(payload.customAmenities));
    if (payload.coordinates) fd.append('coordinates', JSON.stringify(payload.coordinates));

    (payload.images || []).forEach(f => fd.append('images[]', f));
    (payload.videos || []).forEach(f => fd.append('videos[]', f));
    Object.entries(payload.imageCategories || {}).forEach(([cat, files]) => {
      (files || []).forEach(f => fd.append(`imageCategories[${cat}][]`, f));
    });
    (payload.documents || []).forEach(d => {
      fd.append('doc_types[]', d.type);
      fd.append('docs[]', d.file);
    });

    const res = await fetch(`${API_BASE}/properties/create.php`, {
      method: 'POST',
      body: fd,
      credentials: 'include'
    });
    if (!res.ok) {
      const err = await res.json().catch(()=>({}));
      throw new Error(err.error || `Create failed (${res.status})`);
    }
    return res.json();
  },

  async list(): Promise<PropertyRow[]> {
    const res = await fetch(`${API_BASE}/properties/list.php`, { credentials: 'include' });
    if (!res.ok) throw new Error('List failed');
    const payload = await res.json();
    const rows = Array.isArray(payload) ? payload : (payload.properties ?? []);
    return rows.map(mapRow);
  },

  async get(id: string|number): Promise<PropertyRow|null> {
    const res = await fetch(`${API_BASE}/properties/get.php?id=${id}`, { credentials: 'include' });
    if (res.status === 404) return null;
    if (!res.ok) throw new Error('Get failed');
    const payload = await res.json();
    const row = payload.property ?? payload;
    return mapRow(row);
  },
};