import type { Property, Lead, Booking, Document, Campaign, User } from "@shared/schema";

export const mockUser: User = {
  id: "user-1",
  username: "john_smith",
  password: "",
  email: "john.smith@example.com",
  firstName: "John",
  lastName: "Smith",
  createdAt: new Date(),
};

export const mockProperties: Property[] = [
  {
    id: "prop-1",
    sellerId: "user-1",
    title: "Modern 2BR Apartment",
    description: "Beautiful downtown apartment with city views",
    location: "Downtown",
    propertyType: "Apartment",
    price: "7500000",
    size: 1200,
    bedrooms: 2,
    bathrooms: 2,
    status: "active",
    isBiddingEnabled: true,
    isUrgentSale: false,
    views: 156,
    images: ["https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=600"],
    features: ["Parking", "Balcony", "Modern Kitchen"],
    createdAt: new Date("2024-12-01"),
    updatedAt: new Date("2024-12-15"),
  },
  {
    id: "prop-2",
    sellerId: "user-1",
    title: "3BR Villa with Garden",
    description: "Spacious suburban villa with beautiful garden",
    location: "Suburbs",
    propertyType: "Villa",
    price: "12500000",
    size: 2500,
    bedrooms: 3,
    bathrooms: 3,
    status: "pending",
    isBiddingEnabled: false,
    isUrgentSale: false,
    views: 132,
    images: ["https://images.unsplash.com/photo-1600596542815-ffad4c1539a9?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=600"],
    features: ["Garden", "Pool", "Garage"],
    createdAt: new Date("2024-11-15"),
    updatedAt: new Date("2024-12-10"),
  },
  {
    id: "prop-3",
    sellerId: "user-1",
    title: "Penthouse Suite",
    description: "Luxury penthouse with panoramic city views",
    location: "City Center",
    propertyType: "Penthouse",
    price: "18500000",
    size: 1800,
    bedrooms: 3,
    bathrooms: 2,
    status: "active",
    isBiddingEnabled: true,
    isUrgentSale: true,
    views: 98,
    images: ["https://images.unsplash.com/photo-1613977257363-707ba9348227?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=600"],
    features: ["Terrace", "City View", "Premium Finishes"],
    createdAt: new Date("2024-12-05"),
    updatedAt: new Date("2024-12-14"),
  },
];

export const mockLeads: Lead[] = [
  {
    id: "lead-1",
    buyerId: "buyer-1",
    propertyId: "prop-1",
    sellerId: "user-1",
    status: "new",
    interestLevel: "high",
    message: "Very interested in this property. Can we schedule a viewing?",
    notes: "",
    lastContactDate: null,
    createdAt: new Date("2024-12-15T10:30:00"),
  },
  {
    id: "lead-2",
    buyerId: "buyer-2",
    propertyId: "prop-2",
    sellerId: "user-1",
    status: "contacted",
    interestLevel: "medium",
    message: "Looking for a family home in this area",
    notes: "Interested in financing options",
    lastContactDate: new Date("2024-12-14T14:00:00"),
    createdAt: new Date("2024-12-13T09:15:00"),
  },
  {
    id: "lead-3",
    buyerId: "buyer-3",
    propertyId: "prop-3",
    sellerId: "user-1",
    status: "follow-up",
    interestLevel: "high",
    message: "Interested in making an offer",
    notes: "Pre-approved for mortgage, serious buyer",
    lastContactDate: new Date("2024-12-12T16:30:00"),
    createdAt: new Date("2024-12-10T11:00:00"),
  },
  {
    id: "lead-4",
    buyerId: "buyer-4",
    propertyId: "prop-1",
    sellerId: "user-1",
    status: "new",
    interestLevel: "new",
    message: "Is this property still available? What are the amenities?",
    notes: "",
    lastContactDate: null,
    createdAt: new Date("2024-12-15T14:20:00"),
  },
  {
    id: "lead-5",
    buyerId: "buyer-5",
    propertyId: "prop-2",
    sellerId: "user-1",
    status: "contacted",
    interestLevel: "low",
    message: "Can you provide more details about the neighborhood?",
    notes: "First-time buyer, needs guidance",
    lastContactDate: new Date("2024-12-13T11:30:00"),
    createdAt: new Date("2024-12-12T16:45:00"),
  },
  {
    id: "lead-6",
    buyerId: "buyer-6",
    propertyId: "prop-3",
    sellerId: "user-1",
    status: "follow-up",
    interestLevel: "medium",
    message: "What's the final price? Any room for negotiation?",
    notes: "Investor looking for good deals",
    lastContactDate: new Date("2024-12-11T09:15:00"),
    createdAt: new Date("2024-12-09T13:30:00"),
  },
  {
    id: "lead-7",
    buyerId: "buyer-7",
    propertyId: "prop-1",
    sellerId: "user-1",
    status: "new",
    interestLevel: "high",
    message: "I love this apartment! When can I visit?",
    notes: "",
    lastContactDate: null,
    createdAt: new Date("2024-12-15T16:45:00"),
  },
  {
    id: "lead-8",
    buyerId: "buyer-8",
    propertyId: "prop-2",
    sellerId: "user-1",
    status: "contacted",
    interestLevel: "medium",
    message: "Is the garden well maintained? Any pets allowed?",
    notes: "Has two dogs, concerned about pet policy",
    lastContactDate: new Date("2024-12-14T10:20:00"),
    createdAt: new Date("2024-12-13T14:30:00"),
  },
  {
    id: "lead-9",
    buyerId: "buyer-9",
    propertyId: "prop-3",
    sellerId: "user-1",
    status: "closed",
    interestLevel: "low",
    message: "Thanks for the information, but it's out of my budget.",
    notes: "Budget constraints, looking for cheaper options",
    lastContactDate: new Date("2024-12-10T15:45:00"),
    createdAt: new Date("2024-12-08T11:20:00"),
  },
  {
    id: "lead-10",
    buyerId: "buyer-10",
    propertyId: "prop-1",
    sellerId: "user-1",
    status: "follow-up",
    interestLevel: "high",
    message: "Ready to make an offer. What documents do you need?",
    notes: "Serious buyer, has financing pre-approved",
    lastContactDate: new Date("2024-12-13T09:30:00"),
    createdAt: new Date("2024-12-11T16:15:00"),
  },
];

export const mockBuyers: User[] = [
  {
    id: "buyer-1",
    username: "sarah_johnson",
    password: "",
    email: "sarah.j@email.com",
    firstName: "Sarah",
    lastName: "Johnson",
    createdAt: new Date(),
  },
  {
    id: "buyer-2",
    username: "mike_brown",
    password: "",
    email: "mike.brown@email.com",
    firstName: "Mike",
    lastName: "Brown",
    createdAt: new Date(),
  },
  {
    id: "buyer-3",
    username: "anna_kim",
    password: "",
    email: "anna.kim@email.com",
    firstName: "Anna",
    lastName: "Kim",
    createdAt: new Date(),
  },
  {
    id: "buyer-4",
    username: "david_wilson",
    password: "",
    email: "david.wilson@email.com",
    firstName: "David",
    lastName: "Wilson",
    createdAt: new Date(),
  },
  {
    id: "buyer-5",
    username: "emma_davis",
    password: "",
    email: "emma.davis@email.com",
    firstName: "Emma",
    lastName: "Davis",
    createdAt: new Date(),
  },
  {
    id: "buyer-6",
    username: "raj_patel",
    password: "",
    email: "raj.patel@email.com",
    firstName: "Raj",
    lastName: "Patel",
    createdAt: new Date(),
  },
  {
    id: "buyer-7",
    username: "lisa_chen",
    password: "",
    email: "lisa.chen@email.com",
    firstName: "Lisa",
    lastName: "Chen",
    createdAt: new Date(),
  },
  {
    id: "buyer-8",
    username: "mark_taylor",
    password: "",
    email: "mark.taylor@email.com",
    firstName: "Mark",
    lastName: "Taylor",
    createdAt: new Date(),
  },
  {
    id: "buyer-9",
    username: "priya_sharma",
    password: "",
    email: "priya.sharma@email.com",
    firstName: "Priya",
    lastName: "Sharma",
    createdAt: new Date(),
  },
  {
    id: "buyer-10",
    username: "james_rodriguez",
    password: "",
    email: "james.rodriguez@email.com",
    firstName: "James",
    lastName: "Rodriguez",
    createdAt: new Date(),
  },
];

export const mockBookings: Booking[] = [
  {
    id: "booking-1",
    buyerId: "buyer-1",
    propertyId: "prop-1",
    sellerId: "user-1",
    type: "visit",
    scheduledDate: new Date("2024-12-16T14:00:00"),
    status: "pending",
    meetingLink: null,
    notes: "First-time buyer, needs guidance",
    createdAt: new Date("2024-12-15T10:30:00"),
  },
  {
    id: "booking-2",
    buyerId: "buyer-2",
    propertyId: "prop-2",
    sellerId: "user-1",
    type: "virtual",
    scheduledDate: new Date("2024-12-18T10:00:00"),
    status: "pending",
    meetingLink: "https://meet.google.com/abc-defg-hij",
    notes: "Virtual tour requested",
    createdAt: new Date("2024-12-14T16:00:00"),
  },
  {
    id: "booking-3",
    buyerId: "buyer-3",
    propertyId: "prop-3",
    sellerId: "user-1",
    type: "virtual",
    scheduledDate: new Date("2024-12-15T15:00:00"),
    status: "confirmed",
    meetingLink: "https://meet.google.com/xyz-uvwx-rst",
    notes: "Follow-up virtual tour",
    createdAt: new Date("2024-12-12T09:00:00"),
  },
];

export const mockDocuments: Document[] = [
  {
    id: "doc-1",
    sellerId: "user-1",
    propertyId: null,
    fileName: "Property_Deed_001.pdf",
    fileType: "pdf",
    documentType: "Ownership Proof",
    filePath: "/documents/property_deed_001.pdf",
    status: "verified",
    linkedProperties: 2,
    createdAt: new Date("2024-12-10"),
  },
  {
    id: "doc-2",
    sellerId: "user-1",
    propertyId: "prop-1",
    fileName: "Tax_Receipt_2024.pdf",
    fileType: "pdf",
    documentType: "Tax Records",
    filePath: "/documents/tax_receipt_2024.pdf",
    status: "pending",
    linkedProperties: 1,
    createdAt: new Date("2024-12-08"),
  },
  {
    id: "doc-3",
    sellerId: "user-1",
    propertyId: null,
    fileName: "Legal_NOC_Certificate.pdf",
    fileType: "pdf",
    documentType: "Legal Documents",
    filePath: "/documents/legal_noc.pdf",
    status: "verified",
    linkedProperties: 3,
    createdAt: new Date("2024-11-28"),
  },
];

export const mockCampaigns: Campaign[] = [
  {
    id: "campaign-1",
    sellerId: "user-1",
    propertyId: "prop-1",
    name: "Premium Boost",
    type: "featured",
    budget: "50000",
    spent: "31200",
    startDate: new Date("2024-12-01"),
    endDate: new Date("2024-12-15"),
    status: "active",
    clicks: 245,
    views: 856,
    leads: 8,
    createdAt: new Date("2024-12-01"),
  },
  {
    id: "campaign-2",
    sellerId: "user-1",
    propertyId: "prop-2",
    name: "Social Media Push",
    type: "social",
    budget: "75000",
    spent: "45500",
    startDate: new Date("2024-11-20"),
    endDate: new Date("2024-12-20"),
    status: "active",
    clicks: 312,
    views: 1247,
    leads: 15,
    createdAt: new Date("2024-11-20"),
  },
  {
    id: "campaign-3",
    sellerId: "user-1",
    propertyId: "prop-3",
    name: "Google Ads Campaign",
    type: "search",
    budget: "30000",
    spent: "30000",
    startDate: new Date("2024-11-01"),
    endDate: new Date("2024-11-30"),
    status: "completed",
    clicks: 89,
    views: 334,
    leads: 5,
    createdAt: new Date("2024-11-01"),
  },
];

// Helper functions to get related data
export function getPropertyById(id: string): Property | undefined {
  return mockProperties.find(p => p.id === id);
}

export function getBuyerById(id: string): User | undefined {
  return mockBuyers.find(b => b.id === id);
}

export function getLeadsForProperty(propertyId: string): Lead[] {
  return mockLeads.filter(l => l.propertyId === propertyId);
}

export function getBookingsForProperty(propertyId: string): Booking[] {
  return mockBookings.filter(b => b.propertyId === propertyId);
}

export function getCampaignsForProperty(propertyId: string): Campaign[] {
  return mockCampaigns.filter(c => c.propertyId === propertyId);
}

// Dashboard stats
export function getDashboardStats() {
  const totalProperties = mockProperties.length;
  const activeCampaigns = mockCampaigns.filter(c => c.status === "active").length;
  const newInquiries = mockLeads.filter(l => l.status === "new").length;
  const totalViews = mockProperties.reduce((sum, p) => sum + (p.views || 0), 0);
  
  return {
    totalProperties,
    activeCampaigns,
    newInquiries,
    totalViews,
  };
}
