import React, { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../components/ui/select";
import { 
  Brain, 
  TrendingUp, 
  TrendingDown,
  Target,
  Lightbulb,
  AlertTriangle,
  CheckCircle,
  DollarSign,
  Calendar,
  MapPin,
  Users,
  Eye,
  Star,
  Zap,
  Award,
  Activity,
  BarChart3,
  PieChart,
  ArrowUpRight,
  ArrowDownRight,
  Clock,
  ThumbsUp
} from "lucide-react";

// AI-powered insights data
const aiInsights = {
  propertyOptimization: [
    {
      id: "opt-1",
      propertyId: "prop-1",
      propertyName: "Modern Downtown Apartment",
      aiScore: 87,
      recommendations: [
        {
          type: "pricing",
          priority: "high",
          title: "Optimize Pricing Strategy",
          description: "Current price is 8% above market average. Consider reducing by $15K to increase inquiries by 35%.",
          impact: "35% more inquiries",
          effort: "low",
          confidence: 92
        },
        {
          type: "photos",
          priority: "medium", 
          title: "Enhance Photo Quality",
          description: "Professional photos could increase views by 45%. Focus on natural lighting and staging.",
          impact: "45% more views",
          effort: "medium",
          confidence: 88
        },
        {
          type: "description",
          priority: "low",
          title: "Improve Property Description",
          description: "Add keywords like 'modern amenities' and 'walkable neighborhood' to boost search visibility.",
          impact: "20% better SEO",
          effort: "low",
          confidence: 75
        }
      ]
    }
  ],
  marketPredictions: {
    priceForecast: {
      current: 485000,
      next3Months: 492000,
      next6Months: 498000,
      next12Months: 515000,
      confidence: 84,
      trend: "rising"
    },
    demandForecast: {
      current: 8.2,
      next3Months: 8.7,
      next6Months: 9.1,
      confidence: 79,
      factors: ["Interest rates", "Local development", "Seasonal trends"]
    },
    competitorAnalysis: {
      similarProperties: 23,
      avgPrice: 465000,
      avgDaysOnMarket: 52,
      yourAdvantage: "15% faster sales",
      marketShare: 12.5
    }
  },
  leadPredictions: [
    {
      leadId: "lead-1",
      name: "Sarah Johnson",
      conversionProbability: 89,
      predictedAction: "Will make offer within 7 days",
      recommendedActions: [
        "Schedule property visit within 48 hours",
        "Prepare competitive pricing analysis",
        "Share neighborhood amenities information"
      ],
      aiFactors: ["High engagement", "Pre-approved", "Urgent timeline"]
    },
    {
      leadId: "lead-2", 
      name: "Michael Chen",
      conversionProbability: 67,
      predictedAction: "Needs 2-3 more property viewings",
      recommendedActions: [
        "Show similar luxury properties",
        "Provide investment ROI analysis",
        "Connect with mortgage specialist"
      ],
      aiFactors: ["Luxury preference", "Investment focus", "Price sensitive"]
    }
  ],
  performanceInsights: {
    topPerformingFeatures: [
      { feature: "Modern Kitchen", impact: "+23% inquiries" },
      { feature: "Parking Space", impact: "+18% views" },
      { feature: "Balcony/Terrace", impact: "+15% favorites" }
    ],
    underperformingAreas: [
      { area: "Virtual Tour", impact: "-12% engagement" },
      { area: "Neighborhood Info", impact: "-8% time on page" }
    ],
    seasonalTrends: {
      bestMonths: ["March", "April", "September"],
      worstMonths: ["December", "January"],
      currentTrend: "Peak season approaching"
    }
  }
};

export default function AIInsights() {
  const [activeTab, setActiveTab] = useState("optimization");
  const [selectedProperty, setSelectedProperty] = useState("all");

  const PropertyOptimization = () => (
    <div className="space-y-6">
      {aiInsights.propertyOptimization.map((property) => (
        <Card key={property.id}>
          <CardHeader>
            <div className="flex justify-between items-start">
              <div>
                <CardTitle className="text-xl">{property.propertyName}</CardTitle>
                <p className="text-muted-foreground">AI Optimization Score</p>
              </div>
              <div className="text-right">
                <div className="flex items-center gap-2">
                  <span className="text-3xl font-bold text-blue-600">{property.aiScore}</span>
                  <div className="text-sm">
                    <Brain className="w-5 h-5 text-blue-600" />
                  </div>
                </div>
                <p className="text-sm text-muted-foreground">out of 100</p>
              </div>
            </div>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {property.recommendations.map((rec, index) => (
                <Card key={index} className="border-l-4 border-l-blue-500">
                  <CardContent className="p-4">
                    <div className="flex justify-between items-start mb-3">
                      <div className="flex-1">
                        <div className="flex items-center gap-2 mb-2">
                          <h4 className="font-semibold">{rec.title}</h4>
                          <Badge className={
                            rec.priority === "high" ? "bg-red-100 text-red-800" :
                            rec.priority === "medium" ? "bg-orange-100 text-orange-800" :
                            "bg-blue-100 text-blue-800"
                          }>
                            {rec.priority} priority
                          </Badge>
                        </div>
                        <p className="text-sm text-muted-foreground mb-2">{rec.description}</p>
                        <div className="flex items-center gap-4 text-sm">
                          <div className="flex items-center gap-1">
                            <TrendingUp className="w-4 h-4 text-green-600" />
                            <span className="text-green-600 font-medium">{rec.impact}</span>
                          </div>
                          <div className="flex items-center gap-1">
                            <Clock className="w-4 h-4 text-blue-600" />
                            <span>{rec.effort} effort</span>
                          </div>
                          <div className="flex items-center gap-1">
                            <Target className="w-4 h-4 text-purple-600" />
                            <span>{rec.confidence}% confidence</span>
                          </div>
                        </div>
                      </div>
                      <Button size="sm">
                        Implement
                      </Button>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );

  const MarketPredictions = () => (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <TrendingUp className="w-5 h-5 text-green-600" />
              Price Forecast
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div>
                <p className="text-sm text-muted-foreground">Current Value</p>
                <p className="text-2xl font-bold">${aiInsights.marketPredictions.priceForecast.current.toLocaleString()}</p>
              </div>
              <div className="space-y-2">
                <div className="flex justify-between">
                  <span className="text-sm">3 Months</span>
                  <span className="font-medium text-green-600">
                    ${aiInsights.marketPredictions.priceForecast.next3Months.toLocaleString()}
                  </span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm">6 Months</span>
                  <span className="font-medium text-green-600">
                    ${aiInsights.marketPredictions.priceForecast.next6Months.toLocaleString()}
                  </span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm">12 Months</span>
                  <span className="font-medium text-green-600">
                    ${aiInsights.marketPredictions.priceForecast.next12Months.toLocaleString()}
                  </span>
                </div>
              </div>
              <div className="pt-2 border-t">
                <p className="text-xs text-muted-foreground">
                  Confidence: {aiInsights.marketPredictions.priceForecast.confidence}%
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="w-5 h-5 text-blue-600" />
              Demand Forecast
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div>
                <p className="text-sm text-muted-foreground">Current Demand Score</p>
                <p className="text-2xl font-bold">{aiInsights.marketPredictions.demandForecast.current}/10</p>
              </div>
              <div className="space-y-2">
                <div className="flex justify-between">
                  <span className="text-sm">3 Months</span>
                  <span className="font-medium text-blue-600">
                    {aiInsights.marketPredictions.demandForecast.next3Months}/10
                  </span>
                </div>
                <div className="flex justify-between">
                  <span className="text-sm">6 Months</span>
                  <span className="font-medium text-blue-600">
                    {aiInsights.marketPredictions.demandForecast.next6Months}/10
                  </span>
                </div>
              </div>
              <div className="pt-2 border-t">
                <p className="text-xs text-muted-foreground mb-2">Key Factors:</p>
                <div className="flex flex-wrap gap-1">
                  {aiInsights.marketPredictions.demandForecast.factors.map((factor) => (
                    <Badge key={factor} variant="outline" className="text-xs">
                      {factor}
                    </Badge>
                  ))}
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Competitive Analysis</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
            <div className="text-center">
              <p className="text-2xl font-bold text-blue-600">
                {aiInsights.marketPredictions.competitorAnalysis.similarProperties}
              </p>
              <p className="text-sm text-muted-foreground">Similar Properties</p>
            </div>
            <div className="text-center">
              <p className="text-2xl font-bold text-green-600">
                ${aiInsights.marketPredictions.competitorAnalysis.avgPrice.toLocaleString()}
              </p>
              <p className="text-sm text-muted-foreground">Average Price</p>
            </div>
            <div className="text-center">
              <p className="text-2xl font-bold text-orange-600">
                {aiInsights.marketPredictions.competitorAnalysis.avgDaysOnMarket}
              </p>
              <p className="text-sm text-muted-foreground">Avg Days on Market</p>
            </div>
            <div className="text-center">
              <p className="text-2xl font-bold text-purple-600">
                {aiInsights.marketPredictions.competitorAnalysis.marketShare}%
              </p>
              <p className="text-sm text-muted-foreground">Your Market Share</p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );

  const LeadPredictions = () => (
    <div className="space-y-6">
      {aiInsights.leadPredictions.map((lead) => (
        <Card key={lead.leadId}>
          <CardContent className="p-6">
            <div className="flex justify-between items-start mb-4">
              <div>
                <h3 className="text-lg font-semibold">{lead.name}</h3>
                <p className="text-sm text-muted-foreground">{lead.predictedAction}</p>
              </div>
              <div className="text-right">
                <div className="flex items-center gap-2">
                  <span className="text-2xl font-bold text-green-600">{lead.conversionProbability}%</span>
                  <Target className="w-5 h-5 text-green-600" />
                </div>
                <p className="text-sm text-muted-foreground">Conversion Probability</p>
              </div>
            </div>

            <div className="mb-4">
              <h4 className="font-medium mb-2">AI Recommended Actions:</h4>
              <div className="space-y-2">
                {lead.recommendedActions.map((action, index) => (
                  <div key={index} className="flex items-center gap-2 p-2 bg-blue-50 rounded-lg">
                    <Lightbulb className="w-4 h-4 text-blue-600" />
                    <span className="text-sm">{action}</span>
                  </div>
                ))}
              </div>
            </div>

            <div>
              <h4 className="font-medium mb-2">AI Analysis Factors:</h4>
              <div className="flex flex-wrap gap-2">
                {lead.aiFactors.map((factor) => (
                  <Badge key={factor} className="bg-purple-100 text-purple-800">
                    {factor}
                  </Badge>
                ))}
              </div>
            </div>
          </CardContent>
        </Card>
      ))}
    </div>
  );

  const PerformanceInsights = () => (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <ThumbsUp className="w-5 h-5 text-green-600" />
              Top Performing Features
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {aiInsights.performanceInsights.topPerformingFeatures.map((feature, index) => (
                <div key={index} className="flex justify-between items-center p-3 bg-white rounded-lg">
                  <span className="font-medium">{feature.feature}</span>
                  <span className="text-green-600 font-semibold">{feature.impact}</span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <AlertTriangle className="w-5 h-5 text-orange-600" />
              Areas for Improvement
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {aiInsights.performanceInsights.underperformingAreas.map((area, index) => (
                <div key={index} className="flex justify-between items-center p-3 bg-white rounded-lg">
                  <span className="font-medium">{area.area}</span>
                  <span className="text-orange-600 font-semibold">{area.impact}</span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Seasonal Trends Analysis</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <div>
              <h4 className="font-medium mb-3 text-green-600">Best Performing Months</h4>
              <div className="space-y-2">
                {aiInsights.performanceInsights.seasonalTrends.bestMonths.map((month) => (
                  <Badge key={month} className="bg-green-100 text-green-800 mr-2">
                    {month}
                  </Badge>
                ))}
              </div>
            </div>
            <div>
              <h4 className="font-medium mb-3 text-red-600">Challenging Months</h4>
              <div className="space-y-2">
                {aiInsights.performanceInsights.seasonalTrends.worstMonths.map((month) => (
                  <Badge key={month} className="bg-red-100 text-red-800 mr-2">
                    {month}
                  </Badge>
                ))}
              </div>
            </div>
            <div>
              <h4 className="font-medium mb-3 text-blue-600">Current Trend</h4>
              <p className="text-sm bg-blue-50 p-3 rounded-lg">
                {aiInsights.performanceInsights.seasonalTrends.currentTrend}
              </p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">AI-Powered Insights</h1>
          <p className="text-muted-foreground">Advanced AI recommendations and market predictions</p>
        </div>
        <Select value={selectedProperty} onValueChange={setSelectedProperty}>
          <SelectTrigger className="w-48">
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">All Properties</SelectItem>
            <SelectItem value="prop-1">Downtown Apartment</SelectItem>
            <SelectItem value="prop-2">Beachfront Villa</SelectItem>
          </SelectContent>
        </Select>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
        <TabsList className="grid w-full grid-cols-2 md:grid-cols-4 h-auto p-1 bg-gray-100 rounded-lg gap-1">
          <TabsTrigger 
            value="optimization" 
            className={`flex flex-col items-center gap-1 p-3 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
              activeTab === "optimization" 
                ? "bg-[#0056D2] text-white border-[#0056D2]" 
                : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
            }`}
          >
            <Lightbulb className="w-5 h-5" />
            <span className="font-medium text-xs">Optimization</span>
          </TabsTrigger>
          <TabsTrigger 
            value="market" 
            className={`flex flex-col items-center gap-1 p-3 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
              activeTab === "market" 
                ? "bg-[#0056D2] text-white border-[#0056D2]" 
                : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
            }`}
          >
            <TrendingUp className="w-5 h-5" />
            <span className="font-medium text-xs">Market Predictions</span>
          </TabsTrigger>
          <TabsTrigger 
            value="leads" 
            className={`flex flex-col items-center gap-1 p-3 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
              activeTab === "leads" 
                ? "bg-[#0056D2] text-white border-[#0056D2]" 
                : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
            }`}
          >
            <Target className="w-5 h-5" />
            <span className="font-medium text-xs">Lead Predictions</span>
          </TabsTrigger>
          <TabsTrigger 
            value="performance" 
            className={`flex flex-col items-center gap-1 p-3 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
              activeTab === "performance" 
                ? "bg-[#0056D2] text-white border-[#0056D2]" 
                : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
            }`}
          >
            <BarChart3 className="w-5 h-5" />
            <span className="font-medium text-xs">Performance</span>
          </TabsTrigger>
        </TabsList>

        <TabsContent value="optimization" className="mt-6">
          <PropertyOptimization />
        </TabsContent>

        <TabsContent value="market" className="mt-6">
          <MarketPredictions />
        </TabsContent>

        <TabsContent value="leads" className="mt-6">
          <LeadPredictions />
        </TabsContent>

        <TabsContent value="performance" className="mt-6">
          <PerformanceInsights />
        </TabsContent>
      </Tabs>
    </div>
  );
}