import React, { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { useForm, FormProvider } from "react-hook-form";
import { toast } from "sonner";

import { PropertyFormData, FormStep } from "../types";
import { api } from "../utils/api";

// Form Steps
import DetailsStep from "../components/form/DetailsStep";
import DocumentsStep from "../components/form/DocumentsStep";
import IntentStep from "../components/form/IntentStep";
import LocationStep from "../components/form/LocationStep";
import MediaStep from "../components/form/MediaStep";
import ReviewStep from "../components/form/ReviewStep";

const steps: {
  key: FormStep;
  title: string;
  component: React.ComponentType<any>;
}[] = [
  { key: "details", title: "Details", component: DetailsStep },
  { key: "documents", title: "Documents", component: DocumentsStep },
  { key: "intent", title: "Amenities", component: IntentStep },
  { key: "location", title: "Location", component: LocationStep },
  { key: "media", title: "Media", component: MediaStep },
  { key: "review", title: "Review", component: ReviewStep },
];

const AddProperty: React.FC = () => {
  const navigate = useNavigate();

  const [currentStep, setCurrentStep] = useState(0);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [userDetails, setUserDetails] = useState<{
    firstname: string;
    lastname: string;
    mobile: string;
    email: string;
  } | null>(null);

  // Fetch user details on component mount
  useEffect(() => {
    const fetchUser = async () => {
      const user = await api.fetchUserDetails();
      setUserDetails(user);
    };
    fetchUser();
  }, []);

  // Get prefilled data from sessionStorage or asset listing data
  const getPrefillData = () => {
    // Check for asset listing data first
    const assetData = localStorage.getItem("asset-listing-data");
    if (assetData) {
      console.log("🔍 Found asset listing data:", assetData);
      localStorage.removeItem("asset-listing-data");
      const parsed = JSON.parse(assetData);
      console.log("📋 Parsed asset data:", parsed);
      return parsed;
    }

    // Fallback to sessionStorage
    const stored = sessionStorage.getItem("prefillPropertyData");
    if (stored) {
      sessionStorage.removeItem("prefillPropertyData");
      return JSON.parse(stored);
    }
    return null;
  };

  const prefillData = getPrefillData();

  console.log("🔍 Prefill data loaded:", prefillData);

  const methods = useForm<PropertyFormData>({
    defaultValues: prefillData
      ? {
          title: prefillData.title || "",
          type: prefillData.propertyType || prefillData.type || "apartment",
          bhk: prefillData.bedrooms
            ? `${prefillData.bedrooms}BHK`
            : prefillData.bhk || "",
          address: prefillData.address || "",
          city: prefillData.city || "",
          state: prefillData.state || "",
          pincode: prefillData.pincode || "",
          landmark: prefillData.locality || prefillData.landmark || "",
          area: prefillData.builtUpArea || prefillData.area || 0,
          price: prefillData.salePrice || prefillData.price || 0,
          listingIntent: prefillData.listingType || "sale",
          amenities: prefillData.amenities || [],
          customAmenities: prefillData.customAmenities || [],
          sellerType: prefillData.sellerType || "owner",
          description: prefillData.description || prefillData.bio || "",
          images: [],
          videos: [],
          virtualTour: "",
          imageCategories: {},
          documents: [],
          requiredDocuments: [],
          projectBrochure: undefined,
          biddingEnabled: false,
          termsAccepted: false,
          fullname: prefillData.ownerName || "",
          email: prefillData.email || "",
          property_manager_contact: prefillData.mobile || "",
        }
      : {
          title: "",
          type: "apartment",
          bhk: "",
          address: "",
          city: "",
          area: 0,
          price: 0,
          listingIntent: "sale",
          amenities: [],
          customAmenities: [],
          sellerType: "owner",
          description: "",
          images: [],
          videos: [],
          virtualTour: "",
          imageCategories: {},
          documents: [],
          requiredDocuments: [],
          projectBrochure: undefined,
          biddingEnabled: false,
          state: "",
          pincode: "",
          landmark: "",
          termsAccepted: false,
          fullname: "",
          email: "",
          property_manager_contact: "",
        },
  });

  // Update form with user details when available (but don't override asset data)
  useEffect(() => {
    if (userDetails && !prefillData) {
      methods.setValue(
        "fullname",
        `${userDetails.firstname} ${userDetails.lastname}`.trim(),
      );
      methods.setValue("email", userDetails.email);
      methods.setValue("property_manager_contact", userDetails.mobile);
    }
  }, [userDetails, methods, prefillData]);

  const handleNext = () => {
    if (currentStep < steps.length - 1) {
      setCurrentStep(currentStep + 1);
    }
  };

  const handlePrev = () => {
    if (currentStep > 0) {
      setCurrentStep(currentStep - 1);
    }
  };

  const handleSubmit = async (data: PropertyFormData) => {
    try {
      setIsSubmitting(true);

      let result;

      // Check if this is an asset update (not a new property creation)
      if (prefillData?.isAssetUpdate && prefillData?.propertyId) {
        console.log(
          "🔄 Updating existing asset property:",
          prefillData.propertyId,
        );
        // Update existing property instead of creating new one
        await api.updateProperty(prefillData.propertyId, {
          ...data,
        });
        result = { id: prefillData.propertyId };
      } else {
        console.log("➕ Creating new property");
        result = await api.createProperty(data);
      }

      // Success toast
      toast.success(
        prefillData?.isAssetUpdate
          ? "Asset updated successfully!"
          : "Property created successfully!",
        {
          description: `Property ID: ${result.id}`,
          duration: 3000,
        },
      );

      // Dispatch event to refresh property list
      window.dispatchEvent(
        new CustomEvent("propertyAdded", { detail: result }),
      );

      // Small delay to ensure event is processed before navigation
      setTimeout(() => {
        navigate("/seller/dashboard/properties");
      }, 100);
    } catch (e: any) {
      // Error toast
      toast.error(
        prefillData?.isAssetUpdate
          ? "Failed to update asset"
          : "Failed to create property",
        {
          description: e.message || "Please try again later",
          duration: 4000,
        },
      );
    } finally {
      setIsSubmitting(false);
    }
  };

  const CurrentStepComponent = steps[currentStep].component;

  return (
    <div className="flex-1 p-6">
      <div className="max-w-4xl mx-auto">
        {/* Asset Listing Banner */}
        {prefillData && (prefillData.ownerName || prefillData.email) && (
          <div className="mb-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
            <div className="flex items-center space-x-2">
              <svg
                className="w-5 h-5 text-blue-600"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  strokeWidth={2}
                  d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"
                />
              </svg>
              <span className="text-blue-800 font-medium">
                Asset Listing: Form pre-filled with seller details from your
                profile
              </span>
            </div>
            <div className="mt-2 text-sm text-blue-700">
              Owner: {prefillData.ownerName || "N/A"} | Email:{" "}
              {prefillData.email || "N/A"}
            </div>
          </div>
        )}

        {/* Header */}
        <div className="flex justify-end items-center mb-8">
          <div className="text-sm font-medium text-text-muted">
            Step {currentStep + 1} of {steps.length}
          </div>
        </div>
        {/* Progress Bar */}
        <div className="mb-8">
          <div className="flex items-center justify-between mb-4">
            {steps.map((step, index) => (
              <div
                key={step.key}
                className={`flex items-center ${index < steps.length - 1 ? "flex-1" : ""}`}
              >
                <div
                  className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium ${
                    index <= currentStep
                      ? "bg-primary text-white"
                      : "bg-gray-200 text-text-muted"
                  }`}
                >
                  {index < currentStep ? (
                    <svg
                      className="w-5 h-5"
                      fill="none"
                      stroke="currentColor"
                      viewBox="0 0 24 24"
                    >
                      <path
                        strokeLinecap="round"
                        strokeLinejoin="round"
                        strokeWidth={2}
                        d="M5 13l4 4L19 7"
                      />
                    </svg>
                  ) : (
                    index + 1
                  )}
                </div>
                <span
                  className={`ml-2 text-sm font-medium ${
                    index <= currentStep ? "text-primary" : "text-text-muted"
                  }`}
                >
                  {step.title}
                </span>
                {index < steps.length - 1 && (
                  <div
                    className={`flex-1 h-0.5 mx-4 ${
                      index < currentStep ? "bg-primary" : "bg-gray-200"
                    }`}
                  />
                )}
              </div>
            ))}
          </div>
        </div>

        {/* Form Content */}
        <div className="bg-white rounded-lg border border-gray-200 p-8">
          <FormProvider {...methods}>
            <form onSubmit={methods.handleSubmit(handleSubmit)}>
              <CurrentStepComponent
                onNext={handleNext}
                onPrev={handlePrev}
                onSubmit={
                  currentStep === steps.length - 1 ? handleSubmit : undefined
                }
                isFirst={currentStep === 0}
                isLast={currentStep === steps.length - 1}
              />
            </form>
          </FormProvider>
        </div>

        {/* Loading Overlay */}
        {isSubmitting && (
          <div className="fixed inset-0 bg-white bg-opacity-80 flex items-center justify-center z-60">
            <div className="bg-white rounded-lg p-6 flex items-center space-x-4">
              <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-primary"></div>
              <span className="text-text-primary">Submitting property...</span>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default AddProperty;
