import React, { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../components/ui/select";
import { 
  TrendingUp, 
  TrendingDown,
  Eye, 
  Users, 
  DollarSign, 
  Clock,
  MapPin,
  Star,
  Phone,
  MessageCircle,
  Calendar,
  Target,
  BarChart3,
  PieChart,
  Activity,
  Zap,
  Award,
  ArrowUpRight,
  ArrowDownRight
} from "lucide-react";

// Advanced mock data for analytics
const mockAnalytics = {
  propertyPerformance: [
    {
      id: "prop-1",
      title: "Modern Downtown Apartment",
      views: 2340,
      inquiries: 45,
      viewsGrowth: 12.5,
      inquiriesGrowth: -2.1,
      avgTimeOnPage: "3:24",
      conversionRate: 1.92,
      priceViews: 890,
      contactClicks: 23,
      favorited: 67,
      shared: 12,
      virtualTours: 156,
      photoViews: 1240
    },
    {
      id: "prop-2", 
      title: "Luxury Beachfront Villa",
      views: 4560,
      inquiries: 89,
      viewsGrowth: 23.8,
      inquiriesGrowth: 15.4,
      avgTimeOnPage: "4:12",
      conversionRate: 1.95,
      priceViews: 1450,
      contactClicks: 45,
      favorited: 123,
      shared: 28,
      virtualTours: 234,
      photoViews: 2890
    }
  ],
  marketInsights: {
    avgMarketPrice: 485000,
    priceGrowth: 8.2,
    avgDaysOnMarket: 45,
    competitorAnalysis: [
      { name: "Similar Properties", avgPrice: 465000, avgDays: 52 },
      { name: "Your Properties", avgPrice: 495000, avgDays: 38 }
    ],
    demandScore: 8.4,
    marketTrend: "rising"
  },
  leadAnalytics: {
    totalLeads: 234,
    qualifiedLeads: 89,
    conversionRate: 38.0,
    avgResponseTime: "2.3 hours",
    leadSources: [
      { source: "Website", count: 89, percentage: 38 },
      { source: "Social Media", count: 67, percentage: 29 },
      { source: "Referrals", count: 45, percentage: 19 },
      { source: "Direct", count: 33, percentage: 14 }
    ],
    leadQuality: [
      { quality: "Hot", count: 23, color: "text-red-600" },
      { quality: "Warm", count: 66, color: "text-orange-600" },
      { quality: "Cold", count: 145, color: "text-blue-600" }
    ]
  },
  financialMetrics: {
    totalRevenue: 2450000,
    projectedRevenue: 3200000,
    commission: 147000,
    marketingSpend: 12500,
    roi: 1176,
    avgDealSize: 485000,
    monthlyTrend: [
      { month: "Jan", revenue: 245000 },
      { month: "Feb", revenue: 320000 },
      { month: "Mar", revenue: 410000 },
      { month: "Apr", revenue: 385000 },
      { month: "May", revenue: 520000 },
      { month: "Jun", revenue: 570000 }
    ]
  }
};

export default function AdvancedAnalytics() {
  const [activeTab, setActiveTab] = useState("overview");
  const [timeRange, setTimeRange] = useState("30d");

  const PropertyPerformance = () => (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-blue-700 font-medium">Total Views</p>
                <p className="text-2xl font-bold text-blue-800">
                  {mockAnalytics.propertyPerformance.reduce((sum, p) => sum + p.views, 0).toLocaleString()}
                </p>
              </div>
              <Eye className="w-6 h-6 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-green-700 font-medium">Inquiries</p>
                <p className="text-2xl font-bold text-green-800">
                  {mockAnalytics.propertyPerformance.reduce((sum, p) => sum + p.inquiries, 0)}
                </p>
              </div>
              <MessageCircle className="w-6 h-6 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-purple-700 font-medium">Avg Conversion</p>
                <p className="text-2xl font-bold text-purple-800">
                  {(mockAnalytics.propertyPerformance.reduce((sum, p) => sum + p.conversionRate, 0) / mockAnalytics.propertyPerformance.length).toFixed(1)}%
                </p>
              </div>
              <Target className="w-6 h-6 text-purple-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-orange-700 font-medium">Virtual Tours</p>
                <p className="text-2xl font-bold text-orange-800">
                  {mockAnalytics.propertyPerformance.reduce((sum, p) => sum + p.virtualTours, 0)}
                </p>
              </div>
              <Activity className="w-6 h-6 text-orange-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="space-y-4">
        {mockAnalytics.propertyPerformance.map((property) => (
          <Card key={property.id}>
            <CardContent className="p-6">
              <div className="flex justify-between items-start mb-4">
                <div>
                  <h3 className="text-lg font-semibold">{property.title}</h3>
                  <p className="text-sm text-muted-foreground">Performance Analytics</p>
                </div>
                <Badge className="bg-green-100 text-green-800">
                  {property.conversionRate}% conversion
                </Badge>
              </div>

              <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
                <div>
                  <p className="text-sm text-muted-foreground">Views</p>
                  <div className="flex items-center gap-2">
                    <p className="text-xl font-bold">{property.views.toLocaleString()}</p>
                    <div className={`flex items-center text-xs ${property.viewsGrowth > 0 ? 'text-green-600' : 'text-red-600'}`}>
                      {property.viewsGrowth > 0 ? <ArrowUpRight className="w-3 h-3" /> : <ArrowDownRight className="w-3 h-3" />}
                      {Math.abs(property.viewsGrowth)}%
                    </div>
                  </div>
                </div>

                <div>
                  <p className="text-sm text-muted-foreground">Inquiries</p>
                  <div className="flex items-center gap-2">
                    <p className="text-xl font-bold">{property.inquiries}</p>
                    <div className={`flex items-center text-xs ${property.inquiriesGrowth > 0 ? 'text-green-600' : 'text-red-600'}`}>
                      {property.inquiriesGrowth > 0 ? <ArrowUpRight className="w-3 h-3" /> : <ArrowDownRight className="w-3 h-3" />}
                      {Math.abs(property.inquiriesGrowth)}%
                    </div>
                  </div>
                </div>

                <div>
                  <p className="text-sm text-muted-foreground">Avg Time</p>
                  <p className="text-xl font-bold">{property.avgTimeOnPage}</p>
                </div>

                <div>
                  <p className="text-sm text-muted-foreground">Favorited</p>
                  <p className="text-xl font-bold">{property.favorited}</p>
                </div>
              </div>

              <div className="mt-4 pt-4 border-t">
                <div className="grid grid-cols-3 gap-4 text-sm">
                  <div className="flex justify-between">
                    <span>Price Views:</span>
                    <span className="font-medium">{property.priceViews}</span>
                  </div>
                  <div className="flex justify-between">
                    <span>Contact Clicks:</span>
                    <span className="font-medium">{property.contactClicks}</span>
                  </div>
                  <div className="flex justify-between">
                    <span>Shared:</span>
                    <span className="font-medium">{property.shared}</span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    </div>
  );

  const MarketInsights = () => (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div>
                <p className="text-sm text-green-700 font-medium">Market Demand</p>
                <p className="text-3xl font-bold text-green-800">{mockAnalytics.marketInsights.demandScore}/10</p>
              </div>
              <TrendingUp className="w-8 h-8 text-green-600" />
            </div>
            <p className="text-xs text-green-600">High demand in your area</p>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div>
                <p className="text-sm text-blue-700 font-medium">Avg Market Price</p>
                <p className="text-3xl font-bold text-blue-800">${(mockAnalytics.marketInsights.avgMarketPrice / 1000).toFixed(0)}K</p>
              </div>
              <DollarSign className="w-8 h-8 text-blue-600" />
            </div>
            <div className="flex items-center text-xs text-green-600">
              <ArrowUpRight className="w-3 h-3 mr-1" />
              +{mockAnalytics.marketInsights.priceGrowth}% this year
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between mb-4">
              <div>
                <p className="text-sm text-orange-700 font-medium">Days on Market</p>
                <p className="text-3xl font-bold text-orange-800">{mockAnalytics.marketInsights.avgDaysOnMarket}</p>
              </div>
              <Clock className="w-8 h-8 text-orange-600" />
            </div>
            <p className="text-xs text-orange-600">7 days faster than average</p>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Competitive Analysis</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {mockAnalytics.marketInsights.competitorAnalysis.map((comp, index) => (
              <div key={index} className="flex justify-between items-center p-4 bg-muted/30 rounded-lg">
                <div>
                  <p className="font-medium">{comp.name}</p>
                  <p className="text-sm text-muted-foreground">Average Performance</p>
                </div>
                <div className="text-right">
                  <p className="font-semibold">${comp.avgPrice.toLocaleString()}</p>
                  <p className="text-sm text-muted-foreground">{comp.avgDays} days avg</p>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );

  const LeadAnalytics = () => (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-purple-700 font-medium">Total Leads</p>
                <p className="text-2xl font-bold text-purple-800">{mockAnalytics.leadAnalytics.totalLeads}</p>
              </div>
              <Users className="w-6 h-6 text-purple-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-green-700 font-medium">Qualified</p>
                <p className="text-2xl font-bold text-green-800">{mockAnalytics.leadAnalytics.qualifiedLeads}</p>
              </div>
              <Star className="w-6 h-6 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-blue-700 font-medium">Conversion</p>
                <p className="text-2xl font-bold text-blue-800">{mockAnalytics.leadAnalytics.conversionRate}%</p>
              </div>
              <Target className="w-6 h-6 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-orange-700 font-medium">Response Time</p>
                <p className="text-2xl font-bold text-orange-800">{mockAnalytics.leadAnalytics.avgResponseTime}</p>
              </div>
              <Clock className="w-6 h-6 text-orange-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>Lead Sources</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {mockAnalytics.leadAnalytics.leadSources.map((source) => (
                <div key={source.source} className="flex justify-between items-center">
                  <span className="text-sm">{source.source}</span>
                  <div className="flex items-center gap-2">
                    <span className="text-sm font-medium">{source.count}</span>
                    <span className="text-xs text-muted-foreground">({source.percentage}%)</span>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Lead Quality Distribution</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {mockAnalytics.leadAnalytics.leadQuality.map((quality) => (
                <div key={quality.quality} className="flex justify-between items-center">
                  <span className="text-sm">{quality.quality} Leads</span>
                  <span className={`text-sm font-medium ${quality.color}`}>{quality.count}</span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );

  const FinancialMetrics = () => (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-green-700 font-medium">Total Revenue</p>
                <p className="text-2xl font-bold text-green-800">${(mockAnalytics.financialMetrics.totalRevenue / 1000000).toFixed(1)}M</p>
              </div>
              <DollarSign className="w-6 h-6 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-blue-700 font-medium">Commission</p>
                <p className="text-2xl font-bold text-blue-800">${(mockAnalytics.financialMetrics.commission / 1000).toFixed(0)}K</p>
              </div>
              <Award className="w-6 h-6 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-purple-700 font-medium">ROI</p>
                <p className="text-2xl font-bold text-purple-800">{mockAnalytics.financialMetrics.roi}%</p>
              </div>
              <TrendingUp className="w-6 h-6 text-purple-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-orange-700 font-medium">Avg Deal Size</p>
                <p className="text-2xl font-bold text-orange-800">${(mockAnalytics.financialMetrics.avgDealSize / 1000).toFixed(0)}K</p>
              </div>
              <BarChart3 className="w-6 h-6 text-orange-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Revenue Trend (6 Months)</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {mockAnalytics.financialMetrics.monthlyTrend.map((month) => (
              <div key={month.month} className="flex justify-between items-center p-3 bg-muted/30 rounded-lg">
                <span className="font-medium">{month.month}</span>
                <span className="text-lg font-bold">${(month.revenue / 1000).toFixed(0)}K</span>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Advanced Analytics</h1>
          <p className="text-muted-foreground">Deep insights into your property performance and market trends</p>
        </div>
        <Select value={timeRange} onValueChange={setTimeRange}>
          <SelectTrigger className="w-32">
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="7d">Last 7 days</SelectItem>
            <SelectItem value="30d">Last 30 days</SelectItem>
            <SelectItem value="90d">Last 3 months</SelectItem>
            <SelectItem value="1y">Last year</SelectItem>
          </SelectContent>
        </Select>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
        <TabsList className="grid w-full grid-cols-2 md:grid-cols-4 h-auto p-1 bg-gray-100 rounded-lg gap-1">
          <TabsTrigger 
            value="overview" 
            className={`flex flex-col items-center gap-1 p-3 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
              activeTab === "overview" 
                ? "bg-[#0056D2] text-white border-[#0056D2]" 
                : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
            }`}
          >
            <BarChart3 className="w-5 h-5" />
            <span className="font-medium text-xs">Property Performance</span>
          </TabsTrigger>
          <TabsTrigger 
            value="market" 
            className={`flex flex-col items-center gap-1 p-3 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
              activeTab === "market" 
                ? "bg-[#0056D2] text-white border-[#0056D2]" 
                : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
            }`}
          >
            <TrendingUp className="w-5 h-5" />
            <span className="font-medium text-xs">Market Insights</span>
          </TabsTrigger>
          <TabsTrigger 
            value="leads" 
            className={`flex flex-col items-center gap-1 p-3 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
              activeTab === "leads" 
                ? "bg-[#0056D2] text-white border-[#0056D2]" 
                : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
            }`}
          >
            <Users className="w-5 h-5" />
            <span className="font-medium text-xs">Lead Analytics</span>
          </TabsTrigger>
          <TabsTrigger 
            value="financial" 
            className={`flex flex-col items-center gap-1 p-3 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
              activeTab === "financial" 
                ? "bg-[#0056D2] text-white border-[#0056D2]" 
                : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
            }`}
          >
            <DollarSign className="w-5 h-5" />
            <span className="font-medium text-xs">Financial</span>
          </TabsTrigger>
        </TabsList>

        <TabsContent value="overview" className="mt-6">
          <PropertyPerformance />
        </TabsContent>

        <TabsContent value="market" className="mt-6">
          <MarketInsights />
        </TabsContent>

        <TabsContent value="leads" className="mt-6">
          <LeadAnalytics />
        </TabsContent>

        <TabsContent value="financial" className="mt-6">
          <FinancialMetrics />
        </TabsContent>
      </Tabs>
    </div>
  );
}