import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { Building2, TrendingUp, Home, Briefcase, Plus, Eye, Edit, Trash2 } from 'lucide-react';
import { Card, CardContent } from '../components/ui/card';
import { Button } from '../components/ui/button';
import { Badge } from '../components/ui/badge';
import { toast } from '../hooks/use-toast';
import { assetAPI, Asset } from '../../../../services/assetAPI';

const AssetDashboard = () => {
  const navigate = useNavigate();
  const [assets, setAssets] = useState<Asset[]>([]);
  const [loading, setLoading] = useState(true);
  const [stats, setStats] = useState({
    totalProperties: 0,
    totalValue: 0,
    residential: 0,
    commercial: 0
  });

  useEffect(() => {
    fetchAssets();
  }, []);

  const fetchAssets = async () => {
    try {
      setLoading(true);
      const data = await assetAPI.getAssets();
      setAssets(data);
      
      // Calculate stats
      const totalValue = data.reduce((sum, asset) => sum + (asset.current_value || 0), 0);
      const residential = data.filter(a => ['apartment', 'villa', 'house'].includes(a.property_type)).length;
      const commercial = data.filter(a => a.property_type === 'commercial').length;
      
      setStats({
        totalProperties: data.length,
        totalValue,
        residential,
        commercial
      });
    } catch (error) {
      console.error('Error fetching assets:', error);
      toast({ title: 'Failed to load assets', variant: 'destructive' });
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) return `₹${(amount / 10000000).toFixed(2)} Cr`;
    if (amount >= 100000) return `₹${(amount / 100000).toFixed(2)} L`;
    return `₹${amount.toLocaleString('en-IN')}`;
  };

  const handleViewDetails = (assetId: string) => {
    navigate(`/seller/dashboard/assets/${assetId}`);
  };

  const handleEdit = (assetId: string) => {
    toast({ title: 'Edit functionality coming soon!' });
  };

  const handleDelete = async (assetId: string) => {
    if (confirm('Are you sure you want to delete this asset?')) {
      try {
        await assetAPI.deleteAsset(assetId);
        setAssets(assets.filter(a => a.id !== assetId));
        toast({ title: 'Asset deleted successfully' });
      } catch (error) {
        toast({ title: 'Failed to delete asset', variant: 'destructive' });
      }
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'self_occupied': return 'bg-blue-100 text-blue-800';
      case 'rented': return 'bg-green-100 text-green-800';
      case 'vacant': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  return (
    <div className="min-h-screen bg-gray-50 p-6">
      {/* Header */}
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900 mb-2">My Property Assets</h1>
        <p className="text-gray-600">Manage and track your property portfolio</p>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 mb-1">Total Properties</p>
                <p className="text-3xl font-bold text-gray-900">{stats.totalProperties}</p>
              </div>
              <div className="p-3 bg-blue-100 rounded-lg">
                <Building2 className="w-6 h-6 text-blue-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 mb-1">Total Value</p>
                <p className="text-3xl font-bold text-gray-900">{formatCurrency(stats.totalValue)}</p>
              </div>
              <div className="p-3 bg-green-100 rounded-lg">
                <TrendingUp className="w-6 h-6 text-green-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 mb-1">Residential</p>
                <p className="text-3xl font-bold text-gray-900">{stats.residential}</p>
              </div>
              <div className="p-3 bg-purple-100 rounded-lg">
                <Home className="w-6 h-6 text-purple-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600 mb-1">Commercial</p>
                <p className="text-3xl font-bold text-gray-900">{stats.commercial}</p>
              </div>
              <div className="p-3 bg-orange-100 rounded-lg">
                <Briefcase className="w-6 h-6 text-orange-600" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Properties Grid */}
      <div className="mb-6 flex items-center justify-between">
        <h2 className="text-xl font-semibold text-gray-900">Your Properties</h2>
        <Button onClick={() => window.location.href = '/seller/dashboard/properties'}>
          <Plus className="w-4 h-4 mr-2" />
          Add Property
        </Button>
      </div>

      {loading ? (
        <div className="text-center py-12">
          <p className="text-gray-600">Loading assets...</p>
        </div>
      ) : assets.length === 0 ? (
        <Card className="p-12 text-center">
          <Building2 className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h3 className="text-xl font-semibold text-gray-900 mb-2">No Assets Yet</h3>
          <p className="text-gray-600 mb-4">Start building your property portfolio by adding your first asset</p>
          <Button onClick={() => window.location.href = '/seller/dashboard/properties'}>
            <Plus className="w-4 h-4 mr-2" />
            Add Your First Property
          </Button>
        </Card>
      ) : (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {assets.map((asset) => (
          <Card key={asset.id} className="overflow-hidden hover:shadow-lg transition-shadow">
            <div className="relative h-48">
              <img
                src={asset.image || 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=400'}
                alt={asset.title}
                className="w-full h-full object-cover"
              />
              <Badge className={`absolute top-3 right-3 ${getStatusColor(asset.occupancy_status)}`}>
                {asset.occupancy_status.replace('_', ' ').toUpperCase()}
              </Badge>
            </div>
            <CardContent className="p-4">
              <h3 className="text-lg font-semibold text-gray-900 mb-2">{asset.title}</h3>
              <p className="text-sm text-gray-600 mb-3">
                {asset.address}, {asset.city}, {asset.state}
              </p>
              <div className="flex items-center gap-4 text-sm text-gray-700 mb-3">
                <span>{asset.area} sqft</span>
                {asset.bedrooms && <span>{asset.bedrooms} BHK</span>}
              </div>
              {asset.current_value && (
                <p className="text-lg font-bold text-gray-900 mb-4">
                  {formatCurrency(asset.current_value)}
                </p>
              )}
              <div className="flex gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  className="flex-1"
                  onClick={() => handleViewDetails(asset.id)}
                >
                  <Eye className="w-4 h-4 mr-1" />
                  View
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handleEdit(asset.id)}
                >
                  <Edit className="w-4 h-4" />
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handleDelete(asset.id)}
                >
                  <Trash2 className="w-4 h-4 text-red-600" />
                </Button>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
      )}
    </div>
  );
};

export default AssetDashboard;
