import React, { useState, useEffect } from 'react';
import { useRoute, useLocation } from 'wouter';
import { ArrowLeft, Home, Upload, Plus, Download, Trash2, Calendar, DollarSign, Edit2 } from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from '../components/ui/card';
import { Button } from '../components/ui/button';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../components/ui/tabs';
import { Badge } from '../components/ui/badge';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from '../components/ui/dialog';
import { Input } from '../components/ui/input';
import { Label } from '../components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../components/ui/select';
import { Textarea } from '../components/ui/textarea';
import { toast } from '../hooks/use-toast';
import { assetAPI, Asset, AssetDocument, AssetTransaction } from '../../../../services/assetAPI';

const AssetDetails = () => {
  const [, params] = useRoute('/assets/:id');
  const [location, setLocation] = useLocation();
  const id = params?.id;
  
  const [asset, setAsset] = useState<Asset | null>(null);
  const [documents, setDocuments] = useState<AssetDocument[]>([]);
  const [transactions, setTransactions] = useState<AssetTransaction[]>([]);
  const [loading, setLoading] = useState(true);
  const [showDocModal, setShowDocModal] = useState(false);
  const [showTxModal, setShowTxModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const [editField, setEditField] = useState<'purchase_date' | 'purchase_price' | 'current_value' | null>(null);
  const [editValue, setEditValue] = useState('');
  const [uploadFile, setUploadFile] = useState<File | null>(null);
  const [docCategory, setDocCategory] = useState('');
  const [docDescription, setDocDescription] = useState('');
  const [txData, setTxData] = useState({
    type: 'income' as 'income' | 'expense',
    amount: 0,
    category: '',
    transaction_date: new Date().toISOString().split('T')[0],
    description: ''
  });

  useEffect(() => {
    fetchAssetData();
  }, [id]);

  const fetchAssetData = async () => {
    if (!id) return;
    
    setLoading(true);
    try {
      const [assetData, docs, txs] = await Promise.all([
        assetAPI.getAssetById(id),
        assetAPI.getDocuments(id),
        assetAPI.getTransactions(id)
      ]);
      
      setAsset(assetData);
      setDocuments(docs);
      setTransactions(txs);
    } catch (error: any) {
      console.error('Error:', error);
      toast({ 
        title: 'Error loading asset', 
        description: error.message,
        variant: 'destructive' 
      });
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) return `₹${(amount / 10000000).toFixed(2)} Cr`;
    if (amount >= 100000) return `₹${(amount / 100000).toFixed(2)} L`;
    return `₹${amount.toLocaleString('en-IN')}`;
  };

  const totalIncome = transactions.filter(t => t.type === 'income').reduce((sum, t) => sum + t.amount, 0);
  const totalExpense = transactions.filter(t => t.type === 'expense').reduce((sum, t) => sum + t.amount, 0);
  const netProfit = totalIncome - totalExpense;

  const handleUploadDocument = async () => {
    if (!uploadFile || !docCategory) {
      toast({ title: 'Please select file and category', variant: 'destructive' });
      return;
    }
    try {
      console.log('Uploading document:', uploadFile.name, docCategory);
      const doc = await assetAPI.uploadDocument(id!, uploadFile, docCategory, docDescription);
      console.log('Document uploaded:', doc);
      setDocuments([...documents, doc]);
      toast({ title: 'Document uploaded successfully' });
      setShowDocModal(false);
      setUploadFile(null);
      setDocCategory('');
      setDocDescription('');
    } catch (error: any) {
      console.error('Upload error:', error);
      toast({ title: 'Failed to upload document', description: error.message, variant: 'destructive' });
    }
  };

  const handleDownloadDocument = async (docId: string) => {
    try {
      const blob = await assetAPI.downloadDocument(docId);
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = 'document';
      a.click();
      window.URL.revokeObjectURL(url);
    } catch (error: any) {
      console.error('Download error:', error);
      toast({ title: 'Failed to download document', description: error.message, variant: 'destructive' });
    }
  };

  const handleDeleteDocument = async (docId: string) => {
    try {
      console.log('Deleting document:', docId);
      await assetAPI.deleteDocument(docId);
      setDocuments(documents.filter(d => d.id !== docId));
      toast({ title: 'Document deleted successfully' });
    } catch (error: any) {
      console.error('Delete error:', error);
      toast({ title: 'Failed to delete document', description: error.message, variant: 'destructive' });
    }
  };

  const handleAddTransaction = async () => {
    if (!txData.amount || !txData.category) {
      toast({ title: 'Please fill all required fields', variant: 'destructive' });
      return;
    }
    try {
      console.log('Adding transaction:', txData);
      const tx = await assetAPI.addTransaction(id!, txData);
      console.log('Transaction added:', tx);
      setTransactions([...transactions, tx]);
      toast({ title: 'Transaction added successfully' });
      setShowTxModal(false);
      setTxData({
        type: 'income',
        amount: 0,
        category: '',
        transaction_date: new Date().toISOString().split('T')[0],
        description: ''
      });
    } catch (error: any) {
      console.error('Transaction error:', error);
      toast({ title: 'Failed to add transaction', description: error.message, variant: 'destructive' });
    }
  };

  const handleDeleteTransaction = async (txId: string) => {
    try {
      await assetAPI.deleteTransaction(txId);
      setTransactions(transactions.filter(t => t.id !== txId));
      toast({ title: 'Transaction deleted successfully' });
    } catch (error) {
      toast({ title: 'Failed to delete transaction', variant: 'destructive' });
    }
  };

  const handleEditField = (field: 'purchase_date' | 'purchase_price' | 'current_value') => {
    setEditField(field);
    if (field === 'purchase_date') {
      setEditValue(asset?.purchase_date ? new Date(asset.purchase_date).toISOString().split('T')[0] : '');
    } else if (field === 'purchase_price') {
      setEditValue(asset?.purchase_price?.toString() || '');
    } else if (field === 'current_value') {
      setEditValue(asset?.current_value?.toString() || '');
    }
    setShowEditModal(true);
  };

  const handleSaveEdit = async () => {
    if (!editField || !editValue) {
      toast({ title: 'Please enter a value', variant: 'destructive' });
      return;
    }
    
    // TODO: Backend API not implemented yet
    toast({ 
      title: 'Feature Coming Soon', 
      description: 'Edit functionality will be available once backend is ready.',
    });
    setShowEditModal(false);
    setEditField(null);
    setEditValue('');
    
    /* Uncomment when backend is ready:
    try {
      console.log('Saving edit:', editField, editValue);
      const updateData: any = {};
      if (editField === 'purchase_date') {
        updateData.purchase_date = editValue;
      } else if (editField === 'purchase_price') {
        updateData.purchase_price = Number(editValue);
      } else if (editField === 'current_value') {
        updateData.current_value = Number(editValue);
      }
      
      console.log('Update data:', updateData);
      const updatedAsset = await assetAPI.updateAsset(id!, updateData);
      console.log('Updated asset:', updatedAsset);
      setAsset({ ...asset!, ...updateData });
      toast({ title: 'Updated successfully' });
      setShowEditModal(false);
      setEditField(null);
      setEditValue('');
    } catch (error: any) {
      console.error('Update error:', error);
      toast({ title: 'Failed to update', description: error.message, variant: 'destructive' });
    }
    */
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 p-6 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading asset details...</p>
        </div>
      </div>
    );
  }

  if (!asset) {
    return (
      <div className="min-h-screen bg-gray-50 p-6 flex items-center justify-center">
        <div className="text-center">
          <p className="text-gray-600 mb-4">Asset not found</p>
          <Button onClick={() => setLocation('/assets')}>Back to Assets</Button>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 p-6">
      <Button variant="ghost" onClick={() => setLocation('/assets')} className="mb-4">
        <ArrowLeft className="w-4 h-4 mr-2" />
        Back to Assets
      </Button>

      <div className="mb-6">
        <h1 className="text-3xl font-bold text-gray-900 mb-2">{asset.title}</h1>
        <p className="text-gray-600">{asset.address}, {asset.city}</p>
      </div>

      <Tabs defaultValue="info" className="space-y-6">
        <TabsList>
          <TabsTrigger value="info">Property Info</TabsTrigger>
          <TabsTrigger value="documents">Documents ({documents.length})</TabsTrigger>
          <TabsTrigger value="financials">Financials</TabsTrigger>
        </TabsList>

        <TabsContent value="info">
          <Card>
            <CardHeader>
              <CardTitle className="text-gray-900">Property Details</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div>
                  <p className="text-sm text-gray-600 mb-1">Property Type</p>
                  <p className="text-base font-medium text-gray-900 capitalize">{asset.property_type}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600 mb-1">Area</p>
                  <p className="text-base font-medium text-gray-900">{asset.area} sqft</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600 mb-1">Bedrooms</p>
                  <p className="text-base font-medium text-gray-900">{asset.bedrooms} BHK</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600 mb-1">Bathrooms</p>
                  <p className="text-base font-medium text-gray-900">{asset.bathrooms}</p>
                </div>
                <div>
                  <p className="text-sm text-gray-600 mb-1">Purchase Date</p>
                  <div className="flex items-center gap-2">
                    <p className="text-base font-medium text-gray-900">
                      {asset.purchase_date ? new Date(asset.purchase_date).toLocaleDateString() : 'Not specified'}
                    </p>
                    <Button 
                      variant="outline" 
                      size="sm" 
                      onClick={() => handleEditField('purchase_date')}
                      className="h-7 px-2 text-xs bg-gray-100 text-gray-900 hover:bg-gray-200 border-gray-300 font-medium"
                    >
                      <Edit2 className="w-3 h-3 mr-1" />
                      Edit
                    </Button>
                  </div>
                </div>
                <div>
                  <p className="text-sm text-gray-600 mb-1">Purchase Price</p>
                  <div className="flex items-center gap-2">
                    <p className="text-base font-medium text-gray-900">
                      {asset.purchase_price ? formatCurrency(asset.purchase_price) : 'Not specified'}
                    </p>
                    <Button 
                      variant="outline" 
                      size="sm" 
                      onClick={() => handleEditField('purchase_price')}
                      className="h-7 px-2 text-xs bg-gray-100 text-gray-900 hover:bg-gray-200 border-gray-300 font-medium"
                    >
                      <Edit2 className="w-3 h-3 mr-1" />
                      Edit
                    </Button>
                  </div>
                </div>
                <div>
                  <p className="text-sm text-gray-600 mb-1">Current Value</p>
                  <div className="flex items-center gap-2">
                    <p className="text-base font-medium text-gray-900">
                      {asset.current_value ? formatCurrency(asset.current_value) : 'Not specified'}
                    </p>
                    <Button 
                      variant="outline" 
                      size="sm" 
                      onClick={() => handleEditField('current_value')}
                      className="h-7 px-2 text-xs bg-gray-100 text-gray-900 hover:bg-gray-200 border-gray-300 font-medium"
                    >
                      <Edit2 className="w-3 h-3 mr-1" />
                      Edit
                    </Button>
                  </div>
                </div>
                <div>
                  <p className="text-sm text-gray-600 mb-1">Appreciation</p>
                  <p className="text-base font-medium text-gray-900">
                    {asset.purchase_price && asset.current_value ? (
                      <span className={asset.current_value >= asset.purchase_price ? 'text-green-600' : 'text-red-600'}>
                        {asset.current_value >= asset.purchase_price ? '+' : ''}
                        {formatCurrency(asset.current_value - asset.purchase_price)} 
                        ({(((asset.current_value - asset.purchase_price) / asset.purchase_price) * 100).toFixed(1)}%)
                      </span>
                    ) : (
                      'Not available'
                    )}
                  </p>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="documents">
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-gray-900">Documents</CardTitle>
                <Dialog open={showDocModal} onOpenChange={setShowDocModal}>
                  <DialogTrigger asChild>
                    <Button>
                      <Upload className="w-4 h-4 mr-2" />
                      Upload Document
                    </Button>
                  </DialogTrigger>
                  <DialogContent>
                    <DialogHeader>
                      <DialogTitle className="text-gray-900">Upload Document</DialogTitle>
                    </DialogHeader>
                    <div className="space-y-4">
                      <div>
                        <Label className="text-gray-900">Category</Label>
                        <Select value={docCategory} onValueChange={setDocCategory}>
                          <SelectTrigger>
                            <SelectValue placeholder="Select category" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="purchase">Purchase Documents</SelectItem>
                            <SelectItem value="legal">Legal Documents</SelectItem>
                            <SelectItem value="tax">Tax Documents</SelectItem>
                            <SelectItem value="other">Other</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                      <div>
                        <Label className="text-gray-900">File</Label>
                        <Input type="file" onChange={(e) => setUploadFile(e.target.files?.[0] || null)} />
                      </div>
                      <div>
                        <Label className="text-gray-900">Description</Label>
                        <Textarea 
                          placeholder="Add description..." 
                          value={docDescription}
                          onChange={(e) => setDocDescription(e.target.value)}
                        />
                      </div>
                      <Button className="w-full" onClick={handleUploadDocument}>
                        Upload
                      </Button>
                    </div>
                  </DialogContent>
                </Dialog>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {documents.map((doc) => (
                  <div key={doc.id} className="flex items-center justify-between p-4 border rounded-lg">
                    <div className="flex items-center gap-3">
                      <div className="p-2 bg-blue-100 rounded">
                        <Home className="w-5 h-5 text-blue-600" />
                      </div>
                      <div>
                        <p className="font-medium text-gray-900">{doc.file_name}</p>
                        <p className="text-sm text-gray-600">{doc.category} • {(doc.file_size / 1024 / 1024).toFixed(2)} MB • {new Date(doc.uploaded_at).toLocaleDateString()}</p>
                      </div>
                    </div>
                    <div className="flex gap-2">
                      <Button variant="outline" size="sm" onClick={() => handleDownloadDocument(doc.id)}>
                        <Download className="w-4 h-4" />
                      </Button>
                      <Button variant="outline" size="sm" onClick={() => handleDeleteDocument(doc.id)}>
                        <Trash2 className="w-4 h-4 text-red-600" />
                      </Button>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="financials">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
            <Card>
              <CardContent className="p-6">
                <p className="text-sm text-gray-600 mb-1">Total Income</p>
                <p className="text-2xl font-bold text-green-600">{formatCurrency(totalIncome)}</p>
              </CardContent>
            </Card>
            <Card>
              <CardContent className="p-6">
                <p className="text-sm text-gray-600 mb-1">Total Expense</p>
                <p className="text-2xl font-bold text-red-600">{formatCurrency(totalExpense)}</p>
              </CardContent>
            </Card>
            <Card>
              <CardContent className="p-6">
                <p className="text-sm text-gray-600 mb-1">Net Profit</p>
                <p className={`text-2xl font-bold ${netProfit >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                  {formatCurrency(Math.abs(netProfit))}
                </p>
              </CardContent>
            </Card>
          </div>

          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="text-gray-900">Transactions</CardTitle>
                <Dialog open={showTxModal} onOpenChange={setShowTxModal}>
                  <DialogTrigger asChild>
                    <Button>
                      <Plus className="w-4 h-4 mr-2" />
                      Add Transaction
                    </Button>
                  </DialogTrigger>
                  <DialogContent>
                    <DialogHeader>
                      <DialogTitle className="text-gray-900">Add Transaction</DialogTitle>
                    </DialogHeader>
                    <div className="space-y-4">
                      <div>
                        <Label className="text-gray-900">Type</Label>
                        <Select value={txData.type} onValueChange={(value: 'income' | 'expense') => setTxData({...txData, type: value})}>
                          <SelectTrigger>
                            <SelectValue placeholder="Select type" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="income">Income</SelectItem>
                            <SelectItem value="expense">Expense</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                      <div>
                        <Label className="text-gray-900">Amount</Label>
                        <Input 
                          type="number" 
                          placeholder="Enter amount" 
                          value={txData.amount || ''}
                          onChange={(e) => setTxData({...txData, amount: Number(e.target.value)})}
                        />
                      </div>
                      <div>
                        <Label className="text-gray-900">Category</Label>
                        <Input 
                          placeholder="e.g., Rent, Tax, Maintenance" 
                          value={txData.category}
                          onChange={(e) => setTxData({...txData, category: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label className="text-gray-900">Date</Label>
                        <Input 
                          type="date" 
                          value={txData.transaction_date}
                          onChange={(e) => setTxData({...txData, transaction_date: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label className="text-gray-900">Description</Label>
                        <Textarea 
                          placeholder="Add description..." 
                          value={txData.description}
                          onChange={(e) => setTxData({...txData, description: e.target.value})}
                        />
                      </div>
                      <Button className="w-full" onClick={handleAddTransaction}>
                        Save
                      </Button>
                    </div>
                  </DialogContent>
                </Dialog>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {transactions.map((tx) => (
                  <div key={tx.id} className="flex items-center justify-between p-4 border rounded-lg">
                    <div className="flex items-center gap-3">
                      <div className={`p-2 rounded ${tx.type === 'income' ? 'bg-green-100' : 'bg-red-100'}`}>
                        <DollarSign className={`w-5 h-5 ${tx.type === 'income' ? 'text-green-600' : 'text-red-600'}`} />
                      </div>
                      <div>
                        <p className="font-medium text-gray-900">{tx.category}</p>
                        <p className="text-sm text-gray-600">{tx.description} • {new Date(tx.transaction_date).toLocaleDateString()}</p>
                      </div>
                    </div>
                    <div className="flex items-center gap-3">
                      <p className={`font-bold ${tx.type === 'income' ? 'text-green-600' : 'text-red-600'}`}>
                        {tx.type === 'income' ? '+' : '-'}{formatCurrency(tx.amount)}
                      </p>
                      <Button variant="outline" size="sm" onClick={() => handleDeleteTransaction(tx.id)}>
                        <Trash2 className="w-4 h-4 text-red-600" />
                      </Button>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      {/* Edit Modal */}
      <Dialog open={showEditModal} onOpenChange={setShowEditModal}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle className="text-gray-900">
              Edit {editField === 'purchase_date' ? 'Purchase Date' : editField === 'purchase_price' ? 'Purchase Price' : 'Current Value'}
            </DialogTitle>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label className="text-gray-900">
                {editField === 'purchase_date' ? 'Date' : 'Amount (₹)'}
              </Label>
              {editField === 'purchase_date' ? (
                <Input 
                  type="date" 
                  value={editValue}
                  onChange={(e) => setEditValue(e.target.value)}
                />
              ) : (
                <Input 
                  type="number" 
                  placeholder="Enter amount"
                  value={editValue}
                  onChange={(e) => setEditValue(e.target.value)}
                />
              )}
            </div>
            <div className="flex gap-2">
              <Button className="flex-1" onClick={handleSaveEdit}>
                Save
              </Button>
              <Button variant="outline" className="flex-1" onClick={() => setShowEditModal(false)}>
                Cancel
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default AssetDetails;
