import { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../../../ui/card';
import { Button } from '../../../ui/button';
import { Badge } from '../../../ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../../../ui/tabs';
import { Clock, Users, Gavel, Eye, Calendar, IndianRupee } from 'lucide-react';
import { Link } from 'wouter';

interface AuctionSession {
  id: string; // Changed to string for UUID
  propertyId: number;
  propertyTitle: string;
  propertyImage: string;
  status: 'active' | 'upcoming' | 'ended' | 'cancelled' | 'completed' | 'scheduled';
  startTime: string;
  endTime: string;
  highestBid: number;
  bidderCount: number;
  reservePrice: number;
}

export default function AuctionBidding() {
  const [auctions, setAuctions] = useState<AuctionSession[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchAuctions();
  }, []);

  const fetchAuctions = async () => {
    try {
      const token = localStorage.getItem('sellerToken');
      const response = await fetch('http://localhost:8090/api/auction-management/my-sessions', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      if (response.ok) {
        const data = await response.json();
        // Transform backend data to frontend format
        const transformedAuctions = data.sessions.map((session: any) => ({
          id: session.id,
          propertyId: session.property_id,
          propertyTitle: `Property ${session.property_id}`,
          propertyImage: '',
          status: session.session_status, // Use actual status from backend
          startTime: session.start_time,
          endTime: session.end_time,
          highestBid: session.current_highest_bid,
          bidderCount: session.registered_bidders,
          reservePrice: session.reserve_price || session.starting_bid_price
        }));
        setAuctions(transformedAuctions);
      }
    } catch (error) {
      console.error('Error fetching auctions:', error);
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) return `₹${(amount / 10000000).toFixed(1)} Cr`;
    if (amount >= 100000) return `₹${(amount / 100000).toFixed(1)} L`;
    return `₹${amount.toLocaleString()}`;
  };

  const formatDateTime = (dateString: string) => {
    return new Date(dateString).toLocaleString('en-IN', {
      day: 'numeric',
      month: 'short',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const getStatusBadge = (status: string) => {
    const styles = {
      active: 'bg-green-100 text-green-800 border-green-200',
      scheduled: 'bg-blue-100 text-blue-800 border-blue-200',
      upcoming: 'bg-blue-100 text-blue-800 border-blue-200',
      completed: 'bg-gray-100 text-gray-800 border-gray-200',
      ended: 'bg-gray-100 text-gray-800 border-gray-200',
      cancelled: 'bg-red-100 text-red-800 border-red-200'
    };
    return styles[status as keyof typeof styles] || styles.ended;
  };

  const filterAuctions = (status: string) => {
    return auctions.filter(auction => auction.status === status);
  };

  const AuctionCard = ({ auction }: { auction: AuctionSession }) => (
    <Card className="hover:shadow-lg transition-shadow">
      <CardContent className="p-4">
        <div className="flex gap-4">
          <div className="w-20 h-20 bg-gray-200 rounded-lg overflow-hidden flex-shrink-0">
            {auction.propertyImage ? (
              <img 
                src={auction.propertyImage} 
                alt={auction.propertyTitle}
                className="w-full h-full object-cover"
              />
            ) : (
              <div className="w-full h-full flex items-center justify-center">
                <IndianRupee className="w-8 h-8 text-gray-400" />
              </div>
            )}
          </div>

          <div className="flex-1 min-w-0">
            <div className="flex items-start justify-between mb-2">
              <h3 className="font-semibold text-gray-900 truncate pr-2">
                {auction.propertyTitle}
              </h3>
              <Badge className={`${getStatusBadge(auction.status)} border text-xs`}>
                {auction.status.charAt(0).toUpperCase() + auction.status.slice(1)}
              </Badge>
            </div>

            <div className="space-y-2 text-sm text-gray-600">
              <div className="flex items-center gap-4">
                <div className="flex items-center">
                  <Clock className="w-4 h-4 mr-1" />
                  <span>{formatDateTime(auction.startTime)} - {formatDateTime(auction.endTime)}</span>
                </div>
              </div>

              <div className="flex items-center gap-4">
                <div className="flex items-center">
                  <Gavel className="w-4 h-4 mr-1 text-green-600" />
                  <span className="font-medium text-green-600">
                    {formatCurrency(auction.highestBid)}
                  </span>
                </div>
                <div className="flex items-center">
                  <Users className="w-4 h-4 mr-1" />
                  <span>{auction.bidderCount} bidders</span>
                </div>
              </div>
            </div>

            <div className="flex gap-2 mt-3">
              <Link href={`/seller/dashboard/auction-session/${auction.propertyId}/${auction.id}`}>
                <Button size="sm" className="bg-blue-600 hover:bg-blue-700">
                  <Eye className="w-4 h-4 mr-1" />
                  View Details
                </Button>
              </Link>
            </div>
          </div>
        </div>
      </CardContent>
    </Card>
  );

  if (loading) {
    return (
      <div className="flex-1 p-6">
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Auction Dashboard</h1>
          <p className="text-gray-600 mt-1">Monitor and manage your property auctions</p>
        </div>
      </div>

      <Tabs defaultValue="all" className="w-full">
        <TabsList className="grid w-full grid-cols-6">
          <TabsTrigger value="all" className="flex items-center gap-2">
            All ({auctions.length})
          </TabsTrigger>
          <TabsTrigger value="active" className="flex items-center gap-2">
            <div className="w-2 h-2 bg-green-500 rounded-full"></div>
            Active ({filterAuctions('active').length})
          </TabsTrigger>
          <TabsTrigger value="scheduled" className="flex items-center gap-2">
            <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
            Scheduled ({filterAuctions('scheduled').length})
          </TabsTrigger>
          <TabsTrigger value="completed" className="flex items-center gap-2">
            <div className="w-2 h-2 bg-gray-500 rounded-full"></div>
            Completed ({filterAuctions('completed').length})
          </TabsTrigger>
          <TabsTrigger value="cancelled" className="flex items-center gap-2">
            <div className="w-2 h-2 bg-red-500 rounded-full"></div>
            Cancelled ({filterAuctions('cancelled').length})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="all" className="space-y-4">
          {auctions.length > 0 ? (
            <div className="grid gap-4">
              {auctions.map(auction => (
                <AuctionCard key={auction.id} auction={auction} />
              ))}
            </div>
          ) : (
            <Card>
              <CardContent className="p-8 text-center">
                <Gavel className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No auctions found</h3>
                <p className="text-gray-600">Create your first auction to get started</p>
              </CardContent>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="active" className="space-y-4">
          {filterAuctions('active').length > 0 ? (
            <div className="grid gap-4">
              {filterAuctions('active').map(auction => (
                <AuctionCard key={auction.id} auction={auction} />
              ))}
            </div>
          ) : (
            <Card>
              <CardContent className="p-8 text-center">
                <Gavel className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No active auctions</h3>
                <p className="text-gray-600">Start an auction to see it here</p>
              </CardContent>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="scheduled" className="space-y-4">
          {filterAuctions('scheduled').length > 0 ? (
            <div className="grid gap-4">
              {filterAuctions('scheduled').map(auction => (
                <AuctionCard key={auction.id} auction={auction} />
              ))}
            </div>
          ) : (
            <Card>
              <CardContent className="p-8 text-center">
                <Calendar className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No scheduled auctions</h3>
                <p className="text-gray-600">Schedule an auction to see it here</p>
              </CardContent>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="completed" className="space-y-4">
          {filterAuctions('completed').length > 0 ? (
            <div className="grid gap-4">
              {filterAuctions('completed').map(auction => (
                <AuctionCard key={auction.id} auction={auction} />
              ))}
            </div>
          ) : (
            <Card>
              <CardContent className="p-8 text-center">
                <Clock className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No completed auctions</h3>
                <p className="text-gray-600">Completed auctions will appear here</p>
              </CardContent>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="cancelled" className="space-y-4">
          {filterAuctions('cancelled').length > 0 ? (
            <div className="grid gap-4">
              {filterAuctions('cancelled').map(auction => (
                <AuctionCard key={auction.id} auction={auction} />
              ))}
            </div>
          ) : (
            <Card>
              <CardContent className="p-8 text-center">
                <Clock className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <h3 className="text-lg font-medium text-gray-900 mb-2">No cancelled auctions</h3>
                <p className="text-gray-600">Cancelled auctions will appear here</p>
              </CardContent>
            </Card>
          )}
        </TabsContent>
      </Tabs>
    </div>
  );
}