import { useState, useEffect } from 'react';
import { useRoute } from 'wouter';
import { Card, CardContent, CardHeader, CardTitle } from '../../../ui/card';
import { Button } from '../../../ui/button';
import { Badge } from '../../../ui/badge';
import { 
  Clock, Users, Gavel, MapPin, Calendar, IndianRupee, 
  TrendingUp, AlertCircle, Play, Square, Plus, ArrowLeft 
} from 'lucide-react';
import { Link } from 'wouter';

interface Bid {
  id: number;
  amount: number;
  bidderName: string;
  bidderEmail: string;
  timestamp: string;
}

interface AuctionDetail {
  id: string;
  propertyId: number;
  propertyTitle: string;
  propertyImage: string;
  propertyLocation: string;
  status: 'active' | 'upcoming' | 'ended';
  startTime: string;
  endTime: string;
  reservePrice: number;
  startingPrice: number;
  bidIncrement: number;
  highestBid: number;
  bidderCount: number;
  bids: Bid[];
}

export default function AuctionSession() {
  const [match, params] = useRoute('/auction-session/:propertyId/:sessionId');
  const [auction, setAuction] = useState<AuctionDetail | null>(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (params?.sessionId) {
      fetchAuctionDetail();
      
      // Set up auto-refresh every 5 seconds for live updates
      const interval = setInterval(() => {
        fetchAuctionDetail();
      }, 5000);
      
      return () => clearInterval(interval);
    }
  }, [params?.sessionId]);

  const fetchAuctionDetail = async () => {
    try {
      console.log('Fetching auction details for session:', params?.sessionId);
      
      // Get auction session status first
      const statusResponse = await fetch(`http://localhost:8090/api/auctions/${params?.sessionId}/current-status`);
      if (!statusResponse.ok) {
        throw new Error(`Failed to fetch auction status: ${statusResponse.status}`);
      }
      const statusData = await statusResponse.json();
      console.log('Auction status data:', statusData);
      
      // Get bid history with user details
      const bidResponse = await fetch(`http://localhost:8090/api/auctions/${params?.sessionId}/bid-history`);
      let bidData = { bids: [] };
      if (bidResponse.ok) {
        bidData = await bidResponse.json();
        console.log('Bid history data:', bidData);
        console.log('First bid details:', bidData.bids?.[0]);
        console.log('First bid full_name:', bidData.bids?.[0]?.full_name);
        console.log('First bid email:', bidData.bids?.[0]?.email);
      } else {
        console.warn('Failed to fetch bid history:', bidResponse.status);
      }
      
      // Convert to component format
      const auctionDetail = {
        id: statusData.session_id || params?.sessionId,
        propertyId: statusData.property_id || parseInt(params?.propertyId || '0'),
        propertyTitle: statusData.property_title || `Property ${statusData.property_id || params?.propertyId}`,
        propertyImage: statusData.property_image || null,
        propertyLocation: statusData.property_location || 'Location not specified',
        status: statusData.session?.session_status || statusData.session_status || 'active',
        startTime: statusData.session?.start_time || statusData.start_time || new Date().toISOString(),
        endTime: statusData.session?.end_time || statusData.end_time || new Date().toISOString(),
        reservePrice: statusData.session?.reserve_price || statusData.reserve_price || statusData.starting_bid_price || 0,
        startingPrice: statusData.session?.starting_bid_price || statusData.starting_bid_price || 0,
        bidIncrement: statusData.session?.minimum_increment || statusData.minimum_increment || 50000,
        highestBid: statusData.current_highest_bid?.bid_amount || statusData.starting_bid_price || 0,
        bidderCount: statusData.session?.registered_bidders || statusData.registered_bidders || 0,
        bids: (bidData.bids || []).map((bid: any) => ({
          id: bid.id || bid.bid_id,
          amount: parseFloat(bid.bid_amount || bid.amount || 0),
          bidderName: bid.full_name || bid.user_name || 'Anonymous Bidder',
          bidderEmail: bid.email || bid.user_email || '',
          timestamp: bid.bid_time || bid.timestamp || new Date().toISOString()
        }))
      };
      
      console.log('Final auction detail:', auctionDetail);
      setAuction(auctionDetail);
    } catch (error) {
      console.error('Error fetching auction details:', error);
      setAuction(null);
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) return `₹${(amount / 10000000).toFixed(1)} Cr`;
    if (amount >= 100000) return `₹${(amount / 100000).toFixed(1)} L`;
    return `₹${amount.toLocaleString()}`;
  };

  const formatDateTime = (dateString: string) => {
    return new Date(dateString).toLocaleString('en-IN', {
      day: 'numeric',
      month: 'short',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const getStatusBadge = (status: string) => {
    const styles = {
      active: 'bg-green-100 text-green-700',
      upcoming: 'bg-blue-100 text-blue-700',
      scheduled: 'bg-blue-100 text-blue-700',
      ended: 'bg-gray-100 text-gray-700',
      completed: 'bg-gray-100 text-gray-700',
      cancelled: 'bg-red-100 text-red-700'
    };
    return styles[status as keyof typeof styles] || styles.ended;
  };

  const handleCloseAuction = async () => {
    try {
      // Check all possible token keys
      const allKeys = Object.keys(localStorage);
      console.log('All localStorage keys:', allKeys);
      
      const token = localStorage.getItem('sellerToken');
      console.log('Token found:', token ? 'Yes' : 'No');
      console.log('Token value:', token);
      
      const headers = {
        'Content-Type': 'application/json'
      };
      
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }
      
      const response = await fetch(`http://localhost:8090/api/auction-management/${params?.sessionId}/cancel`, {
        method: 'POST',
        headers
      });
      if (response.ok) {
        fetchAuctionDetail();
      } else {
        console.error('Close auction failed:', response.status, response.statusText);
      }
    } catch (error) {
      console.error('Error closing auction:', error);
    }
  };

  const handleExtendTime = async () => {
    try {
      const token = localStorage.getItem('sellerToken');
      const response = await fetch(`http://localhost:8090/api/auction-management/${params?.sessionId}/extend`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({ extension_minutes: 30 })
      });
      if (response.ok) {
        fetchAuctionDetail();
      }
    } catch (error) {
      console.error('Error extending auction:', error);
    }
  };

  if (loading) {
    return (
      <div className="flex-1 p-6">
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      </div>
    );
  }

  if (!auction) {
    return (
      <div className="flex-1 p-6">
        <div className="text-center py-12">
          <AlertCircle className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">Auction not found</h3>
          <p className="text-gray-600">The auction session could not be loaded</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-white">
      {/* Simple Header */}
      <div className="border-b px-6 py-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <Link href="/auction-bidding">
              <Button variant="ghost" size="sm">
                <ArrowLeft className="w-4 h-4 mr-2" />
                Back
              </Button>
            </Link>
            <div>
              <h1 className="text-xl font-semibold">{auction.propertyTitle}</h1>
              <p className="text-sm text-gray-500">{auction.propertyLocation}</p>
            </div>
          </div>
          <Badge className={`${getStatusBadge(auction.status)} px-3 py-1`}>
            {auction.status.charAt(0).toUpperCase() + auction.status.slice(1)}
          </Badge>
        </div>
      </div>

      {/* Main Content */}
      <div className="p-6">
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Live Status */}
          <div className="space-y-4">
              
              {/* Highest Bid */}
              <div className="text-center p-6 bg-green-50 rounded-lg border">
                <p className="text-sm text-gray-600 mb-2">Current Highest Bid</p>
                <p className="text-4xl font-bold text-green-600">
                  {formatCurrency(auction.highestBid)}
                </p>
                <p className="text-sm text-gray-500 mt-2">
                  {auction.bidderCount} bidders • {auction.highestBid > auction.reservePrice ? 'Above' : 'Below'} reserve
                </p>
              </div>

              {/* Quick Stats */}
              <div className="grid grid-cols-2 gap-4">
                <div className="p-4 border rounded-lg">
                  <p className="text-sm text-gray-500">Starting Price</p>
                  <p className="text-lg font-semibold">{formatCurrency(auction.startingPrice)}</p>
                </div>
                <div className="p-4 border rounded-lg">
                  <p className="text-sm text-gray-500">Reserve Price</p>
                  <p className="text-lg font-semibold">{formatCurrency(auction.reservePrice)}</p>
                </div>
              </div>

              {/* Actions */}
              {auction.status === 'active' && (
                <div className="flex space-x-3">
                  <Button onClick={handleExtendTime} className="flex-1">
                    Extend 30min
                  </Button>
                  <Button onClick={handleCloseAuction} variant="destructive" className="flex-1">
                    Close Auction
                  </Button>
                </div>
              )}
            </div>
          
          {/* Empty column for balance */}
          <div></div>
        </div>

        {/* Bottom Section */}
        <div className="mt-8 grid grid-cols-1 lg:grid-cols-2 gap-6">
          
          {/* Auction Details */}
          <div>
            <h2 className="text-lg font-semibold mb-4">Auction Details</h2>
            <div className="space-y-3">
              <div className="flex justify-between py-2 border-b">
                <span className="text-gray-600">Start Time</span>
                <span className="font-medium">{formatDateTime(auction.startTime)}</span>
              </div>
              <div className="flex justify-between py-2 border-b">
                <span className="text-gray-600">End Time</span>
                <span className="font-medium">{formatDateTime(auction.endTime)}</span>
              </div>
              <div className="flex justify-between py-2 border-b">
                <span className="text-gray-600">Bid Increment</span>
                <span className="font-medium">{formatCurrency(auction.bidIncrement)}</span>
              </div>
              <div className="flex justify-between py-2">
                <span className="text-gray-600">Total Bidders</span>
                <span className="font-medium">{auction.bidderCount}</span>
              </div>
            </div>
          </div>

          {/* Bid History */}
          <div>
            <h2 className="text-lg font-semibold mb-4">Recent Bids</h2>
            <div className="space-y-2 max-h-80 overflow-y-auto">
              {auction.bids.length > 0 ? (
                auction.bids.map((bid, index) => (
                  <div key={bid.id} className={`flex justify-between items-center p-3 rounded border ${
                    index === 0 ? 'bg-green-50 border-green-200' : 'bg-gray-50'
                  }`}>
                    <div>
                      <p className="font-semibold">{formatCurrency(bid.amount)}</p>
                      <p className="text-sm text-gray-500">{bid.bidderName}</p>
                      {bid.bidderEmail && (
                        <p className="text-xs text-gray-400">{bid.bidderEmail}</p>
                      )}
                    </div>
                    <div className="text-right">
                      <p className="text-sm text-gray-500">
                        {new Date(bid.timestamp).toLocaleTimeString('en-IN', {
                          hour: '2-digit',
                          minute: '2-digit'
                        })}
                      </p>
                      {index === 0 && (
                        <span className="text-xs bg-green-100 text-green-800 px-2 py-1 rounded">
                          Winning
                        </span>
                      )}
                    </div>
                  </div>
                ))
              ) : (
                <div className="text-center py-8 text-gray-500">
                  <Gavel className="w-8 h-8 mx-auto mb-2 text-gray-400" />
                  <p>No bids yet</p>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}