import { useState, useEffect } from 'react';
import { Card, CardContent } from '../../../ui/card';
import { Badge } from '../../../ui/badge';
import { MapPin, Home, IndianRupee, Bed, Bath, Square, Eye, Gavel } from 'lucide-react';
import { useAzureBlobMedia } from '../../../../hooks/useAzureBlobMedia';

interface Property {
  id: number;
  title: string;
  address: string;
  locality: string;
  city: string;
  state: string;
  price: number;
  property_type: string;
  listing_type: string;
  bidding_enabled: boolean;
  auction_enabled: boolean;
  bhk: string;
  bathrooms: number;
  area: number;
  views: number;
  type: string; // Alternative field name
  sqft_area: number; // Alternative field name
  built_up_area: number; // Alternative field name
  bedrooms: number; // Alternative field name
  bedroom_count: number; // Alternative field name
  no_of_bedrooms: number; // Alternative field name
}

const BiddingProperties: React.FC = () => {
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchBiddingProperties();
  }, []);

  const fetchBiddingProperties = async () => {
    try {
      const sellerId = localStorage.getItem('sellerId');
      const url = sellerId 
        ? `http://localhost:8090/api/properties/?listed_by_id=${sellerId}`
        : `http://localhost:8090/api/properties/`;
      
      const response = await fetch(url);
      if (response.ok) {
        const data = await response.json();
        const allProperties = Array.isArray(data) ? data : (data.properties || []);
        
        // Filter only bidding-enabled properties
        const biddingProperties = allProperties.filter((prop: Property) => 
          prop.listing_type === 'bidding' || 
          prop.bidding_enabled === true ||
          prop.auction_enabled === true
        );
        
        setProperties(biddingProperties);
      }
    } catch (error) {
      console.error('Error fetching properties:', error);
    } finally {
      setLoading(false);
    }
  };

const PropertyCard: React.FC<{ property: Property }> = ({ property }) => {
  const { images: azureImages, loading: mediaLoading } = useAzureBlobMedia(property.id.toString());

  const getImageUrl = () => {
    if (azureImages && azureImages.length > 0) {
      return azureImages[0].url;
    }
    return "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=250";
  };

  const formatCurrency = (amount: number) => {
    if (!amount || amount === 0) return 'Price not set';
    if (amount >= 10000000) return `₹${(amount / 10000000).toFixed(1)} Cr`;
    if (amount >= 100000) return `₹${(amount / 100000).toFixed(1)} L`;
    return `₹${amount.toLocaleString()}`;
  };

  return (
    <Card className="overflow-hidden hover:shadow-lg transition-shadow border-2 border-blue-200">
      <div className="relative">
        <img
          src={getImageUrl()}
          alt={property.title}
          className="w-full h-48 object-cover"
          onError={(e) => {
            e.currentTarget.src = "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=250";
          }}
        />
        <div className="absolute top-3 left-3">
          <Badge className="bg-green-100 text-green-800 border-green-200 text-xs">
            <Gavel className="w-3 h-3 mr-1" />
            Bidding Enabled
          </Badge>
        </div>
      </div>
      
      <CardContent className="p-4">
        <div className="space-y-3">
          <div>
            <h3 className="font-semibold text-lg text-gray-900 line-clamp-1">{property.title}</h3>
            <div className="flex items-center text-gray-600 text-sm mt-1">
              <MapPin className="w-4 h-4 mr-1" />
              <span className="line-clamp-1">{property.locality}, {property.city}</span>
            </div>
          </div>

          <div className="flex items-center justify-between text-sm text-gray-600">
            <div className="flex items-center space-x-4">
              <div className="flex items-center">
                <Bed className="w-4 h-4 mr-1" />
                <span>{property.bhk || property.bedrooms || property.bedroom_count || property.no_of_bedrooms || 'N/A'}</span>
              </div>
              <div className="flex items-center">
                <Bath className="w-4 h-4 mr-1" />
                <span>{property.bathrooms || 'N/A'}</span>
              </div>
              <div className="flex items-center">
                <Square className="w-4 h-4 mr-1" />
                <span>{(property.area || property.sqft_area || property.built_up_area) ? `${property.area || property.sqft_area || property.built_up_area} sqft` : 'N/A'}</span>
              </div>
            </div>
          </div>

          <div className="flex items-center justify-between">
            <div className="text-xl font-bold text-blue-600">
              {formatCurrency(property.price)}
            </div>
            <div className="flex items-center text-gray-500 text-sm">
              <Eye className="w-4 h-4 mr-1" />
              <span>{property.views || 0} views</span>
            </div>
          </div>

          <div className="flex items-center text-sm text-gray-600">
            <Home className="w-4 h-4 mr-1" />
            <span className="capitalize">{property.property_type || property.type}</span>
          </div>
        </div>
      </CardContent>
    </Card>
  );
};

  if (loading) {
    return (
      <div className="flex-1 p-6">
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 overflow-y-auto p-6">
      <div className="bg-white border-b border-gray-200 px-6 py-4 mb-6">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900">Bidding Properties</h1>
          <p className="text-sm text-gray-600">Properties available for auction and bidding</p>
        </div>
      </div>
      
      {properties.length > 0 ? (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
          {properties.map((property) => (
            <PropertyCard key={property.id} property={property} />
          ))}
        </div>
      ) : (
        <div className="text-center py-12">
          <Gavel className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">
            No Bidding Properties
          </h3>
          <p className="text-gray-600">
            You don't have any properties enabled for bidding yet.
          </p>
        </div>
      )}
    </div>
  );
};

export default BiddingProperties;