import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "../components/ui/dialog";
import { Input } from "../components/ui/input";
import { Label } from "../components/ui/label";
import { Textarea } from "../components/ui/textarea";
import { useToast } from "../hooks/use-toast";
import { getSellerVisits, sellerApproveVisit, sellerRejectVisit, VisitSchedule } from "../../../../api/visitSchedules";
import { getUserId } from "../../../../utils/userUtils";
import { 
  Calendar, 
  Video, 
  MapPin, 
  Clock, 
  Phone, 
  MessageCircle, 
  Check, 
  X, 
  Plus,
  CalendarClock,
  Users,
  Zap,
  CheckCircle,
  XCircle,
  DollarSign,
  AlertTriangle,
  FileText,
  RefreshCw
} from "lucide-react";

interface VisitRequest {
  id: string;
  property_id: string;
  buyer_id: string;
  buyer_name: string;
  buyer_email: string;
  buyer_phone: string;
  property_title: string;
  property_address: string;
  preferred_date: string;
  preferred_time: string;
  visit_type: 'physical' | 'virtual';
  status: 'pending' | 'approved' | 'rejected' | 'completed';
  message?: string;
  created_at: string;
}

// Mock data for auction holds

const mockHolds = [
  {
    id: "hold-1",
    auctionId: "auction-1",
    propertyId: "prop-1",
    buyerId: "buyer-1",
    winningBid: 485000,
    holdPeriod: 48,
    holdStarted: new Date("2024-12-16T18:00:00"),
    holdExpires: new Date("2024-12-18T18:00:00"),
    status: "pending_payment",
    depositRequired: 24250,
    depositPaid: false,
    contractSent: true,
    notes: "Buyer requested extended hold period"
  },
  {
    id: "hold-2",
    auctionId: "auction-2",
    propertyId: "prop-2",
    buyerId: "buyer-2",
    winningBid: 367000,
    holdPeriod: 24,
    holdStarted: new Date("2024-12-15T15:00:00"),
    holdExpires: new Date("2024-12-16T15:00:00"),
    status: "payment_confirmed",
    depositRequired: 18350,
    depositPaid: true,
    contractSent: true,
    paymentConfirmedAt: new Date("2024-12-15T20:30:00"),
    notes: ""
  }
];

// Mock data for bookings
const mockBookings = [
  {
    id: "booking-1",
    propertyId: "prop-1",
    buyerId: "buyer-1",
    type: "visit",
    status: "pending",
    scheduledDate: new Date("2024-12-17T14:00:00"),
    notes: "Interested in quick purchase"
  }
];

// Helper functions
const getPropertyById = (id: string) => ({
  title: "Modern Apartment",
  address: "123 Main St",
  location: "Downtown"
});

const getBuyerById = (id: string) => ({
  firstName: "John",
  lastName: "Doe",
  email: "john.doe@example.com"
});

export default function BookingScheduling() {
  const [activeTab, setActiveTab] = useState("property-visits");
  const [visitRequests, setVisitRequests] = useState<VisitRequest[]>([]);
  const [visits, setVisits] = useState<VisitSchedule[]>([]);
  const [loading, setLoading] = useState(true);
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [typeFilter, setTypeFilter] = useState<string>("all");
  const [rescheduleDialog, setRescheduleDialog] = useState<{ open: boolean; bookingId?: string }>({ open: false });
  const [rescheduleData, setRescheduleData] = useState({ date: "", time: "" });
  const [processingAction, setProcessingAction] = useState<{visitId: number, action: 'approve' | 'reject'} | null>(null);
  const { toast } = useToast();

  useEffect(() => {
    fetchVisits();
  }, [statusFilter]);

  const isVideoCallTime = (visit: VisitSchedule) => {
    if (!visit.preferred_date || !visit.preferred_time) return false;
    
    const now = new Date();
    const visitDate = new Date(visit.preferred_date);
    const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const scheduleDate = new Date(visitDate.getFullYear(), visitDate.getMonth(), visitDate.getDate());
    
    // Only show on the scheduled date
    if (today.getTime() !== scheduleDate.getTime()) return false;
    
    // Parse time range (e.g., "2–4 PM")
    const timeRange = visit.preferred_time;
    const currentHour = now.getHours();
    
    if (timeRange.includes('9–11 AM')) return currentHour >= 9 && currentHour < 11;
    if (timeRange.includes('11–1 PM')) return currentHour >= 11 && currentHour < 13;
    if (timeRange.includes('2–4 PM')) return currentHour >= 14 && currentHour < 16;
    if (timeRange.includes('4–6 PM')) return currentHour >= 16 && currentHour < 18;
    
    return true; // Default to show if time format is different
  };

  const fetchVisits = async () => {
    try {
      setLoading(true);
      const sellerId = getUserId();
      if (!sellerId) return;
      
      const statusMap: { [key: string]: string } = {
        'all': '',
        'pending': 'admin_approved',
        'confirmed': 'seller_approved',
        'completed': 'completed',
        'cancelled': 'seller_rejected'
      };
      
      const apiStatus = statusMap[statusFilter] || '';
      const response = await getSellerVisits(sellerId, 1, 50, apiStatus);
      setVisits(response.visits);
    } catch (error) {
      console.error('Error fetching visits:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchVisitRequests();
  }, []);

  const fetchVisitRequests = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
      
      if (!token) {
        console.warn('No authentication token found');
        setVisitRequests([]);
        return;
      }

      // First, get the user info from token to get seller_id
      const userInfoResponse = await fetch('http://localhost:8090/api/seller-profile/me', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (!userInfoResponse.ok) {
        console.error('Failed to get user info:', userInfoResponse.status);
        setVisitRequests([]);
        return;
      }

      const userInfo = await userInfoResponse.json();
      const sellerId = userInfo.user_id;

      if (!sellerId) {
        console.error('No seller ID found in user info');
        setVisitRequests([]);
        return;
      }

      // Fetch all visits for this seller
      console.log('Fetching visits for seller ID:', sellerId);
      
      const visitsResponse = await fetch(`http://localhost:8090/api/visit-schedules/seller/${sellerId}/visits?limit=100`, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (visitsResponse.ok) {
        const visitsData = await visitsResponse.json();
        const visits = visitsData.visits || [];
        
        // Transform the data to match the expected interface
        const transformedVisits = visits.map(visit => ({
          id: visit.id.toString(),
          property_id: visit.property_id.toString(),
          buyer_id: visit.user_id,
          buyer_name: visit.full_name || 'N/A',
          buyer_email: visit.email || 'N/A',
          buyer_phone: visit.mobile || 'N/A',
          property_title: visit.property_title || 'Property',
          property_address: visit.property_address || 'Address not available',
          preferred_date: visit.preferred_date,
          preferred_time: visit.preferred_time || '10:00',
          visit_type: visit.visit_mode === 'Virtual Tour' ? 'virtual' : 'physical',
          status: visit.status,
          message: visit.comments || '',
          created_at: visit.created_at
        }));
        
        console.log('Transformed visits:', transformedVisits);
        setVisitRequests(transformedVisits);
      } else {
        console.error('Failed to fetch visits:', visitsResponse.status);
        setVisitRequests([]);
      }
    } catch (error) {
      console.error('Error fetching visit requests:', error);
      setVisitRequests([]);
    } finally {
      setLoading(false);
    }
  };

  const handleApproveVisit = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'approve'});
      const sellerId = getUserId();
      if (!sellerId) return;
      
      await sellerApproveVisit(visitId, sellerId);
      toast({
        title: "Visit Approved",
        description: "The property visit has been approved successfully.",
      });
      fetchVisits();
    } catch (error) {
      console.error('Error approving visit:', error);
      toast({
        title: "Error",
        description: "Failed to approve visit",
        variant: "destructive",
      });
    } finally {
      setProcessingAction(null);
    }
  };

  const handleApprove = async (visitId: string) => {
    try {
      const token = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
      
      // Get seller ID first
      const userInfoResponse = await fetch('http://localhost:8090/api/seller-profile/me', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (!userInfoResponse.ok) {
        throw new Error('Failed to get user info');
      }

      const userInfo = await userInfoResponse.json();
      const sellerId = userInfo.user_id;

      const response = await fetch(`http://localhost:8090/api/visit-schedules/${visitId}/seller-approve?seller_id=${sellerId}`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        toast({
          title: "Visit Approved",
          description: "The property visit has been approved successfully.",
        });
        fetchVisitRequests();
      } else {
        const errorData = await response.json();
        throw new Error(errorData.detail || 'Failed to approve visit');
      }
    } catch (error) {
      console.error('Error approving visit:', error);
      toast({
        title: "Error",
        description: "Failed to approve visit",
        variant: "destructive",
      });
    } finally {
      setProcessingAction(null);
    }
  };

  const handleRejectVisit = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'reject'});
      const sellerId = getUserId();
      if (!sellerId) return;
      
      await sellerRejectVisit(visitId, sellerId);
      toast({
        title: "Visit Rejected",
        description: "The property visit has been rejected.",
        variant: "destructive",
      });
      fetchVisits();
    } catch (error) {
      console.error('Error rejecting visit:', error);
      toast({
        title: "Error",
        description: "Failed to reject visit",
        variant: "destructive",
      });
    } finally {
      setProcessingAction(null);
    }
  };

  const handleReject = async (visitId: string) => {
    try {
      const token = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
      
      // Get seller ID first
      const userInfoResponse = await fetch('http://localhost:8090/api/seller-profile/me', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (!userInfoResponse.ok) {
        throw new Error('Failed to get user info');
      }

      const userInfo = await userInfoResponse.json();
      const sellerId = userInfo.user_id;

      const response = await fetch(`http://localhost:8090/api/visit-schedules/${visitId}/seller-reject?seller_id=${sellerId}`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        toast({
          title: "Visit Rejected",
          description: "The property visit has been rejected.",
        });
        fetchVisitRequests();
      } else {
        const errorData = await response.json();
        throw new Error(errorData.detail || 'Failed to reject visit');
      }
    } catch (error) {
      console.error('Error rejecting visit:', error);
      toast({
        title: "Error",
        description: "Failed to reject visit",
        variant: "destructive",
      });
    } finally {
      setProcessingAction(null);
    }
  };

  // Property Visit Requests Component
  const PropertyVisitRequests = () => {
    const getStatusColor = (status: string) => {
      switch (status) {
        case "admin_approved":
        case "pending":
          return "bg-orange-100 text-orange-800";
        case "seller_approved":
        case "confirmed":
          return "bg-green-100 text-green-800";
        case "completed":
          return "bg-blue-100 text-blue-800";
        case "seller_rejected":
        case "admin_rejected":
        case "cancelled":
        case "approved":
          return "bg-green-100 text-green-800";
        case "rejected":
          return "bg-red-100 text-red-800";
        default:
          return "bg-gray-100 text-gray-800";
      }
    };

    const getTypeIcon = (type: string) => {
      return type === "Virtual Tour" ? <Video className="w-4 h-4" /> : <MapPin className="w-4 h-4" />;
    };

    const getTypeColor = (type: string) => {
      return type === "Virtual Tour" 
        ? "bg-blue-100 text-blue-800" 
        : "bg-green-100 text-green-800";
    };

    const formatDateTime = (date: string, time: string) => {
      const dateObj = new Date(`${date}T${time}`);
      return dateObj.toLocaleDateString('en-US', {
        month: 'short',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
      });
    };

    const filteredBookings = visits.filter(visit => {
      const typeMatch = typeFilter === "all" || 
        (typeFilter === "virtual" && visit.visit_mode === "Virtual Tour") ||
        (typeFilter === "in-person" && visit.visit_mode === "In-person Visit");
      return typeMatch;
    });

    const todayBookings = visits.filter(v => {
      const today = new Date().toDateString();
      return new Date(v.preferred_date).toDateString() === today;
    }).length;
    
    const pendingBookings = visits.filter(v => v.status === "admin_approved").length;
    const completedVisits = visits.filter(v => v.status === "completed").length;

    const handleApprove = (visitId: number) => {
      handleApproveVisit(visitId);
    };

    const handleReject = (visitId: number) => {
      handleRejectVisit(visitId);
    };

    const handleReschedule = (bookingId: string) => {
      setRescheduleDialog({ open: true, bookingId });
      setRescheduleData({ date: "", time: "" });
    };

    const submitReschedule = () => {
      toast({
        title: "Booking Rescheduled",
        description: `Booking has been rescheduled to ${rescheduleData.date} at ${rescheduleData.time}`,
      });
      setRescheduleDialog({ open: false });
      setRescheduleData({ date: "", time: "" });
    };
    const filteredRequests = visitRequests.filter(request => {
      const statusMatch = statusFilter === "all" || request.status === statusFilter;
      const typeMatch = typeFilter === "all" || request.visit_type === typeFilter;
      return statusMatch && typeMatch;
    });

    const todayRequests = visitRequests.filter(r => {
      const today = new Date().toDateString();
      return new Date(r.preferred_date).toDateString() === today;
    }).length;
    
    const pendingRequests = visitRequests.filter(r => r.status === "pending").length;
    const completedRequests = visitRequests.filter(r => r.status === "completed").length;

    if (loading) {
      return (
        <div className="flex items-center justify-center py-8">
          <RefreshCw className="w-8 h-8 animate-spin text-gray-400" />
          <span className="ml-2 text-gray-600">Loading visit requests...</span>
        </div>
      );
    }

    return (
      <div className="space-y-6">
        {/* Stats Cards */}
        <div className="grid grid-cols-2 md:grid-cols-4 gap-3 md:gap-6">
          <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-blue-700 font-medium">Today's Visits</p>
                  <p className="text-3xl font-bold text-blue-800">{todayRequests}</p>
                </div>
                <Calendar className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>
          
          <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-orange-700 font-medium">Pending Approval</p>
                  <p className="text-3xl font-bold text-orange-800">{pendingRequests}</p>
                </div>
                <Clock className="w-8 h-8 text-orange-600" />
              </div>
            </CardContent>
          </Card>
          
          <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-green-700 font-medium">Completed</p>
                  <p className="text-3xl font-bold text-green-800">{completedVisits}</p>
                </div>
                <Check className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>
          
          <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-purple-700 font-medium">Video Calls</p>
                  <p className="text-3xl font-bold text-purple-800">
                    {visits.filter(v => v.video_call_enabled).length}
                    {visitRequests.filter(r => r.visit_type === "virtual").length}
                  </p>
                </div>
                <Video className="w-8 h-8 text-purple-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Filters */}
        <div className="flex items-center gap-4">
          <Select value={statusFilter} onValueChange={setStatusFilter}>
            <SelectTrigger className="w-40">
              <SelectValue placeholder="All Status" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Status</SelectItem>
              <SelectItem value="pending">Pending</SelectItem>
              <SelectItem value="approved">Approved</SelectItem>
              <SelectItem value="completed">Completed</SelectItem>
              <SelectItem value="rejected">Rejected</SelectItem>
            </SelectContent>
          </Select>
          
          <Select value={typeFilter} onValueChange={setTypeFilter}>
            <SelectTrigger className="w-40">
              <SelectValue placeholder="All Types" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Types</SelectItem>
              <SelectItem value="physical">Physical</SelectItem>
              <SelectItem value="virtual">Virtual</SelectItem>
            </SelectContent>
          </Select>
          
          <Button onClick={fetchVisitRequests} variant="outline" size="sm">
            <RefreshCw className="w-4 h-4 mr-2" />
            Refresh
          </Button>
        </div>

        {/* Visit Requests Table */}
        <Card>
          <CardContent className="p-0">
            <div className="overflow-x-auto">
              <table className="w-full min-w-[800px]">
                <thead className="bg-muted/50 border-b">
                  <tr>
                    <th className="px-2 md:px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase">Visitor</th>
                    <th className="px-2 md:px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase">Property</th>
                    <th className="px-2 md:px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase">Type</th>
                    <th className="px-2 md:px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase">Scheduled</th>
                    <th className="px-2 md:px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase">Status</th>
                    <th className="px-2 md:px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase">Actions</th>
                  </tr>
                </thead>
                <tbody className="divide-y divide-border">
                  {loading ? (
                    <tr>
                      <td colSpan={6} className="px-4 py-8 text-center">
                        <div className="flex justify-center">
                          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
                        </div>
                      </td>
                    </tr>
                  ) : filteredBookings.length === 0 ? (
                    <tr>
                      <td colSpan={6} className="px-4 py-8 text-center text-muted-foreground">
                        No visits found
                      </td>
                    </tr>
                  ) : (
                    filteredBookings.map((visit) => (
                      <tr key={visit.id} className="hover:bg-muted/50">
                        <td className="px-2 md:px-4 py-4">
                          <div className="flex items-center gap-2 md:gap-3">
                            <div className="w-8 h-8 bg-primary rounded-full flex items-center justify-center">
                              <span className="text-primary-foreground text-xs font-medium">
                                {visit.full_name.split(' ').map(n => n[0]).join('').slice(0, 2)}
                              </span>
                            </div>
                            <div className="min-w-0">
                              <p className="text-sm font-medium truncate">{visit.full_name}</p>
                              <p className="text-xs text-muted-foreground truncate">{visit.email}</p>
                            </div>
                          </div>
                        </td>
                        <td className="px-2 md:px-4 py-4">
                          <div className="flex items-center gap-2 md:gap-3">
                            <div className="w-8 h-8 md:w-10 md:h-10 bg-muted rounded flex items-center justify-center">
                              <MapPin className="w-4 h-4 text-muted-foreground" />
                            </div>
                            <div className="min-w-0">
                              <p className="text-sm font-medium truncate">{visit.property_title}</p>
                              <p className="text-xs text-muted-foreground truncate">{visit.property_city}</p>
                            </div>
                          </div>
                        </td>
                        <td className="px-2 md:px-4 py-4">
                          <Badge className={getTypeColor(visit.visit_mode)}>
                            {getTypeIcon(visit.visit_mode)}
                            <span className="ml-1 capitalize text-xs">{visit.visit_mode === "Virtual Tour" ? "virtual" : "visit"}</span>
                          </Badge>
                          {visit.video_call_enabled && (
                            <Badge className="bg-blue-100 text-blue-800 ml-1">
                              <Video className="w-3 h-3 mr-1" />
                              Video
                            </Badge>
                          )}
                        </td>
                        <td className="px-4 py-4">
                          <div>
                            <p className="text-sm font-medium">{new Date(visit.preferred_date).toLocaleDateString('en-US', { month: 'short', day: 'numeric' })}, {visit.preferred_time || 'Time TBD'}</p>
                            <p className="text-xs text-muted-foreground">Duration: 1 hour</p>
                          </div>
                        </td>
                        <td className="px-2 md:px-4 py-4">
                          <Badge className={getStatusColor(visit.status === "admin_approved" ? "pending" : visit.status)}>
                            {visit.status === "admin_approved" ? "Pending" : 
                             visit.status === "seller_approved" ? "Confirmed" :
                             visit.status.charAt(0).toUpperCase() + visit.status.slice(1)}
                          </Badge>
                        </td>
                        <td className="px-2 md:px-4 py-4">
                          <div className="flex items-center gap-1">
                            {visit.status === "admin_approved" && (
                              <>
                                <Button 
                                  size="sm" 
                                  onClick={() => handleApprove(visit.id)} 
                                  className="h-8 w-8 p-0"
                                  disabled={processingAction?.visitId === visit.id && processingAction?.action === 'approve'}
                                >
                                  <Check className="h-3 w-3 md:h-4 md:w-4" />
                                </Button>
                                <Button 
                                  size="sm" 
                                  variant="destructive" 
                                  onClick={() => handleReject(visit.id)} 
                                  className="h-8 w-8 p-0"
                                  disabled={processingAction?.visitId === visit.id && processingAction?.action === 'reject'}
                                >
                                  <X className="h-3 w-3 md:h-4 md:w-4" />
                                </Button>
                              </>
                            )}
                            {visit.status === "seller_approved" && (
                              <>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  className="h-8 w-8 p-0"
                                >
                                  <Phone className="h-3 w-3 md:h-4 md:w-4" />
                                </Button>
                                {(visit.video_call_enabled || visit.visit_mode === "Virtual Tour") && visit.status === "seller_approved" && isVideoCallTime(visit) && (
                                  <Button
                                    variant="outline"
                                    size="sm"
                                    className="h-8 w-8 p-0 bg-blue-50 hover:bg-blue-100"
                                    onClick={() => {
                                      const roomId = visit.video_call_room_id || `NAL_Property_${visit.property_id}_Visit_${visit.id}`;
                                      window.open(`https://meet.jit.si/${roomId}`, '_blank');
                                    }}
                                  >
                                    <Video className="h-3 w-3 md:h-4 md:w-4 text-blue-600" />
                                  </Button>
                                )}
                              </>
                            )}
                          </div>
                        </td>
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>
          </CardContent>
        </Card>
      </div>
    );
  };

  // Auction Hold & Confirmation Component
  const AuctionHoldConfirmation = () => {
    const getStatusColor = (status: string) => {
      switch (status) {
        case "pending_payment":
          return "bg-yellow-100 text-yellow-800";
        case "payment_confirmed":
          return "bg-green-100 text-green-800";
        case "expired":
          return "bg-red-100 text-red-800";
        case "cancelled":
          return "bg-gray-100 text-gray-800";
        default:
          return "bg-gray-100 text-gray-800";
      }
    };

    const getStatusIcon = (status: string) => {
      switch (status) {
        case "pending_payment":
          return <Clock className="w-4 h-4" />;
        case "payment_confirmed":
          return <CheckCircle className="w-4 h-4" />;
        case "expired":
          return <XCircle className="w-4 h-4" />;
        case "cancelled":
          return <XCircle className="w-4 h-4" />;
        default:
          return <Clock className="w-4 h-4" />;
      }
    };

    const formatTimeRemaining = (expireDate: Date) => {
      const now = new Date();
      const diff = expireDate.getTime() - now.getTime();
      
      if (diff <= 0) return "Expired";
      
      const hours = Math.floor(diff / (1000 * 60 * 60));
      const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
      
      if (hours > 24) {
        const days = Math.floor(hours / 24);
        return `${days}d ${hours % 24}h`;
      }
      if (hours > 0) return `${hours}h ${minutes}m`;
      return `${minutes}m`;
    };

    const handleConfirmPayment = (holdId: string) => {
      toast({
        title: "Payment Confirmed",
        description: "The auction hold payment has been confirmed.",
      });
    };

    return (
      <div className="space-y-6">
        {/* Stats Cards */}
        <div className="grid grid-cols-2 md:grid-cols-4 gap-3 md:gap-6">
          <Card className="bg-gradient-to-br from-yellow-50 to-yellow-100 border-yellow-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-yellow-700 font-medium">Pending Confirmations</p>
                  <p className="text-3xl font-bold text-yellow-800">
                    {mockHolds.filter(h => h.status === "pending_payment").length}
                  </p>
                </div>
                <Clock className="w-8 h-8 text-yellow-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-green-700 font-medium">Confirmed Sales</p>
                  <p className="text-3xl font-bold text-green-800">
                    {mockHolds.filter(h => h.status === "payment_confirmed").length}
                  </p>
                </div>
                <CheckCircle className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-red-50 to-red-100 border-red-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-red-700 font-medium">Expired Holds</p>
                  <p className="text-3xl font-bold text-red-800">
                    {mockHolds.filter(h => h.status === "expired").length}
                  </p>
                </div>
                <XCircle className="w-8 h-8 text-red-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-blue-700 font-medium">Total Hold Value</p>
                  <p className="text-3xl font-bold text-blue-800">
                    ${(mockHolds.reduce((sum, hold) => sum + hold.winningBid, 0) / 1000000).toFixed(1)}M
                  </p>
                </div>
                <DollarSign className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Holds List */}
        <div className="space-y-4">
          {mockHolds.map((hold) => {
            const property = getPropertyById(hold.propertyId);
            const buyer = getBuyerById(hold.buyerId);
            const isExpiring = hold.status === "pending_payment" && 
              new Date(hold.holdExpires).getTime() - new Date().getTime() < 6 * 60 * 60 * 1000;
            
            return (
              <Card key={hold.id} className={`${isExpiring ? 'border-orange-300 bg-orange-50' : ''}`}>
                <CardContent className="p-6">
                  <div className="flex justify-between items-start mb-4">
                    <div className="flex-1">
                      <div className="flex items-center gap-3 mb-2">
                        <h3 className="text-lg font-semibold">{property?.title}</h3>
                        <Badge className={getStatusColor(hold.status)} variant="outline">
                          {getStatusIcon(hold.status)}
                          <span className="ml-1">{hold.status.replace('_', ' ')}</span>
                        </Badge>
                        {isExpiring && (
                          <Badge variant="outline" className="bg-orange-100 text-orange-800">
                            <AlertTriangle className="w-3 h-3 mr-1" />
                            Expiring Soon
                          </Badge>
                        )}
                      </div>
                      <p className="text-sm text-muted-foreground">{property?.address}</p>
                    </div>
                    <div className="text-right">
                      <p className="text-2xl font-bold">${hold.winningBid.toLocaleString()}</p>
                      <p className="text-sm text-muted-foreground">Winning Bid</p>
                    </div>
                  </div>

                  <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-4">
                    <div>
                      <h4 className="font-medium mb-2">Buyer Information</h4>
                      <div className="space-y-1">
                        <p className="text-sm">
                          <span className="font-medium">{buyer?.firstName} {buyer?.lastName}</span>
                        </p>
                        <p className="text-sm text-muted-foreground">{buyer?.email}</p>
                      </div>
                    </div>

                    <div>
                      <h4 className="font-medium mb-2">Hold Details</h4>
                      <div className="space-y-1">
                        <p className="text-sm">
                          <span className="text-muted-foreground">Expires:</span> {hold.holdExpires.toLocaleDateString()}
                        </p>
                        <p className="text-sm">
                          <span className="text-muted-foreground">Remaining:</span> {formatTimeRemaining(hold.holdExpires)}
                        </p>
                      </div>
                    </div>

                    <div>
                      <h4 className="font-medium mb-2">Payment Status</h4>
                      <div className="space-y-1">
                        <p className="text-sm">
                          <span className="text-muted-foreground">Deposit Required:</span> ${hold.depositRequired.toLocaleString()}
                        </p>
                        <p className="text-sm">
                          <span className="text-muted-foreground">Deposit Paid:</span> 
                          <Badge variant={hold.depositPaid ? "default" : "secondary"} className="ml-2">
                            {hold.depositPaid ? "Yes" : "No"}
                          </Badge>
                        </p>
                      </div>
                    </div>
                  </div>

                  <div className="flex gap-2 flex-wrap">
                    {hold.status === "pending_payment" && (
                      <Button size="sm" onClick={() => handleConfirmPayment(hold.id)} className="bg-blue-500 hover:bg-blue-600 text-white">
                        <CheckCircle className="w-4 h-4 mr-1" />
                        Confirm Payment
                      </Button>
                    )}
                    <Button variant="outline" size="sm">
                      <MessageCircle className="w-4 h-4 mr-1" />
                      Contact Buyer
                    </Button>
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </div>
      </div>
    );
  };

  // Urgent / Flash Sale Visits Component
  const UrgentFlashSaleVisits = () => {
    const urgentVisits = mockBookings.filter(b => 
      b.type === "visit" && (b.status === "pending" || b.status === "confirmed")
    );

    const getStatusColor = (status: string) => {
      switch (status) {
        case "pending":
          return "bg-yellow-100 text-yellow-800";
        case "confirmed":
          return "bg-green-100 text-green-800";
        case "cancelled":
          return "bg-red-100 text-red-800";
        default:
          return "bg-gray-100 text-gray-800";
      }
    };

    const getStatusIcon = (status: string) => {
      switch (status) {
        case "pending":
          return <Clock className="w-4 h-4" />;
        case "confirmed":
          return <CheckCircle className="w-4 h-4" />;
        case "cancelled":
          return <XCircle className="w-4 h-4" />;
        default:
          return <Clock className="w-4 h-4" />;
      }
    };

    const getTimeRemaining = (date: Date) => {
      const now = new Date();
      const diff = date.getTime() - now.getTime();
      
      if (diff <= 0) return "Expired";
      
      const hours = Math.floor(diff / (1000 * 60 * 60));
      const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
      
      if (hours > 0) return `${hours}h ${minutes}m`;
      return `${minutes}m`;
    };

    return (
      <div className="space-y-6">
        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-3 md:gap-6">
          <Card className="bg-gradient-to-br from-red-50 to-red-100 border-red-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-red-700 font-medium">Flash Sale Visits</p>
                  <p className="text-3xl font-bold text-red-800">4</p>
                </div>
                <Zap className="w-8 h-8 text-red-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-orange-700 font-medium">Urgent Bookings</p>
                  <p className="text-3xl font-bold text-orange-800">6</p>
                </div>
                <Calendar className="w-8 h-8 text-orange-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-blue-700 font-medium">Total Visitors</p>
                  <p className="text-3xl font-bold text-blue-800">23</p>
                </div>
                <Users className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Urgent Visits List */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Zap className="w-5 h-5 text-red-500" />
              Recent Urgent Visits
            </CardTitle>
          </CardHeader>
          <CardContent>
            {urgentVisits.length === 0 ? (
              <div className="text-center py-8">
                <Zap className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                <h3 className="text-lg font-medium text-muted-foreground mb-2">No urgent visits</h3>
                <p className="text-sm text-muted-foreground">No urgent visit requests found.</p>
              </div>
            ) : (
              <div className="space-y-4">
                {urgentVisits.map((visit) => {
                  const property = getPropertyById(visit.propertyId);
                  const buyer = getBuyerById(visit.buyerId);
                  const timeRemaining = getTimeRemaining(visit.scheduledDate);
                  const isUrgent = new Date(visit.scheduledDate).getTime() - new Date().getTime() < 24 * 60 * 60 * 1000;
                  
                  return (
                    <Card key={visit.id} className={`border ${isUrgent ? 'border-red-300 bg-red-50' : ''}`}>
                      <CardContent className="p-4">
                        <div className="flex justify-between items-start mb-3">
                          <div className="flex-1">
                            <div className="flex items-center gap-3 mb-2">
                              <h3 className="text-lg font-semibold">{property?.title || "Unknown Property"}</h3>
                              <Badge className={getStatusColor(visit.status)} variant="outline">
                                {getStatusIcon(visit.status)}
                                <span className="ml-1">{visit.status.charAt(0).toUpperCase() + visit.status.slice(1)}</span>
                              </Badge>
                              {isUrgent && (
                                <Badge variant="outline" className="bg-red-100 text-red-800">
                                  <Zap className="w-3 h-3 mr-1" />
                                  Urgent
                                </Badge>
                              )}
                            </div>
                            <p className="text-sm text-muted-foreground mb-1">
                              <MapPin className="inline w-4 h-4 mr-1" />
                              {property?.location || "Unknown Location"}
                            </p>
                          </div>
                          <div className="text-right">
                            <div className="bg-red-100 text-red-800 px-3 py-1 rounded-full text-sm font-medium">
                              <Clock className="inline w-4 h-4 mr-1" />
                              {timeRemaining}
                            </div>
                          </div>
                        </div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-3">
                          <div>
                            <p className="text-sm">
                              <strong>Buyer:</strong> {buyer?.firstName} {buyer?.lastName}
                            </p>
                            <p className="text-sm text-muted-foreground">{buyer?.email}</p>
                          </div>
                          <div>
                            <p className="text-sm">
                              <strong>Scheduled:</strong> {visit.scheduledDate.toLocaleString()}
                            </p>
                          </div>
                        </div>
                        
                        {visit.notes && (
                          <div className="mb-3 p-3 bg-muted/30 rounded-lg">
                            <p className="text-sm"><strong>Notes:</strong> {visit.notes}</p>
                          </div>
                        )}
                        
                        <div className="flex gap-2">
                          <Button size="sm" variant="outline">
                            <Phone className="w-4 h-4 mr-1" />
                            Contact Buyer
                          </Button>
                          {visit.status === "pending" && (
                            <>
                              <Button size="sm" className="bg-blue-500 hover:bg-blue-600 text-white">
                                <Check className="w-4 h-4 mr-1" />
                                Approve
                              </Button>
                              <Button size="sm" className="bg-blue-500 hover:bg-blue-600 text-white">
                                <X className="w-4 h-4 mr-1" />
                                Reject
                              </Button>
                            </>
                          )}
                        </div>
                      </CardContent>
                    </Card>
                  );
                })}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    );
  };

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Booking & Scheduling</h1>
          <p className="text-muted-foreground">Manage property visits, auctions, and urgent appointments</p>
        </div>
        <Button className="bg-blue-500 hover:bg-blue-600 text-white">
          <Plus className="mr-2 h-4 w-4" />
          Schedule New
        </Button>
      </div>
      
      
      <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
  <TabsList className="grid w-full grid-cols-1 md:grid-cols-3 h-auto p-1 bg-gray-100 rounded-lg gap-2">
    <TabsTrigger
      value="property-visits"
      className={`flex flex-col items-center gap-2 p-4 md:p-5 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
        activeTab === "property-visits" 
          ? "bg-blue-500 text-white border-blue-500" 
          : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
      }`}
    >
      <Calendar className="w-6 h-6 md:w-7 md:h-7" />
      <span className="font-semibold text-sm md:text-base text-center">Property Visit Requests</span>
    </TabsTrigger>

    <TabsTrigger
      value="auction-hold"
      className={`flex flex-col items-center gap-2 p-4 md:p-5 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
        activeTab === "auction-hold" 
          ? "bg-blue-500 text-white border-blue-500" 
          : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
      }`}
    >
      <CheckCircle className="w-6 h-6 md:w-7 md:h-7" />
      <span className="font-semibold text-sm md:text-base text-center">Auction Hold & Confirmation</span>
    </TabsTrigger>

    <TabsTrigger
      value="urgent-visits"
      className={`flex flex-col items-center gap-2 p-4 md:p-5 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
        activeTab === "urgent-visits" 
          ? "bg-blue-500 text-white border-blue-500" 
          : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
      }`}
    >
      <Zap className="w-6 h-6 md:w-7 md:h-7" />
      <span className="font-semibold text-sm md:text-base text-center">Urgent / Flash Sale Visits</span>
    </TabsTrigger>
  </TabsList>

  <TabsContent value="property-visits" className="mt-6">
    <PropertyVisitRequests />
  </TabsContent>

  <TabsContent value="auction-hold" className="mt-6">
    <AuctionHoldConfirmation />
  </TabsContent>

  <TabsContent value="urgent-visits" className="mt-6">
    <UrgentFlashSaleVisits />
  </TabsContent>
</Tabs>

    </div>
  );
}