import { useState, useEffect } from 'react';
import { useRoute } from 'wouter';
import { Card, CardContent, CardHeader, CardTitle } from '../../../ui/card';
import { Button } from '../../../ui/button';
import { Input } from '../../../ui/input';
import { Label } from '../../../ui/label';
import { Textarea } from '../../../ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../../../ui/select';
import { ArrowLeft, Calendar, IndianRupee, Clock, Users } from 'lucide-react';
import { Link } from 'wouter';

interface Property {
  id: number;
  title: string;
  price: string;
  city: string;
  locality?: string;
  images: string[];
}

export default function EnableBidding() {
  const [match, params] = useRoute('/seller/dashboard/enable-bidding/:propertyId');
  const [property, setProperty] = useState<Property | null>(null);
  const [loading, setLoading] = useState(true);
  const [submitting, setSubmitting] = useState(false);
  
  const [biddingForm, setBiddingForm] = useState({
    startTime: '',
    endTime: '',
    startingPrice: '',
    reservePrice: '',
    bidIncrement: '10000',
    description: '',
    autoExtend: true,
    extensionMinutes: '10'
  });

  useEffect(() => {
    if (params?.propertyId) {
      fetchProperty();
    }
  }, [params?.propertyId]);

  const fetchProperty = async () => {
    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch(`http://localhost:8090/api/properties/${params?.propertyId}`, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      if (response.ok) {
        const data = await response.json();
        setProperty({
          id: data.id,
          title: data.title || `Property ${data.id}`,
          price: data.sale_price?.toString() || '0',
          city: data.city || 'Unknown',
          locality: data.locality || '',
          images: data.images || []
        });
        // Pre-fill starting price with property price
        setBiddingForm(prev => ({
          ...prev,
          startingPrice: data.sale_price?.toString() || '',
          reservePrice: data.sale_price?.toString() || ''
        }));
      }
    } catch (error) {
      console.error('Error fetching property:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setSubmitting(true);

    try {
      const token = localStorage.getItem('access_token');
      const response = await fetch('http://localhost:8090/api/auction-management/create-session', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({
          property_id: parseInt(params?.propertyId || '0'),
          starting_bid_price: parseFloat(biddingForm.startingPrice),
          reserve_price: parseFloat(biddingForm.reservePrice),
          minimum_increment: parseFloat(biddingForm.bidIncrement),
          registration_fee: 25000, // Default registration fee
          start_time: new Date(biddingForm.startTime).toISOString(),
          end_time: new Date(biddingForm.endTime).toISOString(),
          auto_extension_enabled: biddingForm.autoExtend,
          extension_time_minutes: parseInt(biddingForm.extensionMinutes)
        })
      });

      if (response.ok) {
        // Redirect to auction dashboard
        window.location.href = '/seller/dashboard/auction-bidding';
      } else {
        const errorData = await response.json();
        console.error('Failed to create auction:', errorData);
        
        // Handle specific error for existing auction
        if (errorData.detail && errorData.detail.includes('already has an')) {
          alert('This property already has an active auction. Please wait for it to complete or cancel it before creating a new one.');
        } else {
          alert('Failed to create auction: ' + (errorData.detail || 'Unknown error'));
        }
      }
    } catch (error) {
      console.error('Error creating auction:', error);
      alert('Error creating auction: ' + error);
    } finally {
      setSubmitting(false);
    }
  };

  const formatCurrency = (amount: string) => {
    const num = parseFloat(amount) || 0;
    if (num >= 10000000) return `₹${(num / 10000000).toFixed(1)} Cr`;
    if (num >= 100000) return `₹${(num / 100000).toFixed(1)} L`;
    return `₹${num.toLocaleString()}`;
  };

  if (loading) {
    return (
      <div className="flex-1 p-6">
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      </div>
    );
  }

  if (!property) {
    return (
      <div className="flex-1 p-6">
        <div className="text-center py-12">
          <h3 className="text-lg font-medium text-gray-900 mb-2">Property not found</h3>
          <p className="text-gray-600">The property could not be loaded</p>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 p-6 space-y-6">
      {/* Header */}
      <div className="flex items-center gap-4">
        <Link href="/seller/dashboard/properties">
          <Button variant="outline" size="sm">
            <ArrowLeft className="w-4 h-4 mr-2" />
            Back to Properties
          </Button>
        </Link>
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Enable Bidding</h1>
          <p className="text-gray-600 mt-1">Set up auction for {property.title}</p>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Left Column - Property Info */}
        <div className="lg:col-span-1">
          <Card>
            <CardHeader>
              <CardTitle>Property Details</CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              {property.images.length > 0 && (
                <img 
                  src={property.images[0]} 
                  alt={property.title}
                  className="w-full h-32 object-cover rounded-lg"
                />
              )}
              <div>
                <h3 className="font-semibold text-gray-900">{property.title}</h3>
                <p className="text-sm text-gray-600">{property.locality}, {property.city}</p>
                <p className="text-lg font-bold text-green-600 mt-2">
                  {formatCurrency(property.price)}
                </p>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Right Column - Bidding Configuration */}
        <div className="lg:col-span-2">
          <Card>
            <CardHeader>
              <CardTitle>Auction Configuration</CardTitle>
            </CardHeader>
            <CardContent>
              <form onSubmit={handleSubmit} className="space-y-6">
                {/* Timing */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="startTime">Auction Start Time</Label>
                    <Input
                      id="startTime"
                      type="datetime-local"
                      value={biddingForm.startTime}
                      onChange={(e) => setBiddingForm({...biddingForm, startTime: e.target.value})}
                      required
                      min={new Date().toISOString().slice(0, 16)}
                    />
                  </div>
                  <div>
                    <Label htmlFor="endTime">Auction End Time</Label>
                    <Input
                      id="endTime"
                      type="datetime-local"
                      value={biddingForm.endTime}
                      onChange={(e) => setBiddingForm({...biddingForm, endTime: e.target.value})}
                      required
                      min={biddingForm.startTime || new Date().toISOString().slice(0, 16)}
                    />
                  </div>
                </div>

                {/* Pricing */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="startingPrice">Starting Price (₹)</Label>
                    <Input
                      id="startingPrice"
                      type="number"
                      value={biddingForm.startingPrice}
                      onChange={(e) => setBiddingForm({...biddingForm, startingPrice: e.target.value})}
                      required
                      min="1"
                    />
                    <p className="text-xs text-gray-500 mt-1">
                      Current: {formatCurrency(biddingForm.startingPrice)}
                    </p>
                  </div>
                  <div>
                    <Label htmlFor="reservePrice">Reserve Price (₹)</Label>
                    <Input
                      id="reservePrice"
                      type="number"
                      value={biddingForm.reservePrice}
                      onChange={(e) => setBiddingForm({...biddingForm, reservePrice: e.target.value})}
                      required
                      min={biddingForm.startingPrice || "1"}
                    />
                    <p className="text-xs text-gray-500 mt-1">
                      Current: {formatCurrency(biddingForm.reservePrice)}
                    </p>
                  </div>
                </div>

                {/* Bid Increment */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label htmlFor="bidIncrement">Minimum Bid Increment (₹)</Label>
                    <Select 
                      value={biddingForm.bidIncrement} 
                      onValueChange={(value) => setBiddingForm({...biddingForm, bidIncrement: value})}
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="5000">₹5,000</SelectItem>
                        <SelectItem value="10000">₹10,000</SelectItem>
                        <SelectItem value="25000">₹25,000</SelectItem>
                        <SelectItem value="50000">₹50,000</SelectItem>
                        <SelectItem value="100000">₹1,00,000</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div>
                    <Label htmlFor="extensionMinutes">Auto-extend Time (minutes)</Label>
                    <Select 
                      value={biddingForm.extensionMinutes} 
                      onValueChange={(value) => setBiddingForm({...biddingForm, extensionMinutes: value})}
                    >
                      <SelectTrigger>
                        <SelectValue />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="5">5 minutes</SelectItem>
                        <SelectItem value="10">10 minutes</SelectItem>
                        <SelectItem value="15">15 minutes</SelectItem>
                        <SelectItem value="30">30 minutes</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                </div>

                {/* Description */}
                <div>
                  <Label htmlFor="description">Auction Description (Optional)</Label>
                  <Textarea
                    id="description"
                    value={biddingForm.description}
                    onChange={(e) => setBiddingForm({...biddingForm, description: e.target.value})}
                    placeholder="Add any special terms or conditions for this auction..."
                    rows={3}
                  />
                </div>

                {/* Summary */}
                <div className="bg-blue-50 p-4 rounded-lg border border-blue-200">
                  <h4 className="font-medium text-blue-900 mb-2">Auction Summary</h4>
                  <div className="grid grid-cols-2 gap-2 text-sm">
                    <div className="flex items-center">
                      <Calendar className="w-4 h-4 mr-2 text-blue-600" />
                      <span>Duration: {biddingForm.startTime && biddingForm.endTime ? 
                        Math.round((new Date(biddingForm.endTime).getTime() - new Date(biddingForm.startTime).getTime()) / (1000 * 60 * 60)) + ' hours' : 
                        'Not set'}</span>
                    </div>
                    <div className="flex items-center">
                      <IndianRupee className="w-4 h-4 mr-2 text-blue-600" />
                      <span>Price Range: {formatCurrency(biddingForm.startingPrice)} - {formatCurrency(biddingForm.reservePrice)}</span>
                    </div>
                  </div>
                </div>

                {/* Submit */}
                <div className="flex gap-3">
                  <Link href="/seller/dashboard/properties">
                    <Button type="button" variant="outline">Cancel</Button>
                  </Link>
                  <Button 
                    type="submit" 
                    disabled={submitting}
                    className="bg-green-600 hover:bg-green-700"
                  >
                    {submitting ? 'Enabling...' : 'Enable Bidding'}
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}