import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Badge } from "../components/ui/badge";
import { Button } from "../components/ui/button";
import { Input } from "../components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../components/ui/select";
import { Phone, Mail, MessageCircle, Star, Target, Users, TrendingUp, Filter, Search } from "lucide-react";
import { useState } from "react";

export default function LeadPipeline() {
  const [filterStage, setFilterStage] = useState("all");
  const [searchTerm, setSearchTerm] = useState("");

  const pipelineStages = [
    { name: "New Leads", count: 24, value: 1200000, color: "bg-blue-500" },
    { name: "Qualified", count: 18, value: 950000, color: "bg-green-500" },
    { name: "Proposal Sent", count: 12, value: 720000, color: "bg-orange-500" },
    { name: "Negotiation", count: 8, value: 480000, color: "bg-purple-500" },
    { name: "Closing", count: 4, value: 280000, color: "bg-red-500" }
  ];

  const leads = [
    {
      id: 1,
      name: "Sarah Johnson",
      email: "sarah@email.com",
      phone: "+1 555-0123",
      property: "Downtown Penthouse",
      stage: "negotiation",
      value: 850000,
      score: 92,
      lastContact: "2 hours ago",
      source: "Website",
      probability: 85
    },
    {
      id: 2,
      name: "Michael Chen",
      email: "michael@email.com",
      phone: "+1 555-0124",
      property: "Suburban Villa",
      stage: "proposal",
      value: 650000,
      score: 78,
      lastContact: "1 day ago",
      source: "Referral",
      probability: 65
    },
    {
      id: 3,
      name: "Emily Rodriguez",
      email: "emily@email.com",
      phone: "+1 555-0125",
      property: "Beachfront Condo",
      stage: "qualified",
      value: 420000,
      score: 65,
      lastContact: "3 days ago",
      source: "Social Media",
      probability: 45
    }
  ];

  const getStageColor = (stage: string) => {
    const colors = {
      "new": "bg-blue-100 text-blue-800",
      "qualified": "bg-green-100 text-green-800",
      "proposal": "bg-orange-100 text-orange-800",
      "negotiation": "bg-purple-100 text-purple-800",
      "closing": "bg-red-100 text-red-800"
    };
    return colors[stage as keyof typeof colors] || "bg-gray-100 text-gray-800";
  };

  const getScoreColor = (score: number) => {
    if (score >= 80) return "text-green-600";
    if (score >= 60) return "text-orange-600";
    return "text-red-600";
  };

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Lead Pipeline</h1>
          <p className="text-muted-foreground">Track and manage your sales pipeline</p>
        </div>
        <Button>
          <Target className="mr-2 h-4 w-4" />
          Add Lead
        </Button>
      </div>

      {/* Pipeline Overview */}
      <div className="grid grid-cols-1 md:grid-cols-5 gap-4">
        {pipelineStages.map((stage, index) => (
          <Card key={index} className="relative overflow-hidden">
            <div className={`absolute top-0 left-0 right-0 h-1 ${stage.color}`}></div>
            <CardContent className="p-4">
              <div className="text-center">
                <h3 className="font-medium text-sm mb-2">{stage.name}</h3>
                <p className="text-2xl font-bold mb-1">{stage.count}</p>
                <p className="text-xs text-muted-foreground">${(stage.value / 1000).toFixed(0)}K value</p>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Pipeline Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-blue-700 font-medium">Total Pipeline</p>
                <p className="text-3xl font-bold text-blue-800">$3.6M</p>
              </div>
              <Target className="w-8 h-8 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-green-700 font-medium">Conversion Rate</p>
                <p className="text-3xl font-bold text-green-800">24%</p>
              </div>
              <TrendingUp className="w-8 h-8 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-purple-700 font-medium">Avg Deal Size</p>
                <p className="text-3xl font-bold text-purple-800">$485K</p>
              </div>
              <Star className="w-8 h-8 text-purple-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-orange-700 font-medium">Active Leads</p>
                <p className="text-3xl font-bold text-orange-800">66</p>
              </div>
              <Users className="w-8 h-8 text-orange-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Lead Management */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle>Lead Management</CardTitle>
            <div className="flex gap-2">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                <Input
                  placeholder="Search leads..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10 w-64"
                />
              </div>
              <Select value={filterStage} onValueChange={setFilterStage}>
                <SelectTrigger className="w-32">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Stages</SelectItem>
                  <SelectItem value="new">New</SelectItem>
                  <SelectItem value="qualified">Qualified</SelectItem>
                  <SelectItem value="proposal">Proposal</SelectItem>
                  <SelectItem value="negotiation">Negotiation</SelectItem>
                  <SelectItem value="closing">Closing</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {leads.map((lead) => (
              <Card key={lead.id} className="border hover:shadow-md transition-shadow">
                <CardContent className="p-4">
                  <div className="flex items-center justify-between mb-3">
                    <div className="flex items-center gap-3">
                      <div>
                        <h4 className="font-semibold">{lead.name}</h4>
                        <p className="text-sm text-muted-foreground">{lead.email}</p>
                      </div>
                      <Badge className={getStageColor(lead.stage)}>
                        {lead.stage.charAt(0).toUpperCase() + lead.stage.slice(1)}
                      </Badge>
                      <div className="flex items-center gap-1">
                        <Star className="w-4 h-4 text-yellow-500" />
                        <span className={`font-semibold ${getScoreColor(lead.score)}`}>
                          {lead.score}
                        </span>
                      </div>
                    </div>
                    <div className="text-right">
                      <p className="font-bold text-lg">${(lead.value / 1000).toFixed(0)}K</p>
                      <p className="text-sm text-muted-foreground">{lead.probability}% probability</p>
                    </div>
                  </div>
                  
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-3">
                    <div>
                      <p className="text-sm"><strong>Property:</strong> {lead.property}</p>
                      <p className="text-sm text-muted-foreground">Source: {lead.source}</p>
                    </div>
                    <div>
                      <p className="text-sm"><strong>Phone:</strong> {lead.phone}</p>
                      <p className="text-sm text-muted-foreground">Last contact: {lead.lastContact}</p>
                    </div>
                    <div className="flex gap-2">
                      <Button size="sm" variant="outline">
                        <Phone className="w-4 h-4 mr-1" />
                        Call
                      </Button>
                      <Button size="sm" variant="outline">
                        <Mail className="w-4 h-4 mr-1" />
                        Email
                      </Button>
                      <Button size="sm" variant="outline">
                        <MessageCircle className="w-4 h-4 mr-1" />
                        SMS
                      </Button>
                    </div>
                  </div>
                  
                  {/* Progress Bar */}
                  <div className="w-full bg-gray-200 rounded-full h-2">
                    <div 
                      className="bg-blue-600 h-2 rounded-full" 
                      style={{ width: `${lead.probability}%` }}
                    ></div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}