import React, { useState, useEffect } from "react";
import {
  Shield,
  FileText,
  Upload,
  CheckCircle,
  AlertTriangle,
} from "lucide-react";
import { useSearchParams, useLocation } from "react-router-dom";
import { LegalDocumentUpload } from "../../../../components/legal/LegalDocumentUpload";
import { LegalHealthStatus } from "../../../../components/legal/LegalHealthStatus";
import { useLegalHealth } from "../../../../hooks/useLegalHealth";

interface Property {
  id: string;
  title: string;
  address: string;
  city: string;
}

export default function LegalDocuments() {
  const [searchParams] = useSearchParams();
  const location = useLocation();
  const [properties, setProperties] = useState<Property[]>([]);
  const [selectedPropertyId, setSelectedPropertyId] = useState<string>("");
  const [loading, setLoading] = useState(true);

  const {
    legalHealth,
    loading: legalLoading,
    refetch,
  } = useLegalHealth(selectedPropertyId);

  useEffect(() => {
    fetchProperties();
  }, []);

  useEffect(() => {
    // Get property ID from URL params or navigation state
    const propertyIdFromUrl = searchParams.get("propertyId");
    const propertyIdFromState = location.state?.propertyId;
    const propertyId = propertyIdFromUrl || propertyIdFromState;

    if (propertyId && properties.length > 0) {
      setSelectedPropertyId(propertyId);
    }
  }, [searchParams, location.state, properties]);

  const fetchProperties = async () => {
    try {
      const token =
        localStorage.getItem("sellerToken") ||
        localStorage.getItem("token") ||
        localStorage.getItem("accessToken") ||
        localStorage.getItem("authToken");

      const sellerId = localStorage.getItem("sellerId");
      const timestamp = new Date().getTime();

      console.log("Legal Documents - Token:", token ? "Present" : "Missing");
      console.log("Legal Documents - Seller ID:", sellerId);

      if (!token) {
        console.log("Legal Documents - No token found");
        setLoading(false);
        return;
      }

      // Use the general properties endpoint with seller filter
      const url = sellerId
        ? `http://localhost:8090/api/properties?listed_by_id=${sellerId}&_t=${timestamp}`
        : `http://localhost:8090/api/properties?_t=${timestamp}`;

      const response = await fetch(url, {
        cache: "no-store",
        headers: {
          "Content-Type": "application/json",
          "Cache-Control": "no-cache",
          Pragma: "no-cache",
        },
      });

      if (response.ok) {
        const data = await response.json();
        const propertiesList = data.properties || [];
        console.log("Properties found:", propertiesList.length);
        setProperties(propertiesList);

        // Only auto-select first property if no property ID in URL/state
        const propertyIdFromUrl = new URLSearchParams(
          window.location.search,
        ).get("propertyId");
        if (
          propertiesList.length > 0 &&
          !selectedPropertyId &&
          !propertyIdFromUrl
        ) {
          setSelectedPropertyId(propertiesList[0].id.toString());
        }
      } else {
        console.log(
          "Failed to fetch properties:",
          response.status,
          response.statusText,
        );
        const errorText = await response.text().catch(() => "Unknown error");
        console.log("Error response:", errorText);
      }
    } catch (error) {
      console.error("Legal Documents - Error fetching properties:", error);
    } finally {
      setLoading(false);
    }
  };

  const handleUploadComplete = () => {
    refetch();
  };

  if (loading) {
    return (
      <div className="flex-1 p-8 overflow-y-auto">
        <div className="flex items-center justify-center h-64">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
            <p className="text-gray-600">Loading properties...</p>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 p-8 overflow-y-auto bg-gray-50">
      <div className="max-w-6xl mx-auto">
        {/* Header */}
        <div className="mb-8">
          <div className="flex items-center gap-3 mb-2">
            <Shield className="w-8 h-8 text-blue-600" />
            <h1 className="text-3xl font-bold text-gray-900">
              Legal Documents
            </h1>
          </div>
          <p className="text-gray-600">
            Manage legal documents for your properties to improve legal health
            scores
          </p>
        </div>

        {/* Property Selection */}
        {properties.length > 0 && (
          <div className="bg-white rounded-lg border border-gray-200 p-6 mb-6">
            <label className="block text-sm font-medium text-gray-700 mb-3">
              Select Property
            </label>
            <select
              value={selectedPropertyId}
              onChange={(e) => setSelectedPropertyId(e.target.value)}
              className="w-full max-w-md px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
              {properties.map((property) => (
                <option key={property.id} value={property.id}>
                  {property.title} - {property.city}
                </option>
              ))}
            </select>
          </div>
        )}

        {/* No Properties State */}
        {properties.length === 0 && (
          <div className="text-center py-12">
            <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-xl font-semibold text-gray-900 mb-2">
              No Properties Found
            </h3>
            <p className="text-gray-600 mb-6">
              Add a property first to manage legal documents
            </p>
            <button
              onClick={() =>
                (window.location.href = "/seller/dashboard/add-property")
              }
              className="bg-blue-600 hover:bg-blue-700 text-white px-6 py-3 rounded-lg font-medium transition-colors"
            >
              Add Property
            </button>
          </div>
        )}

        {/* Legal Health Status & Document Upload */}
        {selectedPropertyId && (
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Legal Health Status */}
            <div>
              {legalHealth ? (
                <LegalHealthStatus
                  legalHealth={legalHealth}
                  showDetails={true}
                  userRole="seller"
                />
              ) : legalLoading ? (
                <div className="bg-white border border-gray-200 rounded-lg p-6">
                  <div className="animate-pulse">
                    <div className="h-4 bg-gray-200 rounded w-1/2 mb-4"></div>
                    <div className="h-8 bg-gray-200 rounded mb-4"></div>
                    <div className="space-y-2">
                      <div className="h-4 bg-gray-200 rounded"></div>
                      <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                    </div>
                  </div>
                </div>
              ) : (
                <div className="bg-white border border-gray-200 rounded-lg p-6">
                  <div className="text-center py-8">
                    <AlertTriangle className="w-12 h-12 text-yellow-500 mx-auto mb-4" />
                    <h3 className="text-lg font-semibold text-gray-900 mb-2">
                      No Legal Health Data
                    </h3>
                    <p className="text-gray-600">
                      Upload documents to generate legal health status
                    </p>
                  </div>
                </div>
              )}
            </div>

            {/* Document Upload */}
            <div>
              {selectedPropertyId ? (
                <LegalDocumentUpload
                  propertyId={selectedPropertyId}
                  onUploadComplete={handleUploadComplete}
                />
              ) : null}
            </div>
          </div>
        )}

        {/* Help Section */}
        <div className="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-blue-900 mb-3">
            Why Upload Legal Documents?
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm text-blue-800">
            <div className="flex items-start gap-2">
              <CheckCircle className="w-4 h-4 text-blue-600 mt-0.5 flex-shrink-0" />
              <span>Build trust with potential buyers</span>
            </div>
            <div className="flex items-start gap-2">
              <CheckCircle className="w-4 h-4 text-blue-600 mt-0.5 flex-shrink-0" />
              <span>Faster property transactions</span>
            </div>
            <div className="flex items-start gap-2">
              <CheckCircle className="w-4 h-4 text-blue-600 mt-0.5 flex-shrink-0" />
              <span>Reduce legal risks and disputes</span>
            </div>
            <div className="flex items-start gap-2">
              <CheckCircle className="w-4 h-4 text-blue-600 mt-0.5 flex-shrink-0" />
              <span>Higher property listing visibility</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
