import { useState } from 'react';
import { Link } from 'wouter';
import { FileText, Clock, CheckCircle, AlertCircle, Upload, Calendar, RefreshCw } from 'lucide-react';

export default function LegalHealthReports() {
  const [activeTab, setActiveTab] = useState<'requests' | 'verified'>('requests');
  const [filter, setFilter] = useState('all');

  // Mock data for requests
  const requests = [
    {
      id: 1,
      propertyTitle: 'Modern 2BR Apartment - Downtown',
      buyerName: 'Rajesh Kumar',
      requestDate: '2024-01-15',
      status: 'pending_documents',
      documentsUploaded: 2,
      documentsRequired: 5
    },
    {
      id: 2,
      propertyTitle: '3BR Villa with Garden - Suburbs',
      buyerName: 'Priya Sharma',
      requestDate: '2024-01-10',
      status: 'under_verification',
      documentsUploaded: 5,
      documentsRequired: 5
    },
    {
      id: 3,
      propertyTitle: 'Luxury Villa - Whitefield',
      buyerName: 'Amit Patel',
      requestDate: '2024-01-05',
      status: 'completed',
      documentsUploaded: 5,
      documentsRequired: 5
    }
  ];

  // Mock data for verified properties
  const verifiedProperties = [
    {
      id: 1,
      title: 'Luxury Villa - Whitefield',
      verifiedDate: '2024-01-05',
      expiryDate: '2025-01-05',
      reportId: 'LHR-2024-001',
      status: 'active'
    },
    {
      id: 2,
      title: '3BR Villa with Garden - Suburbs',
      verifiedDate: '2023-12-20',
      expiryDate: '2024-12-20',
      reportId: 'LHR-2023-089',
      status: 'expiring_soon'
    }
  ];

  const getStatusConfig = (status: string) => {
    switch (status) {
      case 'pending_documents':
        return { label: 'Pending Documents', color: 'bg-yellow-100 text-yellow-800', icon: Clock };
      case 'documents_submitted':
        return { label: 'Documents Submitted', color: 'bg-blue-100 text-blue-800', icon: Upload };
      case 'under_verification':
        return { label: 'Under Verification', color: 'bg-purple-100 text-purple-800', icon: FileText };
      case 'completed':
        return { label: 'Completed', color: 'bg-green-100 text-green-800', icon: CheckCircle };
      default:
        return { label: 'Unknown', color: 'bg-gray-100 text-gray-800', icon: AlertCircle };
    }
  };

  const getVerifiedStatusBadge = (status: string) => {
    switch (status) {
      case 'active':
        return <span className="text-xs px-2 py-1 bg-green-100 text-green-800 rounded-full">Active</span>;
      case 'expiring_soon':
        return <span className="text-xs px-2 py-1 bg-yellow-100 text-yellow-800 rounded-full">Expiring Soon</span>;
      case 'expired':
        return <span className="text-xs px-2 py-1 bg-red-100 text-red-800 rounded-full">Expired</span>;
      default:
        return null;
    }
  };

  const filteredRequests = filter === 'all' 
    ? requests 
    : requests.filter(r => r.status === filter);

  const stats = {
    total: requests.length,
    pending: requests.filter(r => r.status === 'pending_documents').length,
    inProgress: requests.filter(r => r.status === 'under_verification').length,
    completed: requests.filter(r => r.status === 'completed').length
  };

  return (
    <div className="flex-1 p-8 overflow-y-auto">
      <div className="mb-6">
        <h1 className="text-2xl font-bold text-gray-900">Legal Health Reports</h1>
        <p className="text-sm text-gray-600 mt-1">Manage verification requests and verified properties</p>
      </div>

      {/* Tabs */}
      <div className="flex gap-2 mb-6 border-b border-gray-200">
        <button
          onClick={() => setActiveTab('requests')}
          className={`px-4 py-2 text-sm font-medium border-b-2 transition-colors ${
            activeTab === 'requests'
              ? 'border-blue-600 text-blue-600'
              : 'border-transparent text-gray-600 hover:text-gray-900'
          }`}
        >
          Buyer Requests
        </button>
        <button
          onClick={() => setActiveTab('verified')}
          className={`px-4 py-2 text-sm font-medium border-b-2 transition-colors ${
            activeTab === 'verified'
              ? 'border-blue-600 text-blue-600'
              : 'border-transparent text-gray-600 hover:text-gray-900'
          }`}
        >
          Verified Properties
        </button>
      </div>

      {/* Requests Tab */}
      {activeTab === 'requests' && (
        <>
          {/* Stats */}
          <div className="grid grid-cols-4 gap-4 mb-6">
            <div className="bg-white p-4 rounded-lg border border-gray-200">
              <div className="text-2xl font-bold text-gray-900">{stats.total}</div>
              <div className="text-xs text-gray-600">Total Requests</div>
            </div>
            <div className="bg-white p-4 rounded-lg border border-gray-200">
              <div className="text-2xl font-bold text-yellow-600">{stats.pending}</div>
              <div className="text-xs text-gray-600">Pending Action</div>
            </div>
            <div className="bg-white p-4 rounded-lg border border-gray-200">
              <div className="text-2xl font-bold text-purple-600">{stats.inProgress}</div>
              <div className="text-xs text-gray-600">In Verification</div>
            </div>
            <div className="bg-white p-4 rounded-lg border border-gray-200">
              <div className="text-2xl font-bold text-green-600">{stats.completed}</div>
              <div className="text-xs text-gray-600">Completed</div>
            </div>
          </div>

          {/* Filters */}
          <div className="flex gap-2 mb-4">
            <button
              onClick={() => setFilter('all')}
              className={`px-3 py-1 text-sm rounded-lg ${filter === 'all' ? 'bg-blue-600 text-white' : 'bg-white text-gray-700 border border-gray-200'}`}
            >
              All
            </button>
            <button
              onClick={() => setFilter('pending_documents')}
              className={`px-3 py-1 text-sm rounded-lg ${filter === 'pending_documents' ? 'bg-blue-600 text-white' : 'bg-white text-gray-700 border border-gray-200'}`}
            >
              Pending
            </button>
            <button
              onClick={() => setFilter('under_verification')}
              className={`px-3 py-1 text-sm rounded-lg ${filter === 'under_verification' ? 'bg-blue-600 text-white' : 'bg-white text-gray-700 border border-gray-200'}`}
            >
              In Progress
            </button>
            <button
              onClick={() => setFilter('completed')}
              className={`px-3 py-1 text-sm rounded-lg ${filter === 'completed' ? 'bg-blue-600 text-white' : 'bg-white text-gray-700 border border-gray-200'}`}
            >
              Completed
            </button>
          </div>

          {/* Requests Table */}
          <div className="bg-white rounded-lg border border-gray-200">
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-gray-50 border-b border-gray-200">
                  <tr>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-600">Property</th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-600">Buyer</th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-600">Request Date</th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-600">Documents</th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-600">Status</th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-gray-600">Action</th>
                  </tr>
                </thead>
                <tbody className="divide-y divide-gray-200">
                  {filteredRequests.map((request) => {
                    const statusConfig = getStatusConfig(request.status);
                    const StatusIcon = statusConfig.icon;
                    return (
                      <tr key={request.id} className="hover:bg-gray-50">
                        <td className="px-4 py-3 text-sm text-gray-900">{request.propertyTitle}</td>
                        <td className="px-4 py-3 text-sm text-gray-900">{request.buyerName}</td>
                        <td className="px-4 py-3 text-sm text-gray-600">{new Date(request.requestDate).toLocaleDateString()}</td>
                        <td className="px-4 py-3 text-sm text-gray-600">
                          {request.documentsUploaded}/{request.documentsRequired}
                        </td>
                        <td className="px-4 py-3">
                          <span className={`inline-flex items-center gap-1 px-2 py-1 text-xs font-medium rounded-full ${statusConfig.color}`}>
                            <StatusIcon className="w-3 h-3" />
                            {statusConfig.label}
                          </span>
                        </td>
                        <td className="px-4 py-3">
                          <Link
                            href={`/legal-health-requests/${request.id}`}
                            className="text-sm text-blue-600 hover:text-blue-800 font-medium"
                          >
                            View Details
                          </Link>
                        </td>
                      </tr>
                    );
                  })}
                </tbody>
              </table>
            </div>
          </div>
        </>
      )}

      {/* Verified Properties Tab */}
      {activeTab === 'verified' && (
        <>
          {/* Benefits Banner */}
          <div className="bg-gradient-to-r from-green-50 to-blue-50 p-4 rounded-lg border border-green-200 mb-6">
            <div className="flex items-start gap-3">
              <CheckCircle className="w-5 h-5 text-green-600 mt-0.5" />
              <div>
                <h3 className="text-sm font-semibold text-gray-900 mb-1">Benefits of Verified Badge</h3>
                <ul className="text-xs text-gray-700 space-y-1">
                  <li>✓ Property shows "Legal Health Verified" tag to all buyers</li>
                  <li>✓ Faster sales - Buyers trust verified properties</li>
                  <li>✓ Higher price - Can justify better pricing</li>
                  <li>✓ Transparency - Shows you have nothing to hide</li>
                </ul>
              </div>
            </div>
          </div>

          {/* Verified Properties List */}
          <div className="space-y-4">
            {verifiedProperties.map((property) => (
              <div key={property.id} className="bg-white p-4 rounded-lg border border-gray-200">
                <div className="flex items-start justify-between">
                  <div className="flex-1">
                    <div className="flex items-center gap-2 mb-2">
                      <h3 className="text-base font-semibold text-gray-900">{property.title}</h3>
                      <span className="inline-flex items-center gap-1 px-2 py-1 bg-green-100 text-green-800 text-xs font-medium rounded-full">
                        <CheckCircle className="w-3 h-3" />
                        Verified
                      </span>
                      {getVerifiedStatusBadge(property.status)}
                    </div>
                    
                    <div className="grid grid-cols-3 gap-4 text-sm">
                      <div>
                        <span className="text-gray-600">Report ID:</span>
                        <span className="ml-2 text-gray-900 font-medium">{property.reportId}</span>
                      </div>
                      <div>
                        <span className="text-gray-600">Verified On:</span>
                        <span className="ml-2 text-gray-900">{new Date(property.verifiedDate).toLocaleDateString()}</span>
                      </div>
                      <div>
                        <span className="text-gray-600">Valid Until:</span>
                        <span className="ml-2 text-gray-900">{new Date(property.expiryDate).toLocaleDateString()}</span>
                      </div>
                    </div>
                  </div>
                  
                  <div className="flex gap-2">
                    <button className="px-3 py-1 text-sm text-blue-600 border border-blue-600 rounded hover:bg-blue-50">
                      View Report
                    </button>
                    {property.status === 'expiring_soon' && (
                      <button className="px-3 py-1 text-sm bg-blue-600 text-white rounded hover:bg-blue-700 flex items-center gap-1">
                        <RefreshCw className="w-3 h-3" />
                        Renew
                      </button>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>

          {/* Empty State */}
          {verifiedProperties.length === 0 && (
            <div className="bg-white p-12 rounded-lg border border-gray-200 text-center">
              <CheckCircle className="w-12 h-12 text-gray-400 mx-auto mb-3" />
              <h3 className="text-base font-semibold text-gray-900 mb-1">No Verified Properties Yet</h3>
              <p className="text-sm text-gray-600 mb-4">
                Once buyers request legal health reports and verification is completed, your properties will appear here with verified badges.
              </p>
            </div>
          )}
        </>
      )}
    </div>
  );
}
