import React from "react";
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
} from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "../components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "../components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "../components/ui/alert-dialog";
import { Input } from "../components/ui/input";
import { Label } from "../components/ui/label";
import { Textarea } from "../components/ui/textarea";
import {
  Eye,
  Edit,
  Pause,
  Trash2,
  Plus,
  MapPin,
  Calendar,
  DollarSign,
  Bed,
  Bath,
  Square,
  Play,
  X,
} from "lucide-react";
import { Link } from "wouter";
import { useState, useEffect, useMemo } from "react";
import { toast } from "sonner";
import { api } from "../utils/api";
import { api as mainApi } from "../../../../utils/api";
import type { Property as PropertyRow } from "../types";
import { mockProperties } from "../lib/mock-data";
import { PropertyCard as MainPropertyCard } from "../../../../components/PropertyCard";
import SellerPropertyCard from "../components/SellerPropertyCard";
import AddAssetModal from "../components/AddAssetModal";
import AddPropertyChoiceModal from "../components/AddPropertyChoiceModal";
import PropertyDetailsModal from "../components/PropertyDetailsModal";

// PropertyDetailContent component for full property view
const PropertyDetailContent = ({ property }: { property: any }) => {
  const formatPrice = (price: string | number) => {
    const numPrice = typeof price === "string" ? parseFloat(price) : price;
    if (isNaN(numPrice) || numPrice === 0) return "Price on request";
    if (numPrice >= 10000000) return `₹${(numPrice / 10000000).toFixed(1)} Cr`;
    if (numPrice >= 100000) return `₹${(numPrice / 100000).toFixed(0)} L`;
    return `₹${numPrice.toLocaleString()}`;
  };

  return (
    <div className="max-w-7xl mx-auto px-4 py-6">
      {/* Property Header */}
      <div className="mb-6">
        <h1 className="text-3xl font-bold text-gray-900 mb-2">
          {property.title}
        </h1>
        <p className="text-gray-600 mb-4">
          {property.city}, {property.state}
        </p>
        <div className="flex items-center justify-between mb-4">
          <span className="text-3xl font-bold text-blue-600">
            {formatPrice(property.price || 0)}
          </span>
        </div>
      </div>

      {/* Property Details Grid */}
      <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
        <div className="text-center p-4 bg-gray-50 rounded-lg">
          <div className="text-2xl font-bold text-blue-600">
            {property.bhk || "N/A"}
          </div>
          <div className="text-sm text-gray-600">BHK</div>
        </div>
        <div className="text-center p-4 bg-gray-50 rounded-lg">
          <div className="text-2xl font-bold text-blue-600">
            {property.bathrooms || "N/A"}
          </div>
          <div className="text-sm text-gray-600">Bathrooms</div>
        </div>
        <div className="text-center p-4 bg-gray-50 rounded-lg">
          <div className="text-2xl font-bold text-blue-600">
            {property.area || "N/A"}
          </div>
          <div className="text-sm text-gray-600">sq.ft</div>
        </div>
        <div className="text-center p-4 bg-gray-50 rounded-lg">
          <div className="text-2xl font-bold text-blue-600 capitalize">
            {property.type || "Property"}
          </div>
          <div className="text-sm text-gray-600">Type</div>
        </div>
      </div>

      {/* Property Features */}
      <div className="mb-6">
        <h3 className="text-lg font-semibold mb-3">Property Features</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="flex items-center space-x-2">
            <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
            <span>Property Type: {property.type}</span>
          </div>
          <div className="flex items-center space-x-2">
            <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
            <span>Status: {property.status}</span>
          </div>
          <div className="flex items-center space-x-2">
            <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
            <span>Area: {property.area} sq.ft</span>
          </div>
          <div className="flex items-center space-x-2">
            <span className="w-2 h-2 bg-blue-500 rounded-full"></span>
            <span>City: {property.city}</span>
          </div>
        </div>
      </div>

      {/* Amenities */}
      {property.amenities && property.amenities.length > 0 && (
        <div className="mb-6">
          <h3 className="text-lg font-semibold mb-3">Amenities</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {property.amenities.map((amenity: string, index: number) => (
              <div
                key={index}
                className="flex items-center space-x-2 p-3 bg-gray-50 rounded-lg"
              >
                <svg
                  className="w-5 h-5 text-green-500"
                  fill="currentColor"
                  viewBox="0 0 20 20"
                >
                  <path
                    fillRule="evenodd"
                    d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z"
                    clipRule="evenodd"
                  />
                </svg>
                <span className="text-gray-700">{amenity}</span>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Location */}
      <div className="mb-6">
        <h3 className="text-lg font-semibold mb-3">Location</h3>
        <div className="bg-gray-50 p-4 rounded-lg">
          <p className="text-gray-700 mb-2">
            <strong>Address:</strong>{" "}
            {property.address || "Address not available"}
          </p>
          <p className="text-gray-700 mb-2">
            <strong>City:</strong> {property.city}, {property.state}
          </p>
          {property.pincode && (
            <p className="text-gray-700">
              <strong>Pincode:</strong> {property.pincode}
            </p>
          )}
        </div>
      </div>
    </div>
  );
};

export default function Properties() {
  const [filters, setFilters] = useState({
    status: "all",
    listingIntent: "all",
    city: "",
    search: "",
  });
  const [selectedProperty, setSelectedProperty] = useState<any>(null);
  const [isViewDialogOpen, setIsViewDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [isAddChoiceModalOpen, setIsAddChoiceModalOpen] = useState(false);
  const [isAddAssetModalOpen, setIsAddAssetModalOpen] = useState(false);
  const [editForm, setEditForm] = useState({
    title: "",
    location: "",
    status: "",
    price: "",
    description: "",
  });
  const [properties, setProperties] = useState<PropertyRow[]>([]);
  const [loading, setLoading] = useState(true);

  const loadProperties = async (showLoading = true) => {
    try {
      if (showLoading) setLoading(true);
      
      // Auto-cleanup duplicates on first load
      if (showLoading) {
        try {
          const cleanupResult = await mainApi.cleanupDuplicates();
          if (cleanupResult.deleted_ids?.length > 0) {
            console.log(`🧹 Cleaned up ${cleanupResult.deleted_ids.length} duplicate properties`);
          }
        } catch (error) {
          console.warn('Cleanup failed:', error);
        }
      }
      
      const list = await mainApi.fetchProperties();
      
      // CRITICAL FIX: Remove duplicates based on property ID
      const uniqueProperties = list.filter((property, index, self) => 
        index === self.findIndex(p => p.id === property.id)
      );
      
      if (list.length !== uniqueProperties.length) {
        console.warn(`🚨 Removed ${list.length - uniqueProperties.length} duplicate properties`);
      }
      
      setProperties(uniqueProperties);
    } catch (error) {
      console.error("Failed to load properties:", error);
      // Fallback to mock data if API is unreachable
      const fallback: PropertyRow[] = mockProperties.map((p: any) => ({
        id: String(p.id),
        title: p.title,
        type: (p.type || p.propertyType || "apartment")
          .toString()
          .toLowerCase(),
        bhk: p.bhk ?? undefined,
        address: p.address || p.location || "",
        city: p.city || "",
        state: p.state || "",
        pincode: p.pincode || "",
        landmark: p.landmark || "",
        area: Number(p.area ?? p.size ?? 0),
        price: Number(p.price ?? 0),
        listingIntent:
          p.listingIntent || (p.isUrgentSale ? "urgent-sale" : "sale"),
        status: p.status || "active",
        amenities: p.features || [],
        customAmenities: [],
        virtualTour: undefined,
        sellerType: "owner",
        biddingEnabled: Boolean(p.isBiddingEnabled),
        coordinates: undefined,
        views: Number(p.views || 0),
        inquiries: 0,
        createdAt:
          p.createdAt instanceof Date
            ? p.createdAt.toISOString()
            : String(p.createdAt || ""),
        updatedAt:
          p.updatedAt instanceof Date
            ? p.updatedAt.toISOString()
            : String(p.updatedAt || ""),
      }));
      setProperties(fallback);
    } finally {
      if (showLoading) setLoading(false);
    }
  };

  useEffect(() => {
    loadProperties();

    // Listen for property added event
    const handlePropertyAdded = (event: CustomEvent) => {
      loadProperties(false);
      if (event.detail) {
        toast.success("Property added successfully!", {
          description: "Your new property is now visible in the list.",
          duration: 3000,
        });
      }
    };
    window.addEventListener(
      "propertyAdded",
      handlePropertyAdded as EventListener,
    );

    return () => {
      window.removeEventListener(
        "propertyAdded",
        handlePropertyAdded as EventListener,
      );
    };
  }, []);

  const getStatusColor = (status: string) => {
    switch (status) {
      case "active":
        return "bg-green-100 text-green-800";
      case "inactive":
        return "bg-orange-100 text-orange-800";
      case "under_review":
        return "bg-blue-100 text-blue-800";
      case "sold":
        return "bg-purple-100 text-purple-800";
      case "rented":
        return "bg-indigo-100 text-indigo-800";
      case "pending":
        return "bg-yellow-100 text-yellow-800";
      case "approved":
        return "bg-emerald-100 text-emerald-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case "active":
        return "Active";
      case "inactive":
        return "Inactive";
      case "under_review":
        return "Under Review";
      case "sold":
        return "Sold";
      case "rented":
        return "Rented";
      case "pending":
        return "Pending";
      case "approved":
        return "Approved";
      default:
        return status;
    }
  };

  // Filter properties based on current filters
  const filteredProperties = useMemo(() => {
    return properties.filter((property: PropertyRow) => {
      // Show assets only after admin approval (approved/active/sold)
      // Show regular listings including pending ones
      const isAsset = property.isasset === true;
      const isValidAsset = isAsset && ["approved", "active", "sold"].includes(property.status);
      const isValidRegularListing = !isAsset && ["active", "pending", "sold", "rented", "under_review", "inactive"].includes(property.status);

      // Only show valid assets or valid regular listings
      if (!isValidAsset && !isValidRegularListing) {
        return false;
      }

      const matchesStatus =
        filters.status === "all" || property.status === filters.status;
      const matchesIntent =
        filters.listingIntent === "all" ||
        property.listingIntent === filters.listingIntent;
      const matchesCity = !filters.city || property.city === filters.city;
      const matchesSearch =
        !filters.search ||
        property.title.toLowerCase().includes(filters.search.toLowerCase()) ||
        property.address.toLowerCase().includes(filters.search.toLowerCase()) ||
        property.city.toLowerCase().includes(filters.search.toLowerCase());

      return matchesStatus && matchesIntent && matchesCity && matchesSearch;
    });
  }, [properties, filters]);

  if (loading) return <div className="p-4">Loading properties…</div>;

  const handleListForSale = async (property: any) => {
    try {
      console.log('🚀 Listing asset for sale:', property.id, property.title);
      console.log('🔍 Property before listing:', property);
      
      const result = await mainApi.listAssetForSale(property.id);
      
      console.log('✅ Asset listed successfully:', result);
      console.log('🔍 Seller details received:', result.seller_details);
      
      toast.success("Asset listed for sale!", {
        description:
          "Your asset is now pending admin approval for public listing.",
      });
      
      // Store seller details and property data for the post property form
      const formData = {
        // CRITICAL: Store the original property ID to update instead of create
        propertyId: property.id,
        isAssetUpdate: true,
        
        // Property details
        title: property.title,
        description: property.description || '',
        propertyType: property.type,
        listingType: 'sell',
        address: property.address,
        city: property.city,
        state: property.state,
        pincode: property.pincode,
        locality: property.landmark,
        builtUpArea: property.area,
        bedrooms: property.bhk ? property.bhk.replace('BHK', '') : '',
        bathrooms: property.bathrooms,
        salePrice: property.price,
        
        // Seller details from backend
        ownerName: result.seller_details.fullname || '',
        email: result.seller_details.email || '',
        mobile: result.seller_details.mobile || '',
        
        // Additional seller details if available
        bio: result.seller_details.bio || '',
        ownershipType: result.seller_details.ownership_type || '',
        sellingMethod: result.seller_details.selling_method || '',
        sellingTimeline: result.seller_details.selling_timeline || '',
        priceFlexibility: result.seller_details.price_flexibility || false
      };
      
      console.log('📋 Storing form data:', formData);
      
      // Store in localStorage for the post property form
      localStorage.setItem('asset-listing-data', JSON.stringify(formData));
      
      console.log('💾 Stored in localStorage:', localStorage.getItem('asset-listing-data'));
      
      // Navigate to post property form
      window.location.href = `/listing-form?from=asset&propertyId=${property.id}`;
      
      // Refresh properties list
      console.log('🔄 Refreshing properties list...');
      loadProperties(false);
    } catch (error: any) {
      console.error('❌ Failed to list asset:', error);
      toast.error("Failed to list asset for sale", {
        description: error.message || "Please try again later",
      });
    }
  };

  const handleView = (property: any) => {
    setSelectedProperty(property);
    setIsViewDialogOpen(true);
  };

  const handleEdit = (property: any) => {
    setSelectedProperty(property);
    setEditForm({
      title: property.title,
      location: property.location,
      status: property.status,
      price: property.price,
      description: property.description || "",
    });
    setIsEditDialogOpen(true);
  };

  const handlePause = (property: any) => {
    // Toggle between active and inactive for demo purposes
    const newStatus = property.status === "active" ? "inactive" : "active";

    // Update the properties state to trigger re-render
    setProperties((prevProperties) =>
      prevProperties.map((p) =>
        p.id === property.id ? { ...p, status: newStatus } : p,
      ),
    );

    // In real app, update backend here
    // No alert needed - state change is immediate and visible
  };

  const handleDelete = (property: any) => {
    setSelectedProperty(property);
    setIsDeleteDialogOpen(true);
  };

  const confirmDelete = async () => {
    if (selectedProperty) {
      try {
        await mainApi.deleteProperty(selectedProperty.id);
        setProperties((prevProperties) =>
          prevProperties.filter((p) => p.id !== selectedProperty.id),
        );
        toast.success("Property deleted successfully!");
      } catch (error: any) {
        toast.error("Failed to delete property", {
          description: error.message || "Please try again later",
        });
      } finally {
        setIsDeleteDialogOpen(false);
        setSelectedProperty(null);
      }
    }
  };

  return (
    <div className="flex-1 overflow-y-auto">
      {/* Header */}
      <div className="bg-white border-b border-gray-200 px-4 py-3">
        <h1 className="text-2xl font-semibold text-gray-900">My Properties</h1>
        <p className="text-sm text-gray-600">
          Manage and track your property listings
        </p>
      </div>
      <div className="p-0">
        <div className="flex flex-col lg:flex-row lg:items-center justify-between gap-4 mb-6">
          <div>
            <h2 className="text-2xl font-bold text-gray-900">
              Properties ({filteredProperties.length})
            </h2>
            <p className="text-gray-600 mt-1">Manage your property listings</p>
          </div>

          <div className="flex flex-col sm:flex-row gap-4 items-start sm:items-center">
            <div className="flex gap-2">
              <Input
                placeholder="Search properties..."
                value={filters.search}
                onChange={(e) =>
                  setFilters({ ...filters, search: e.target.value })
                }
                className="w-48"
              />
              <Select
                value={filters.status}
                onValueChange={(value) =>
                  setFilters({ ...filters, status: value })
                }
              >
                <SelectTrigger className="w-32">
                  <SelectValue placeholder="Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="inactive">Inactive</SelectItem>
                  <SelectItem value="under_review">Under Review</SelectItem>
                  <SelectItem value="sold">Sold</SelectItem>
                  <SelectItem value="rented">Rented</SelectItem>
                  <SelectItem value="pending">Pending</SelectItem>
                  <SelectItem value="approved">Approved</SelectItem>
                </SelectContent>
              </Select>
              <Select
                value={filters.listingIntent}
                onValueChange={(value) =>
                  setFilters({ ...filters, listingIntent: value })
                }
              >
                <SelectTrigger className="w-32">
                  <SelectValue placeholder="Intent" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Intent</SelectItem>
                  <SelectItem value="sale">Sale</SelectItem>
                  <SelectItem value="rent">Rent</SelectItem>
                  <SelectItem value="urgent-sale">Urgent Sale</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <Button onClick={() => setIsAddChoiceModalOpen(true)}>
              <Plus className="mr-2 h-4 w-4" />
              Add Property
            </Button>
          </div>
        </div>

        {/* Properties Grid */}
        {filteredProperties.length > 0 ? (
          <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
            {filteredProperties.map((property) => (
              <SellerPropertyCard
                key={property.id}
                property={property}
                onEdit={handleEdit}
                onPause={handlePause}
                onDelete={handleDelete}
                onView={handleView}
                onListForSale={handleListForSale}
              />
            ))}
          </div>
        ) : (
          <div className="text-center py-12">
            <svg
              className="mx-auto w-16 h-16 text-gray-400 mb-4"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={1}
                d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"
              />
            </svg>
            <h3 className="text-lg font-medium text-gray-900 mb-2">
              No properties found
            </h3>
            <p className="text-gray-600 mb-4">
              Try adjusting your filters or search terms
            </p>
            <button
              onClick={() =>
                setFilters({
                  status: "all",
                  listingIntent: "all",
                  city: "",
                  search: "",
                })
              }
              className="text-blue-600 hover:underline"
            >
              Clear all filters
            </button>
          </div>
        )}

        {/* Property Details Modal */}
        <PropertyDetailsModal
          property={selectedProperty}
          isOpen={isViewDialogOpen}
          onClose={() => setIsViewDialogOpen(false)}
          onEdit={(id) => {
            const property = properties.find((p) => p.id === id);
            if (property) handleEdit(property);
          }}
          onDelete={(id) => {
            const property = properties.find((p) => p.id === id);
            if (property) handleDelete(property);
          }}
        />

        {/* Edit Property Dialog */}
        <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
          <DialogContent className="max-w-3xl">
            <DialogHeader>
              <DialogTitle>Edit Property</DialogTitle>
            </DialogHeader>
            {selectedProperty && (
              <form
                className="space-y-4"
                onSubmit={(e) => {
                  e.preventDefault();

                  // Update the properties state with new data
                  setProperties((prevProperties) =>
                    prevProperties.map((p) =>
                      p.id === selectedProperty.id
                        ? {
                            ...p,
                            title: editForm.title,
                            location: editForm.location,
                            status: editForm.status,
                            price: Number(editForm.price),
                            description: editForm.description,
                          }
                        : p,
                    ),
                  );

                  // In real app, submit updated data to backend here
                  setIsEditDialogOpen(false);
                }}
              >
                <div>
                  <Label htmlFor="title">Property Title</Label>
                  <Input
                    id="title"
                    value={editForm.title}
                    onChange={(e) =>
                      setEditForm({ ...editForm, title: e.target.value })
                    }
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="location">Location</Label>
                  <Input
                    id="location"
                    value={editForm.location}
                    onChange={(e) =>
                      setEditForm({ ...editForm, location: e.target.value })
                    }
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="status">Status</Label>
                  <Select
                    value={editForm.status}
                    onValueChange={(value) =>
                      setEditForm({ ...editForm, status: value })
                    }
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="active">Active</SelectItem>
                      <SelectItem value="inactive">Inactive</SelectItem>
                      <SelectItem value="under_review">Under Review</SelectItem>
                      <SelectItem value="sold">Sold</SelectItem>
                      <SelectItem value="rented">Rented</SelectItem>
                      <SelectItem value="pending">Pending</SelectItem>
                      <SelectItem value="approved">Approved</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label htmlFor="price">Price</Label>
                  <Input
                    id="price"
                    type="number"
                    value={editForm.price}
                    onChange={(e) =>
                      setEditForm({ ...editForm, price: e.target.value })
                    }
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="description">Description</Label>
                  <Textarea
                    id="description"
                    value={editForm.description}
                    onChange={(e) =>
                      setEditForm({ ...editForm, description: e.target.value })
                    }
                    rows={3}
                  />
                </div>
                <div className="flex justify-end gap-2">
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() => setIsEditDialogOpen(false)}
                  >
                    Cancel
                  </Button>
                  <Button type="submit">Save</Button>
                </div>
              </form>
            )}
          </DialogContent>
        </Dialog>

        {/* Delete Confirmation Dialog */}
        <AlertDialog
          open={isDeleteDialogOpen}
          onOpenChange={setIsDeleteDialogOpen}
        >
          <AlertDialogTrigger />
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Confirm Delete</AlertDialogTitle>
              <AlertDialogDescription>
                Are you sure you want to delete the property "
                {selectedProperty?.title}"? This action cannot be undone.
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Cancel</AlertDialogCancel>
              <AlertDialogAction onClick={confirmDelete}>
                Delete
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        {/* Add Property Choice Modal */}
        <Dialog
          open={isAddChoiceModalOpen}
          onOpenChange={setIsAddChoiceModalOpen}
        >
          <DialogContent className="max-w-4xl">
            <AddPropertyChoiceModal
              onClose={() => setIsAddChoiceModalOpen(false)}
              onAddAsset={() => {
                setIsAddChoiceModalOpen(false);
                setIsAddAssetModalOpen(true);
              }}
              onListProperty={() => {
                setIsAddChoiceModalOpen(false);
                window.location.href = "/seller/dashboard/add-property";
              }}
            />
          </DialogContent>
        </Dialog>

        {/* Add Asset Modal */}
        <Dialog
          open={isAddAssetModalOpen}
          onOpenChange={setIsAddAssetModalOpen}
        >
          <DialogContent className="max-w-3xl h-[80vh] flex flex-col">
            <DialogHeader className="flex-shrink-0">
              <DialogTitle>Add Property to Assets</DialogTitle>
            </DialogHeader>
            <div className="flex-1 overflow-y-auto p-1">
              <AddAssetModal onClose={() => setIsAddAssetModalOpen(false)} />
            </div>
          </DialogContent>
        </Dialog>
      </div>
    </div>
  );
}
