import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Badge } from "../components/ui/badge";
import { Button } from "../components/ui/button";
import { TrendingUp, TrendingDown, Activity, MapPin, Eye, Target, AlertTriangle, Zap, Loader2, BarChart3, Users, Home } from "lucide-react";
import { useState, useEffect } from "react";
import { toast } from "sonner";

interface MarketSummary {
  seller_summary: {
    total_properties: number;
    active_properties: number;
    sold_properties: number;
    avg_listed_price: number;
    avg_days_on_market: number;
  };
  market_overview: {
    total_active_properties: number;
    total_localities: number;
    market_avg_price: number;
    properties_sold_30d: number;
    price_trend: string;
  };
}

interface AreaAnalysis {
  locality: string;
  avg_market_price: number;
  active_supply: number;
  total_views: number;
  demand_vs_supply_ratio: number;
  avg_days_on_market: number;
  market_hotness: string;
  avg_market_price_formatted: string;
}

interface PriceBand {
  price_range: string;
  avg_days_to_sell: number;
  total_properties: number;
  sold_properties: number;
  success_rate: number;
  recommendation: string;
}

interface SeasonalTrend {
  month: string;
  avg_price: number;
  sold_count: number;
  avg_days_on_market: number;
  demand_level: string;
  avg_price_formatted: string;
  activity_count?: number;
}

interface SellerPropertyPerformance {
  total_views: number;
  views_trend: number;
  total_inquiries: number;
  inquiries_trend: number;
  days_listed: number;
  area_avg_days: number;
}

interface CompetitivePricing {
  property_title: string;
  property_locality: string;
  your_price: number;
  market_avg_price: number;
  price_difference_percent: number;
  optimal_price_min: number;
  optimal_price_max: number;
  your_price_per_sqft: number;
  market_price_per_sqft: number;
  similar_properties_count: number;
  recommendation: string;
}

interface TimeToSell {
  estimated_days_min: number;
  estimated_days_max: number;
  if_reduced_5_percent_min: number;
  if_reduced_5_percent_max: number;
  if_reduced_10_percent_min: number;
  if_reduced_10_percent_max: number;
  market_avg_days: number;
}

interface DemandAnalysis {
  demand_level: string;
  demand_trend: string;
  searches_per_week: number;
  demand_score: number;
  insight_message: string;
}

interface SellerProperty {
  id: number;
  title: string;
  price: number;
  locality: string;
  days_listed: number;
  views: number;
  inquiries: number;
  price_comparison: string;
  recommendation: string;
}

export default function MarketIntelligence() {
  const [loading, setLoading] = useState(true);
  const [marketSummary, setMarketSummary] = useState<MarketSummary | null>(null);
  const [areaAnalysis, setAreaAnalysis] = useState<AreaAnalysis[]>([]);
  const [priceBands, setPriceBands] = useState<PriceBand[]>([]);
  const [seasonalTrends, setSeasonalTrends] = useState<SeasonalTrend[]>([]);
  const [propertyPerformance, setPropertyPerformance] = useState<SellerPropertyPerformance | null>(null);
  const [competitivePricing, setCompetitivePricing] = useState<CompetitivePricing | null>(null);
  const [timeToSell, setTimeToSell] = useState<TimeToSell | null>(null);
  const [demandAnalysis, setDemandAnalysis] = useState<DemandAnalysis | null>(null);

  useEffect(() => {
    fetchMarketData();
  }, []);

  const fetchMarketData = async () => {
    try {
      setLoading(true);
      const baseUrl = 'http://localhost:8090/api';
      
      // Test backend connectivity first
      console.log('Testing backend connectivity at:', baseUrl);
      const testRes = await fetch(`${baseUrl}/seller-market-trends/test-data`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });
      
      if (!testRes.ok) {
        const errorText = await testRes.text();
        console.error('Backend test failed:', testRes.status, errorText);
        throw new Error(`Backend returned ${testRes.status}: ${errorText}`);
      }
      
      const testData = await testRes.json();
      console.log('Backend test successful:', testData);
      
      // Fetch all market intelligence data
      const [summaryRes, areaRes, bandsRes, trendsRes] = await Promise.all([
        fetch(`${baseUrl}/seller-market-trends/market-summary`),
        fetch(`${baseUrl}/seller-market-trends/area-analysis`),
        fetch(`${baseUrl}/seller-market-trends/price-bands`),
        fetch(`${baseUrl}/seller-market-trends/seasonal-trends`)
      ]);

      let dataLoaded = false;

      if (summaryRes.ok) {
        const summaryData = await summaryRes.json();
        console.log('Market summary data:', summaryData);
        if (summaryData.success) {
          setMarketSummary(summaryData.data);
          dataLoaded = true;
        }
      } else {
        console.error('Market summary API failed:', summaryRes.status, await summaryRes.text());
      }

      if (areaRes.ok) {
        const areaData = await areaRes.json();
        console.log('Area analysis data:', areaData);
        if (areaData.success) {
          setAreaAnalysis(areaData.data.areas || []);
          dataLoaded = true;
        }
      } else {
        console.error('Area analysis API failed:', areaRes.status, await areaRes.text());
      }

      if (bandsRes.ok) {
        const bandsData = await bandsRes.json();
        console.log('Price bands data:', bandsData);
        if (bandsData.success) {
          setPriceBands(bandsData.data.price_bands || []);
          dataLoaded = true;
        }
      } else {
        console.error('Price bands API failed:', bandsRes.status, await bandsRes.text());
      }

      if (trendsRes.ok) {
        const trendsData = await trendsRes.json();
        console.log('Seasonal trends data:', trendsData);
        if (trendsData.success) {
          setSeasonalTrends(trendsData.data.trends || []);
          dataLoaded = true;
        }
      } else {
        console.error('Seasonal trends API failed:', trendsRes.status, await trendsRes.text());
      }

      // Fetch seller-specific data
      const [perfRes, pricingRes, timeRes, demandRes] = await Promise.all([
        fetch(`${baseUrl}/seller-market-trends/property-performance`),
        fetch(`${baseUrl}/seller-market-trends/competitive-pricing`),
        fetch(`${baseUrl}/seller-market-trends/time-to-sell`),
        fetch(`${baseUrl}/seller-market-trends/demand-analysis`)
      ]);

      if (perfRes.ok) {
        const perfData = await perfRes.json();
        if (perfData.success) {
          setPropertyPerformance(perfData.data);
          dataLoaded = true;
        }
      }

      if (pricingRes.ok) {
        const pricingData = await pricingRes.json();
        if (pricingData.success) {
          setCompetitivePricing(pricingData.data);
          dataLoaded = true;
        }
      }

      if (timeRes.ok) {
        const timeData = await timeRes.json();
        if (timeData.success) {
          setTimeToSell(timeData.data);
          dataLoaded = true;
        }
      }

      if (demandRes.ok) {
        const demandData = await demandRes.json();
        if (demandData.success) {
          setDemandAnalysis(demandData.data);
          dataLoaded = true;
        }
      }

      if (dataLoaded) {
        toast.success('Market intelligence data loaded successfully');
      } else {
        toast.warning('Backend connected but no data available');
      }

    } catch (error) {
      console.error('Error fetching market data:', error);
      toast.error(`Backend connection failed: ${error.message}`);
    } finally {
      setLoading(false);
    }
  };

  const getHotnessColor = (hotness: string) => {
    switch (hotness.toLowerCase()) {
      case 'hot': return 'text-red-600 bg-red-50 border-red-200';
      case 'warm': return 'text-orange-600 bg-orange-50 border-orange-200';
      case 'cool': return 'text-blue-600 bg-blue-50 border-blue-200';
      default: return 'text-gray-600 bg-gray-50 border-gray-200';
    }
  };

  const getDemandColor = (level: string) => {
    switch (level.toLowerCase()) {
      case 'high': return '!bg-green-600 !text-white !border-green-600';
      case 'medium': return '!bg-yellow-600 !text-white !border-yellow-600';
      case 'low': return '!bg-blue-600 !text-white !border-blue-600';
      default: return '!bg-gray-600 !text-white !border-gray-600';
    }
  };

  if (loading) {
    return (
      <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
        <div className="flex items-center justify-center h-64">
          <Loader2 className="w-8 h-8 animate-spin" />
          <span className="ml-2">Loading market intelligence...</span>
        </div>
      </div>
    );
  }

  // Show message if no data is available
  const hasData = marketSummary || areaAnalysis.length > 0 || priceBands.length > 0 || seasonalTrends.length > 0;
  
  if (!hasData) {
    return (
      <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
        <div className="flex items-center justify-between">
          <div>
            <h1 className="text-3xl font-bold tracking-tight">Market Analysis</h1>
            <p className="text-muted-foreground">Real-time market intelligence and competitive analysis</p>
          </div>
          <Button onClick={fetchMarketData}>
            <Activity className="mr-2 h-4 w-4" />
            Retry Connection
          </Button>
        </div>
        
        <Card className="p-8 text-center">
          <div className="space-y-4">
            <AlertTriangle className="w-16 h-16 mx-auto text-yellow-500" />
            <h3 className="text-xl font-semibold">Backend Connection Required</h3>
            <p className="text-muted-foreground max-w-md mx-auto">
              Market intelligence data requires the backend server to be running. 
              Please ensure the NAL backend is running on port 8090.
            </p>
            <div className="bg-muted/30 p-4 rounded-lg text-sm text-left max-w-lg mx-auto">
              <p className="font-medium mb-2">To start the backend server:</p>
              <div className="space-y-2">
                <code className="block bg-black text-green-400 p-2 rounded">
                  cd NAL-BE
                </code>
                <code className="block bg-black text-green-400 p-2 rounded">
                  python -m src
                </code>
              </div>
              <p className="text-xs text-muted-foreground mt-2">
                Backend should be running on port 8090
              </p>
            </div>
            <Button onClick={fetchMarketData} className="!bg-blue-600 !hover:bg-blue-700 !text-white border-0 mt-4" style={{backgroundColor: '#2563eb', color: 'white'}}>
              <Activity className="mr-2 h-4 w-4" />
              Test Connection
            </Button>
          </div>
        </Card>
      </div>
    );
  }

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Market Analysis</h1>
          <p className="text-muted-foreground">Seller-focused insights to optimize pricing and selling strategy</p>
        </div>
        <Button onClick={fetchMarketData} className="!bg-blue-600 !hover:bg-blue-700 !text-white border-0" style={{backgroundColor: '#2563eb', color: 'white'}}>
          <Activity className="mr-2 h-4 w-4" />
          Refresh Data
        </Button>
      </div>

      {/* Property Listing Quality Score */}
      {propertyPerformance && competitivePricing && (
        <Card className="bg-gradient-to-br from-pink-50 to-rose-50 border-pink-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Target className="w-5 h-5 text-pink-600" />
              Listing Quality Score
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">How well-optimized is your property listing</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              <div className="text-center">
                <div className="relative inline-flex items-center justify-center w-32 h-32 mb-4">
                  <svg className="w-32 h-32 transform -rotate-90">
                    <circle cx="64" cy="64" r="56" stroke="#e5e7eb" strokeWidth="8" fill="none" />
                    <circle 
                      cx="64" cy="64" r="56" 
                      stroke="#ec4899" 
                      strokeWidth="8" 
                      fill="none"
                      strokeDasharray={`${((propertyPerformance.total_views / 500) * 100 * 3.52)} 352`}
                      strokeLinecap="round"
                    />
                  </svg>
                  <div className="absolute">
                    <p className="text-3xl font-bold text-pink-600">{Math.min(100, Math.round((propertyPerformance.total_views / 500) * 100))}</p>
                    <p className="text-xs text-gray-600">Score</p>
                  </div>
                </div>
                <p className="text-sm font-semibold text-gray-900">Overall Quality</p>
                <p className="text-xs text-gray-600 mt-1">
                  {Math.round((propertyPerformance.total_views / 500) * 100) > 70 ? 'Excellent' :
                   Math.round((propertyPerformance.total_views / 500) * 100) > 40 ? 'Good' : 'Needs Improvement'}
                </p>
              </div>
              
              <div className="col-span-2 space-y-3">
                <div className="flex items-center justify-between p-3 bg-white rounded-lg border">
                  <div className="flex items-center gap-3">
                    <div className={`w-3 h-3 rounded-full ${
                      propertyPerformance.total_views > 100 ? 'bg-green-500' : 'bg-red-500'
                    }`}></div>
                    <span className="text-sm">Visibility & Reach</span>
                  </div>
                  <span className="text-sm font-semibold">{propertyPerformance.total_views > 100 ? 'Good' : 'Low'}</span>
                </div>
                
                <div className="flex items-center justify-between p-3 bg-white rounded-lg border">
                  <div className="flex items-center gap-3">
                    <div className={`w-3 h-3 rounded-full ${
                      competitivePricing.price_difference_percent <= 10 ? 'bg-green-500' : 'bg-red-500'
                    }`}></div>
                    <span className="text-sm">Competitive Pricing</span>
                  </div>
                  <span className="text-sm font-semibold">{competitivePricing.price_difference_percent <= 10 ? 'Optimal' : 'Needs Adjustment'}</span>
                </div>
                
                <div className="flex items-center justify-between p-3 bg-white rounded-lg border">
                  <div className="flex items-center gap-3">
                    <div className={`w-3 h-3 rounded-full ${
                      propertyPerformance.total_inquiries > 5 ? 'bg-green-500' : 'bg-yellow-500'
                    }`}></div>
                    <span className="text-sm">Buyer Interest</span>
                  </div>
                  <span className="text-sm font-semibold">{propertyPerformance.total_inquiries > 5 ? 'High' : 'Moderate'}</span>
                </div>
                
                <div className="flex items-center justify-between p-3 bg-white rounded-lg border">
                  <div className="flex items-center gap-3">
                    <div className={`w-3 h-3 rounded-full ${
                      propertyPerformance.days_listed < propertyPerformance.area_avg_days ? 'bg-green-500' : 'bg-yellow-500'
                    }`}></div>
                    <span className="text-sm">Time on Market</span>
                  </div>
                  <span className="text-sm font-semibold">{propertyPerformance.days_listed < propertyPerformance.area_avg_days ? 'Fresh' : 'Average'}</span>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Property Status Overview */}
      {propertyPerformance && marketSummary && (
        <Card className="bg-gradient-to-br from-slate-50 to-gray-50 border-slate-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Activity className="w-5 h-5 text-slate-600" />
              Property Status & Market Position
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Complete overview of your property's market standing</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div className="bg-white p-4 rounded-lg border">
                <p className="text-xs text-gray-600 mb-1">Property Status</p>
                <Badge className="bg-green-100 text-green-700 border-green-300 text-base px-3 py-1">Active</Badge>
                <p className="text-xs text-gray-600 mt-2">Listed for {propertyPerformance.days_listed} days</p>
              </div>
              <div className="bg-white p-4 rounded-lg border">
                <p className="text-xs text-gray-600 mb-1">Market Position</p>
                <p className="text-2xl font-bold text-slate-700">
                  {propertyPerformance.days_listed < propertyPerformance.area_avg_days ? 'Fresh' :
                   propertyPerformance.days_listed < propertyPerformance.area_avg_days * 1.5 ? 'Active' : 'Stale'}
                </p>
                <p className="text-xs text-gray-600 mt-1">
                  {propertyPerformance.days_listed < propertyPerformance.area_avg_days ? 'Recently listed' :
                   propertyPerformance.days_listed < propertyPerformance.area_avg_days * 1.5 ? 'Normal duration' : 'Needs attention'}
                </p>
              </div>
              <div className="bg-white p-4 rounded-lg border">
                <p className="text-xs text-gray-600 mb-1">Visibility Score</p>
                <p className="text-2xl font-bold text-slate-700">
                  {Math.min(10, Math.round((propertyPerformance.total_views / 100) * 10))}/10
                </p>
                <p className="text-xs text-gray-600 mt-1">{propertyPerformance.total_views} total views</p>
              </div>
              <div className="bg-white p-4 rounded-lg border">
                <p className="text-xs text-gray-600 mb-1">Buyer Interest</p>
                <p className="text-2xl font-bold text-slate-700">
                  {propertyPerformance.total_inquiries > 10 ? 'High' :
                   propertyPerformance.total_inquiries > 5 ? 'Medium' : 'Low'}
                </p>
                <p className="text-xs text-gray-600 mt-1">{propertyPerformance.total_inquiries} inquiries</p>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Seller Property Performance - Enhanced */}
      {propertyPerformance && (
        <Card className="bg-gradient-to-br from-indigo-50 to-purple-50 border-indigo-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Target className="w-5 h-5 text-indigo-600" />
              Your Property Performance
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Track how your property is performing compared to similar listings</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-4">
              <div className="bg-white p-4 rounded-lg border border-indigo-100">
                <p className="text-sm text-gray-600 mb-1">Total Views</p>
                <p className="text-2xl font-bold text-indigo-600">{propertyPerformance.total_views.toLocaleString()}</p>
                <p className={`text-xs mt-1 ${propertyPerformance.views_trend >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                  {propertyPerformance.views_trend >= 0 ? '↑' : '↓'} {Math.abs(propertyPerformance.views_trend)}% vs last week
                </p>
              </div>
              <div className="bg-white p-4 rounded-lg border border-indigo-100">
                <p className="text-sm text-gray-600 mb-1">Inquiries</p>
                <p className="text-2xl font-bold text-indigo-600">{propertyPerformance.total_inquiries}</p>
                <p className={`text-xs mt-1 ${propertyPerformance.inquiries_trend >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                  {propertyPerformance.inquiries_trend >= 0 ? '↑' : '↓'} {Math.abs(propertyPerformance.inquiries_trend)}% vs last week
                </p>
              </div>
              <div className="bg-white p-4 rounded-lg border border-indigo-100">
                <p className="text-sm text-gray-600 mb-1">Days Listed</p>
                <p className="text-2xl font-bold text-indigo-600">{propertyPerformance.days_listed}</p>
                <p className="text-xs text-gray-600 mt-1">Area avg: {propertyPerformance.area_avg_days} days</p>
              </div>
              <div className="bg-white p-4 rounded-lg border border-indigo-100">
                <p className="text-sm text-gray-600 mb-1">Conversion Rate</p>
                <p className="text-2xl font-bold text-indigo-600">
                  {propertyPerformance.total_views > 0 ? ((propertyPerformance.total_inquiries / propertyPerformance.total_views) * 100).toFixed(1) : 0}%
                </p>
                <p className="text-xs text-gray-600 mt-1">Inquiries per view</p>
              </div>
            </div>
            {(() => {
              const conversionRate = propertyPerformance.total_views > 0 ? (propertyPerformance.total_inquiries / propertyPerformance.total_views) * 100 : 0;
              const daysComparison = ((propertyPerformance.days_listed - propertyPerformance.area_avg_days) / propertyPerformance.area_avg_days) * 100;
              let message = '';
              let messageColor = 'bg-blue-50 border-blue-200 text-blue-800';
              
              if (propertyPerformance.days_listed > propertyPerformance.area_avg_days * 1.5) {
                message = `Your property has been listed ${Math.round(daysComparison)}% longer than average. Consider adjusting price or improving listing quality to attract more buyers.`;
                messageColor = 'bg-orange-50 border-orange-200 text-orange-800';
              } else if (conversionRate < 2) {
                message = `Your conversion rate (${conversionRate.toFixed(1)}%) is below average. Enhance your listing with better photos and detailed descriptions to convert views into inquiries.`;
                messageColor = 'bg-yellow-50 border-yellow-200 text-yellow-800';
              } else if (propertyPerformance.views_trend > 10) {
                message = `Great! Your views increased by ${propertyPerformance.views_trend}% this week. Your property is gaining traction in the market.`;
                messageColor = 'bg-green-50 border-green-200 text-green-800';
              } else {
                message = `Your property is performing within normal range. Keep monitoring trends and respond quickly to inquiries.`;
              }
              
              return (
                <div className={`p-3 rounded-lg border ${messageColor} text-sm`}>
                  <p className="font-medium">Performance Insight:</p>
                  <p className="mt-1">{message}</p>
                </div>
              );
            })()}
          </CardContent>
        </Card>
      )}

      {/* How Fast Can It Be Sold */}
      {timeToSell && competitivePricing && (
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          <Card className="bg-gradient-to-br from-teal-50 to-cyan-50 border-teal-200">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Activity className="w-5 h-5 text-teal-600" />
                Selling Speed Prediction
              </CardTitle>
              <p className="text-sm text-muted-foreground mt-1">How fast can your property sell based on market data</p>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="text-center p-6 bg-white rounded-lg border-2 border-teal-200">
                  <p className="text-sm text-gray-600 mb-2">Estimated Time to Sell</p>
                  <p className="text-5xl font-bold text-teal-600 mb-1">{timeToSell.estimated_days_min}-{timeToSell.estimated_days_max}</p>
                  <p className="text-sm text-gray-600">Days at current price</p>
                  <Badge className={`mt-2 ${
                    timeToSell.estimated_days_max < timeToSell.market_avg_days ? 'bg-green-100 text-green-700' :
                    timeToSell.estimated_days_min > timeToSell.market_avg_days ? 'bg-orange-100 text-orange-700' :
                    'bg-blue-100 text-blue-700'
                  }`}>
                    {timeToSell.estimated_days_max < timeToSell.market_avg_days ? 'Faster than average' :
                     timeToSell.estimated_days_min > timeToSell.market_avg_days ? 'Slower than average' :
                     'Average speed'}
                  </Badge>
                </div>
                <div className="bg-white p-4 rounded-lg border">
                  <h4 className="font-semibold text-gray-900 mb-3">Speed Up Your Sale</h4>
                  <div className="space-y-2">
                    <div className="flex items-center justify-between p-2 bg-teal-50 rounded">
                      <span className="text-sm text-gray-700">Reduce price by 5%</span>
                      <span className="font-semibold text-teal-600">{timeToSell.if_reduced_5_percent_min}-{timeToSell.if_reduced_5_percent_max} days</span>
                    </div>
                    <div className="flex items-center justify-between p-2 bg-teal-50 rounded">
                      <span className="text-sm text-gray-700">Reduce price by 10%</span>
                      <span className="font-semibold text-teal-600">{timeToSell.if_reduced_10_percent_min}-{timeToSell.if_reduced_10_percent_max} days</span>
                    </div>
                    <div className="flex items-center justify-between p-2 bg-gray-50 rounded">
                      <span className="text-sm text-gray-700">Market average</span>
                      <span className="font-semibold text-gray-600">{timeToSell.market_avg_days} days</span>
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-violet-50 to-purple-50 border-violet-200">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <BarChart3 className="w-5 h-5 text-violet-600" />
                Property Competitiveness
              </CardTitle>
              <p className="text-sm text-muted-foreground mt-1">How your property compares to similar listings</p>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="text-center p-6 bg-white rounded-lg border-2 border-violet-200">
                  <p className="text-sm text-gray-600 mb-2">Competitiveness Score</p>
                  <p className="text-5xl font-bold text-violet-600 mb-1">
                    {(() => {
                      const priceDiff = Math.abs(competitivePricing.price_difference_percent);
                      const score = Math.max(0, Math.min(10, 10 - (priceDiff / 5)));
                      return score.toFixed(1);
                    })()}
                  </p>
                  <p className="text-sm text-gray-600">Out of 10</p>
                  <Badge className={`mt-2 ${
                    competitivePricing.price_difference_percent <= 5 ? 'bg-green-100 text-green-700' :
                    competitivePricing.price_difference_percent <= 15 ? 'bg-yellow-100 text-yellow-700' :
                    'bg-red-100 text-red-700'
                  }`}>
                    {competitivePricing.price_difference_percent <= 5 ? 'Highly Competitive' :
                     competitivePricing.price_difference_percent <= 15 ? 'Moderately Competitive' :
                     'Low Competitiveness'}
                  </Badge>
                </div>
                <div className="bg-white p-4 rounded-lg border">
                  <h4 className="font-semibold text-gray-900 mb-3">Market Position</h4>
                  <div className="space-y-3">
                    <div>
                      <div className="flex justify-between text-sm mb-1">
                        <span className="text-gray-600">Similar Properties</span>
                        <span className="font-semibold">{competitivePricing.similar_properties_count} active</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div className="bg-violet-600 h-2 rounded-full" style={{width: `${Math.min(100, (competitivePricing.similar_properties_count / 20) * 100)}%`}}></div>
                      </div>
                    </div>
                    <div>
                      <div className="flex justify-between text-sm mb-1">
                        <span className="text-gray-600">Price vs Market</span>
                        <span className={`font-semibold ${
                          competitivePricing.price_difference_percent > 0 ? 'text-red-600' : 'text-green-600'
                        }`}>
                          {competitivePricing.price_difference_percent > 0 ? '+' : ''}{competitivePricing.price_difference_percent}%
                        </span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div className={`h-2 rounded-full ${
                          competitivePricing.price_difference_percent > 0 ? 'bg-red-500' : 'bg-green-500'
                        }`} style={{width: `${Math.min(100, Math.abs(competitivePricing.price_difference_percent) * 2)}%`}}></div>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Competitive Pricing Analysis - Enhanced */}
      {competitivePricing && (
        <Card className="bg-gradient-to-br from-amber-50 to-orange-50 border-amber-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <TrendingUp className="w-5 h-5 text-amber-600" />
              Price Comparison & Market Positioning
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">See how your price compares to similar properties</p>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="bg-white p-5 rounded-lg border-l-4 border-amber-500">
                <div className="flex items-start justify-between mb-4">
                  <div className="flex-1">
                    <h4 className="font-semibold text-gray-900 text-lg mb-1">{competitivePricing.property_title}</h4>
                    <p className="text-sm text-gray-600">{competitivePricing.property_locality}</p>
                  </div>
                  <Badge className={`text-base px-4 py-1 ${
                    competitivePricing.price_difference_percent > 10 ? 'bg-red-100 text-red-700 border-red-300' :
                    competitivePricing.price_difference_percent > 0 ? 'bg-orange-100 text-orange-700 border-orange-300' :
                    'bg-green-100 text-green-700 border-green-300'
                  }`}>
                    {competitivePricing.price_difference_percent > 0 ? 
                      `${competitivePricing.price_difference_percent}% Higher` : 
                      `${Math.abs(competitivePricing.price_difference_percent)}% Lower`}
                  </Badge>
                </div>
                
                <div className="grid grid-cols-2 gap-6 mb-4">
                  <div className="text-center p-4 bg-amber-50 rounded-lg">
                    <p className="text-sm text-gray-600 mb-1">Your Price</p>
                    <p className="text-3xl font-bold text-gray-900">₹{(competitivePricing.your_price / 10000000).toFixed(2)} Cr</p>
                    <p className="text-xs text-gray-600 mt-1">₹{competitivePricing.your_price_per_sqft.toLocaleString()}/sq.ft</p>
                  </div>
                  <div className="text-center p-4 bg-green-50 rounded-lg">
                    <p className="text-sm text-gray-600 mb-1">Market Average</p>
                    <p className="text-3xl font-bold text-gray-900">₹{(competitivePricing.market_avg_price / 10000000).toFixed(2)} Cr</p>
                    <p className="text-xs text-gray-600 mt-1">₹{competitivePricing.market_price_per_sqft.toLocaleString()}/sq.ft</p>
                  </div>
                </div>

                <div className="bg-green-50 p-4 rounded-lg border border-green-200 mb-4">
                  <p className="text-sm font-semibold text-green-900 mb-2">Optimal Price Range</p>
                  <p className="text-2xl font-bold text-green-700">
                    ₹{(competitivePricing.optimal_price_min / 10000000).toFixed(2)} - {(competitivePricing.optimal_price_max / 10000000).toFixed(2)} Cr
                  </p>
                  <p className="text-xs text-green-800 mt-1">Recommended pricing to attract buyers</p>
                </div>

                <div className="bg-amber-50 rounded-lg border border-amber-200 p-4">
                  <div className="flex items-start gap-3">
                    <AlertTriangle className="w-5 h-5 text-amber-600 flex-shrink-0 mt-0.5" />
                    <div className="flex-1">
                      <p className="font-semibold text-amber-900 text-sm mb-1">Pricing Recommendation</p>
                      <p className="text-sm text-amber-800">{competitivePricing.recommendation}</p>
                      {competitivePricing.price_difference_percent > 5 && (
                        <p className="text-xs text-amber-700 mt-2">
                          💡 Tip: Properties priced within 5% of market average sell {Math.round(15 + (competitivePricing.price_difference_percent * 2))} days faster.
                        </p>
                      )}
                    </div>
                  </div>
                </div>
              </div>
              
              <div className="grid grid-cols-3 gap-4">
                <div className="text-center p-4 bg-white rounded-lg border">
                  <p className="text-xs text-gray-600 mb-2">Similar Properties</p>
                  <p className="text-3xl font-bold text-gray-900">{competitivePricing.similar_properties_count}</p>
                  <p className="text-xs text-gray-600 mt-1">Active in your area</p>
                  <Badge className={`mt-2 ${
                    competitivePricing.similar_properties_count > 15 ? 'bg-red-100 text-red-700' :
                    competitivePricing.similar_properties_count > 8 ? 'bg-yellow-100 text-yellow-700' :
                    'bg-green-100 text-green-700'
                  }`}>
                    {competitivePricing.similar_properties_count > 15 ? 'High Competition' :
                     competitivePricing.similar_properties_count > 8 ? 'Moderate' : 'Low Competition'}
                  </Badge>
                </div>
                <div className="text-center p-4 bg-white rounded-lg border">
                  <p className="text-xs text-gray-600 mb-2">Price Status</p>
                  <p className="text-xl font-bold text-gray-900">
                    {competitivePricing.price_difference_percent > 10 ? 'Overpriced' :
                     competitivePricing.price_difference_percent > 5 ? 'Slightly High' :
                     competitivePricing.price_difference_percent > -5 ? 'Fair Price' :
                     'Underpriced'}
                  </p>
                  <p className="text-xs text-gray-600 mt-1">Market comparison</p>
                </div>
                <div className="text-center p-4 bg-white rounded-lg border">
                  <p className="text-xs text-gray-600 mb-2">Competitiveness</p>
                  <p className="text-3xl font-bold text-gray-900">
                    {(() => {
                      const score = Math.max(0, Math.min(10, 10 - (Math.abs(competitivePricing.price_difference_percent) / 5)));
                      return score.toFixed(1);
                    })()}
                  </p>
                  <p className="text-xs text-gray-600 mt-1">Out of 10</p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Price Trends & Forecast */}
      {seasonalTrends.length > 0 && competitivePricing && (
        <Card className="bg-gradient-to-br from-indigo-50 to-blue-50 border-indigo-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <TrendingUp className="w-5 h-5 text-indigo-600" />
              Price Trends & Forecast
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Historical trends and future price predictions</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <div className="bg-white p-5 rounded-lg border">
                <h4 className="font-semibold text-gray-900 mb-4">Recent Price Movement</h4>
                <div className="space-y-3">
                  {seasonalTrends.slice(0, 4).map((trend, idx) => (
                    <div key={idx} className="flex items-center justify-between p-3 bg-indigo-50 rounded">
                      <div>
                        <p className="text-sm font-medium text-gray-900">{trend.month}</p>
                        <p className="text-xs text-gray-600">{trend.sold_count} sales</p>
                      </div>
                      <div className="text-right">
                        <p className="text-sm font-bold text-indigo-600">{trend.avg_price_formatted}</p>
                        <Badge className={`text-xs ${
                          trend.demand_level.toLowerCase() === 'high' ? 'bg-green-100 text-green-700' :
                          trend.demand_level.toLowerCase() === 'medium' ? 'bg-yellow-100 text-yellow-700' :
                          'bg-blue-100 text-blue-700'
                        }`}>
                          {trend.demand_level}
                        </Badge>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
              
              <div className="bg-white p-5 rounded-lg border">
                <h4 className="font-semibold text-gray-900 mb-4">Price Forecast</h4>
                <div className="space-y-4">
                  <div className="p-4 bg-green-50 rounded-lg border border-green-200">
                    <div className="flex items-center justify-between mb-2">
                      <p className="text-sm font-medium text-gray-900">If Sold This Month</p>
                      <TrendingUp className="w-4 h-4 text-green-600" />
                    </div>
                    <p className="text-2xl font-bold text-green-700">
                      ₹{(competitivePricing.your_price / 10000000).toFixed(2)} Cr
                    </p>
                    <p className="text-xs text-green-600 mt-1">Current market rate</p>
                  </div>
                  
                  <div className="p-4 bg-yellow-50 rounded-lg border border-yellow-200">
                    <div className="flex items-center justify-between mb-2">
                      <p className="text-sm font-medium text-gray-900">3 Months Later</p>
                      <Activity className="w-4 h-4 text-yellow-600" />
                    </div>
                    <p className="text-2xl font-bold text-yellow-700">
                      ₹{((competitivePricing.your_price * 0.98) / 10000000).toFixed(2)} Cr
                    </p>
                    <p className="text-xs text-yellow-600 mt-1">Estimated 2% depreciation</p>
                  </div>
                  
                  <div className="p-4 bg-red-50 rounded-lg border border-red-200">
                    <div className="flex items-center justify-between mb-2">
                      <p className="text-sm font-medium text-gray-900">6 Months Later</p>
                      <TrendingDown className="w-4 h-4 text-red-600" />
                    </div>
                    <p className="text-2xl font-bold text-red-700">
                      ₹{((competitivePricing.your_price * 0.95) / 10000000).toFixed(2)} Cr
                    </p>
                    <p className="text-xs text-red-600 mt-1">Estimated 5% depreciation</p>
                  </div>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Market Demand Analysis - Enhanced */}
      {demandAnalysis && (
        <Card className="bg-gradient-to-br from-blue-50 to-cyan-50 border-blue-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Eye className="w-5 h-5 text-blue-600" />
              Market Demand in Your Area
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Current buyer interest and demand trends</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div className="text-center p-6 bg-white rounded-lg border-2 border-blue-200">
                <p className="text-sm text-gray-600 mb-2">Demand Level</p>
                <Badge className={`text-xl px-6 py-2 ${
                  demandAnalysis.demand_level.toLowerCase() === 'high' ? 'bg-green-100 text-green-700 border-green-300' :
                  demandAnalysis.demand_level.toLowerCase() === 'medium' ? 'bg-yellow-100 text-yellow-700 border-yellow-300' :
                  'bg-blue-100 text-blue-700 border-blue-300'
                }`}>
                  {demandAnalysis.demand_level}
                </Badge>
                <p className={`text-sm mt-2 font-medium ${
                  demandAnalysis.demand_trend === 'increasing' ? 'text-green-600' :
                  demandAnalysis.demand_trend === 'decreasing' ? 'text-red-600' :
                  'text-gray-600'
                }`}>
                  {demandAnalysis.demand_trend === 'increasing' ? '↑ Increasing' :
                   demandAnalysis.demand_trend === 'decreasing' ? '↓ Decreasing' :
                   '→ Stable'}
                </p>
              </div>
              <div className="p-6 bg-white rounded-lg border text-center">
                <p className="text-sm text-gray-600 mb-1">Weekly Searches</p>
                <p className="text-4xl font-bold text-blue-600">{demandAnalysis.searches_per_week}</p>
                <p className="text-xs text-gray-600 mt-1">Buyers looking in your area</p>
              </div>
              <div className="p-6 bg-white rounded-lg border text-center">
                <p className="text-sm text-gray-600 mb-1">Demand Score</p>
                <p className="text-4xl font-bold text-blue-600">{demandAnalysis.demand_score}<span className="text-xl text-gray-400">/10</span></p>
                <div className="w-full bg-gray-200 rounded-full h-2 mt-2">
                  <div className="bg-blue-600 h-2 rounded-full" style={{width: `${demandAnalysis.demand_score * 10}%`}}></div>
                </div>
              </div>
            </div>
            <div className="mt-4 p-4 bg-blue-50 rounded-lg border border-blue-200">
              <div className="flex items-start gap-2">
                <Eye className="w-5 h-5 text-blue-600 mt-0.5 flex-shrink-0" />
                <div>
                  <p className="font-semibold text-blue-900 mb-1">Market Insight</p>
                  <p className="text-sm text-blue-800">{demandAnalysis.insight_message}</p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Time-to-Sell Analysis */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {timeToSell && (
          <Card className="bg-gradient-to-br from-green-50 to-emerald-50 border-green-200">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Activity className="w-5 h-5 text-green-600" />
                Time-to-Sell Prediction
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="text-center p-6 bg-white rounded-lg border-2 border-green-200">
                  <p className="text-sm text-gray-600 mb-2">Estimated Time to Sell</p>
                  <p className="text-4xl font-bold text-green-600 mb-1">{timeToSell.estimated_days_min}-{timeToSell.estimated_days_max}</p>
                  <p className="text-sm text-gray-600">Days (at current price)</p>
                </div>
                <div className="space-y-2">
                  <div className="flex items-center justify-between p-3 bg-white rounded border">
                    <span className="text-sm text-gray-700">If reduced by 5%</span>
                    <span className="font-semibold text-green-600">{timeToSell.if_reduced_5_percent_min}-{timeToSell.if_reduced_5_percent_max} days</span>
                  </div>
                  <div className="flex items-center justify-between p-3 bg-white rounded border">
                    <span className="text-sm text-gray-700">If reduced by 10%</span>
                    <span className="font-semibold text-green-600">{timeToSell.if_reduced_10_percent_min}-{timeToSell.if_reduced_10_percent_max} days</span>
                  </div>
                  <div className="flex items-center justify-between p-3 bg-white rounded border">
                    <span className="text-sm text-gray-700">Market average</span>
                    <span className="font-semibold text-gray-600">{timeToSell.market_avg_days} days</span>
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>
        )}

        {demandAnalysis && (
          <Card className="bg-gradient-to-br from-blue-50 to-cyan-50 border-blue-200">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Eye className="w-5 h-5 text-blue-600" />
                Market Demand Analysis
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="text-center p-6 bg-white rounded-lg border-2 border-blue-200">
                  <p className="text-sm text-gray-600 mb-2">Buyer Demand in Your Area</p>
                  <Badge className={`text-lg px-4 py-2 ${
                    demandAnalysis.demand_level.toLowerCase() === 'high' ? 'bg-green-100 text-green-700 border-green-300' :
                    demandAnalysis.demand_level.toLowerCase() === 'medium' ? 'bg-yellow-100 text-yellow-700 border-yellow-300' :
                    'bg-blue-100 text-blue-700 border-blue-300'
                  }`}>
                    {demandAnalysis.demand_level} Demand
                  </Badge>
                  <p className="text-xs text-gray-600 mt-2">{demandAnalysis.demand_trend === 'increasing' ? '↑' : demandAnalysis.demand_trend === 'decreasing' ? '↓' : '→'} {demandAnalysis.demand_trend} trend</p>
                </div>
                <div className="grid grid-cols-2 gap-3">
                  <div className="p-3 bg-white rounded border text-center">
                    <p className="text-xs text-gray-600 mb-1">Searches/Week</p>
                    <p className="text-xl font-bold text-blue-600">{demandAnalysis.searches_per_week}</p>
                  </div>
                  <div className="p-3 bg-white rounded border text-center">
                    <p className="text-xs text-gray-600 mb-1">Demand Score</p>
                    <p className="text-xl font-bold text-blue-600">{demandAnalysis.demand_score}/10</p>
                  </div>
                </div>
                <div className="p-3 bg-blue-50 rounded-lg border border-blue-200">
                  <p className="text-xs font-semibold text-blue-900 mb-1">Market Insight</p>
                  <p className="text-xs text-blue-800">{demandAnalysis.insight_message}</p>
                </div>
              </div>
            </CardContent>
          </Card>
        )}
      </div>

      {/* Market Condition & Supply-Demand */}
      {areaAnalysis.length > 0 && competitivePricing && (
        <Card className="bg-gradient-to-br from-emerald-50 to-teal-50 border-emerald-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <MapPin className="w-5 h-5 text-emerald-600" />
              Market Condition in Your Area
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Understand demand-supply dynamics and market temperature</p>
          </CardHeader>
          <CardContent>
            {(() => {
              const yourArea = areaAnalysis.find(a => a.locality === competitivePricing.property_locality) || areaAnalysis[0];
              const marketTemp = yourArea.market_hotness.toLowerCase();
              const demandSupplyRatio = yourArea.demand_vs_supply_ratio;
              
              return (
                <div className="space-y-4">
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div className={`p-6 rounded-lg border-2 text-center ${
                      marketTemp === 'hot' ? 'bg-red-50 border-red-300' :
                      marketTemp === 'warm' ? 'bg-orange-50 border-orange-300' :
                      'bg-blue-50 border-blue-300'
                    }`}>
                      <p className="text-sm text-gray-600 mb-2">Market Temperature</p>
                      <Badge className={`text-2xl px-6 py-2 ${
                        marketTemp === 'hot' ? 'bg-red-100 text-red-700 border-red-300' :
                        marketTemp === 'warm' ? 'bg-orange-100 text-orange-700 border-orange-300' :
                        'bg-blue-100 text-blue-700 border-blue-300'
                      }`}>
                        {yourArea.market_hotness}
                      </Badge>
                      <p className="text-xs text-gray-600 mt-2">
                        {marketTemp === 'hot' ? 'High buyer demand' :
                         marketTemp === 'warm' ? 'Moderate activity' :
                         'Lower demand'}
                      </p>
                    </div>
                    
                    <div className="p-6 bg-white rounded-lg border text-center">
                      <p className="text-sm text-gray-600 mb-2">Demand-Supply Ratio</p>
                      <p className="text-4xl font-bold text-emerald-600">{demandSupplyRatio}</p>
                      <p className="text-xs text-gray-600 mt-2">
                        {demandSupplyRatio > 1.5 ? 'High demand, low supply' :
                         demandSupplyRatio > 0.8 ? 'Balanced market' :
                         'High supply, lower demand'}
                      </p>
                    </div>
                    
                    <div className="p-6 bg-white rounded-lg border text-center">
                      <p className="text-sm text-gray-600 mb-2">Active Competition</p>
                      <p className="text-4xl font-bold text-emerald-600">{yourArea.active_supply}</p>
                      <p className="text-xs text-gray-600 mt-2">Similar properties listed</p>
                    </div>
                  </div>

                  <div className="bg-white p-5 rounded-lg border">
                    <h4 className="font-semibold text-gray-900 mb-3">What This Means for You</h4>
                    <div className="space-y-2">
                      {marketTemp === 'hot' && (
                        <div className="flex items-start gap-2 p-3 bg-green-50 rounded">
                          <span className="text-green-600 text-xl">✅</span>
                          <p className="text-sm text-gray-700">
                            <strong>Great time to sell!</strong> High buyer demand means you can maintain your price and expect faster sales. Properties in hot markets sell {Math.round(yourArea.avg_days_on_market)} days on average.
                          </p>
                        </div>
                      )}
                      {marketTemp === 'warm' && (
                        <div className="flex items-start gap-2 p-3 bg-yellow-50 rounded">
                          <span className="text-yellow-600 text-xl">⚠️</span>
                          <p className="text-sm text-gray-700">
                            <strong>Moderate market.</strong> Competitive pricing and good listing quality are important. Average selling time is {Math.round(yourArea.avg_days_on_market)} days.
                          </p>
                        </div>
                      )}
                      {marketTemp === 'cool' && (
                        <div className="flex items-start gap-2 p-3 bg-blue-50 rounded">
                          <span className="text-blue-600 text-xl">💡</span>
                          <p className="text-sm text-gray-700">
                            <strong>Buyer's market.</strong> Consider competitive pricing and enhanced listing quality. Properties take around {Math.round(yourArea.avg_days_on_market)} days to sell. Patience and flexibility are key.
                          </p>
                        </div>
                      )}
                      
                      {demandSupplyRatio > 1.5 && (
                        <div className="flex items-start gap-2 p-3 bg-emerald-50 rounded">
                          <span className="text-emerald-600 text-xl">📈</span>
                          <p className="text-sm text-gray-700">
                            <strong>Seller's advantage:</strong> With {yourArea.active_supply} properties competing for {Math.round(yourArea.total_views / yourArea.active_supply)} average views each, demand exceeds supply. You have pricing power.
                          </p>
                        </div>
                      )}
                      
                      {demandSupplyRatio < 0.8 && yourArea.active_supply > 10 && (
                        <div className="flex items-start gap-2 p-3 bg-orange-50 rounded">
                          <span className="text-orange-600 text-xl">⚠️</span>
                          <p className="text-sm text-gray-700">
                            <strong>High competition:</strong> With {yourArea.active_supply} similar properties, standing out is crucial. Focus on competitive pricing, professional photos, and quick responses to inquiries.
                          </p>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              );
            })()}
          </CardContent>
        </Card>
      )}

      {/* Market Conditions for Optimal Selling */}
      {seasonalTrends.length > 0 && (
        <Card className="bg-gradient-to-br from-blue-50 to-cyan-50 border-blue-200">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Activity className="w-5 h-5 text-blue-600" />
              Market Conditions for Optimal Selling
            </CardTitle>
            <p className="text-sm text-muted-foreground mt-1">Key factors that influence the best time to sell</p>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="bg-white p-4 rounded-lg border">
                <div className="flex items-center gap-2 mb-3">
                  <div className="p-2 bg-green-100 rounded-lg">
                    <TrendingUp className="w-5 h-5 text-green-600" />
                  </div>
                  <h4 className="font-semibold text-gray-900">Strong Market Conditions</h4>
                </div>
                <p className="text-sm text-gray-700 mb-2">When property prices are rising and buyer demand is high</p>
                <div className="bg-green-50 p-3 rounded text-xs space-y-1">
                  <p className="text-gray-700">• Better prices and faster deals</p>
                  <p className="text-gray-700">• Multiple buyer interest</p>
                  <p className="text-gray-700">• Stronger negotiation position</p>
                </div>
              </div>

              <div className="bg-white p-4 rounded-lg border">
                <div className="flex items-center gap-2 mb-3">
                  <div className="p-2 bg-orange-100 rounded-lg">
                    <Zap className="w-5 h-5 text-orange-600" />
                  </div>
                  <h4 className="font-semibold text-gray-900">Festival & Financial Year</h4>
                </div>
                <p className="text-sm text-gray-700 mb-2">Peak buying periods in India</p>
                <div className="bg-orange-50 p-3 rounded text-xs space-y-1">
                  <p className="text-gray-700">• Jan-Mar: Financial year planning</p>
                  <p className="text-gray-700">• Sep-Dec: Festival season (Diwali)</p>
                  <p className="text-gray-700">• Increased buyer sentiment</p>
                </div>
              </div>

              <div className="bg-white p-4 rounded-lg border">
                <div className="flex items-center gap-2 mb-3">
                  <div className="p-2 bg-purple-100 rounded-lg">
                    <MapPin className="w-5 h-5 text-purple-600" />
                  </div>
                  <h4 className="font-semibold text-gray-900">Low Inventory Period</h4>
                </div>
                <p className="text-sm text-gray-700 mb-2">When fewer properties are available</p>
                <div className="bg-purple-50 p-3 rounded text-xs space-y-1">
                  <p className="text-gray-700">• More bargaining power</p>
                  <p className="text-gray-700">• Less competition from sellers</p>
                  <p className="text-gray-700">• Premium pricing possible</p>
                </div>
              </div>

              <div className="bg-white p-4 rounded-lg border">
                <div className="flex items-center gap-2 mb-3">
                  <div className="p-2 bg-blue-100 rounded-lg">
                    <MapPin className="w-5 h-5 text-blue-600" />
                  </div>
                  <h4 className="font-semibold text-gray-900">Infrastructure Activity</h4>
                </div>
                <p className="text-sm text-gray-700 mb-2">Major development announcements</p>
                <div className="bg-blue-50 p-3 rounded text-xs space-y-1">
                  <p className="text-gray-700">• New metro lines or highways</p>
                  <p className="text-gray-700">• IT parks and business hubs</p>
                  <p className="text-gray-700">• Increased property demand</p>
                </div>
              </div>
            </div>

            <div className="mt-4 bg-white p-4 rounded-lg border">
              <div className="flex items-center gap-2 mb-3">
                <div className="p-2 bg-teal-100 rounded-lg">
                  <Users className="w-5 h-5 text-teal-600" />
                </div>
                <h4 className="font-semibold text-gray-900">High Buyer Activity Indicators</h4>
              </div>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
                <div className="bg-teal-50 p-3 rounded">
                  <p className="text-xs font-semibold text-teal-900 mb-1">Low Interest Rates</p>
                  <p className="text-xs text-gray-700">Home loan rates are favorable for buyers</p>
                </div>
                <div className="bg-teal-50 p-3 rounded">
                  <p className="text-xs font-semibold text-teal-900 mb-1">Economic Stability</p>
                  <p className="text-xs text-gray-700">Stable economic conditions boost confidence</p>
                </div>
                <div className="bg-teal-50 p-3 rounded">
                  <p className="text-xs font-semibold text-teal-900 mb-1">Job Market Growth</p>
                  <p className="text-xs text-gray-700">IT hiring and business expansion</p>
                </div>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Seller Action Recommendations - Dynamic */}
      {(() => {
        const recommendations = [];
        let recIndex = 1;

        // Pricing recommendation
        if (competitivePricing && competitivePricing.price_difference_percent > 5) {
          const reduction = Math.round(competitivePricing.price_difference_percent * 0.7);
          const newPriceMin = (competitivePricing.optimal_price_min / 10000000).toFixed(2);
          const newPriceMax = (competitivePricing.optimal_price_max / 10000000).toFixed(2);
          const timeSaved = timeToSell ? Math.round((timeToSell.estimated_days_max - timeToSell.if_reduced_5_percent_max) * (reduction / 5)) : 15;
          recommendations.push({
            color: 'rose',
            title: 'Adjust Pricing Strategy',
            text: `Reduce price by ${reduction}% to ₹${newPriceMin}-${newPriceMax} Cr to match market average and attract more buyers. This could reduce selling time by ${timeSaved} days.`
          });
        }

        // Listing quality recommendation
        if (propertyPerformance && propertyPerformance.total_views < 100) {
          const inquiryRate = propertyPerformance.total_inquiries > 0 ? ((propertyPerformance.total_inquiries / propertyPerformance.total_views) * 100).toFixed(0) : 0;
          recommendations.push({
            color: 'amber',
            title: 'Improve Listing Quality',
            text: `Add professional photos, virtual tour, and detailed amenities. Your current inquiry rate is ${inquiryRate}%. Complete listings typically get 40% more inquiries.`
          });
        }

        // Seasonal promotion recommendation
        if (seasonalTrends.length > 0) {
          const currentMonth = new Date().toLocaleString('default', { month: 'long' });
          const currentTrend = seasonalTrends.find(t => t.month.includes(currentMonth)) || seasonalTrends[0];
          if (currentTrend.demand_level.toLowerCase() === 'high') {
            recommendations.push({
              color: 'green',
              title: 'Promote During Peak Season',
              text: `Current month (${currentMonth}) shows ${currentTrend.demand_level.toLowerCase()} demand with ${currentTrend.sold_count} recent sales. Consider premium promotion to maximize visibility and leverage high buyer demand.`
            });
          }
        }

        // Response time recommendation
        if (propertyPerformance && propertyPerformance.total_inquiries > 0) {
          const conversionRate = demandAnalysis ? Math.round(demandAnalysis.demand_score * 6) : 60;
          recommendations.push({
            color: 'blue',
            title: 'Respond Quickly to Inquiries',
            text: `You have ${propertyPerformance.total_inquiries} inquiries. Fast response time (within 2 hours) increases conversion by ${conversionRate}%. Enable instant notifications for new inquiries.`
          });
        }

        // Demand-based recommendation
        if (demandAnalysis && demandAnalysis.demand_level.toLowerCase() === 'low') {
          recommendations.push({
            color: 'purple',
            title: 'Consider Market Timing',
            text: `Current demand in your area is ${demandAnalysis.demand_level.toLowerCase()} with ${demandAnalysis.searches_per_week} searches/week. ${demandAnalysis.insight_message}`
          });
        }

        if (recommendations.length === 0) {
          recommendations.push({
            color: 'green',
            title: 'Your Property is Well Positioned',
            text: 'Your pricing and listing quality are competitive. Continue monitoring market trends and responding promptly to inquiries.'
          });
        }

        const colorMap = {
          rose: { bg: 'bg-rose-100', text: 'text-rose-600', border: 'border-rose-500' },
          amber: { bg: 'bg-amber-100', text: 'text-amber-600', border: 'border-amber-500' },
          green: { bg: 'bg-green-100', text: 'text-green-600', border: 'border-green-500' },
          blue: { bg: 'bg-blue-100', text: 'text-blue-600', border: 'border-blue-500' },
          purple: { bg: 'bg-purple-100', text: 'text-purple-600', border: 'border-purple-500' }
        };

        return (
          <Card className="bg-gradient-to-br from-rose-50 to-red-50 border-rose-200">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <AlertTriangle className="w-5 h-5 text-rose-600" />
                Recommended Actions
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {recommendations.map((rec, idx) => {
                  const colors = colorMap[rec.color];
                  return (
                    <div key={idx} className={`flex items-start gap-3 p-4 bg-white rounded-lg border-l-4 ${colors.border}`}>
                      <div className={`w-8 h-8 rounded-full ${colors.bg} flex items-center justify-center flex-shrink-0`}>
                        <span className={`${colors.text} font-bold`}>{idx + 1}</span>
                      </div>
                      <div className="flex-1">
                        <h4 className="font-semibold text-gray-900 mb-1">{rec.title}</h4>
                        <p className="text-sm text-gray-700">{rec.text}</p>
                      </div>
                    </div>
                  );
                })}
              </div>
            </CardContent>
          </Card>
        );
      })()}

      {/* Market Overview */}
      {marketSummary && (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-blue-700 font-medium">Market Avg Price</p>
                  <p className="text-3xl font-bold text-blue-800">
                    ₹{(marketSummary.market_overview.market_avg_price / 100000).toFixed(1)}L
                  </p>
                  <div className="flex items-center mt-2">
                    {marketSummary.market_overview.price_trend === 'up' ? (
                      <TrendingUp className="w-4 h-4 text-green-600 mr-1" />
                    ) : (
                      <TrendingDown className="w-4 h-4 text-red-600 mr-1" />
                    )}
                    <span className={`text-sm font-medium ${
                      marketSummary.market_overview.price_trend === 'up' ? 'text-green-600' : 'text-red-600'
                    }`}>
                      {marketSummary.market_overview.price_trend === 'up' ? 'Rising' : 'Declining'}
                    </span>
                  </div>
                </div>
                <Target className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-green-700 font-medium">Avg Days on Market</p>
                  <p className="text-3xl font-bold text-green-800">
                    {Math.round(marketSummary.seller_summary.avg_days_on_market)}
                  </p>
                  <p className="text-sm text-green-600 mt-2">Days to sell</p>
                </div>
                <Activity className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-purple-700 font-medium">Active Properties</p>
                  <p className="text-3xl font-bold text-purple-800">
                    {marketSummary.market_overview.total_active_properties}
                  </p>
                  <p className="text-sm text-purple-600 mt-2">Market supply</p>
                </div>
                <Zap className="w-8 h-8 text-purple-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-orange-700 font-medium">Recent Sales</p>
                  <p className="text-3xl font-bold text-orange-800">
                    {marketSummary.market_overview.properties_sold_30d}
                  </p>
                  <p className="text-sm text-orange-600 mt-2">Last 30 days</p>
                </div>
                <Eye className="w-8 h-8 text-orange-600" />
              </div>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Area Analysis */}
      {areaAnalysis.length > 0 && (
        <Card>
          <CardHeader>
            <CardTitle>Market Trends by Area</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              {areaAnalysis.slice(0, 6).map((area, index) => (
                <div key={index} className="flex items-center justify-between p-4 bg-muted/30 rounded-lg">
                  <div className="flex items-center gap-4">
                    <MapPin className="w-5 h-5 text-muted-foreground" />
                    <div>
                      <h4 className="font-semibold">{area.locality}</h4>
                      <p className="text-sm text-muted-foreground">{area.active_supply} active listings</p>
                    </div>
                  </div>
                  <div className="text-center">
                    <p className="font-bold text-lg">{area.avg_market_price_formatted}</p>
                    <p className="text-sm text-muted-foreground">{area.avg_days_on_market} days avg</p>
                  </div>
                  <div className="text-center">
                    <p className="text-sm text-muted-foreground">Demand Ratio</p>
                    <p className="font-bold">{area.demand_vs_supply_ratio}</p>
                  </div>
                  <Badge className={getHotnessColor(area.market_hotness)}>
                    {area.market_hotness}
                  </Badge>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Price Bands & Seasonal Trends */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {priceBands.length > 0 && (
          <Card>
            <CardHeader>
              <CardTitle>Price Band Analysis</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {priceBands.map((band, index) => (
                  <div key={index} className="p-4 border rounded-lg">
                    <div className="flex items-center justify-between mb-2">
                      <h4 className="font-semibold">{band.price_range}</h4>
                      <Badge variant="outline">
                        {band.success_rate}% success
                      </Badge>
                    </div>
                    <div className="grid grid-cols-3 gap-4 text-sm">
                      <div>
                        <p className="text-muted-foreground">Avg Days</p>
                        <p className="font-medium">{band.avg_days_to_sell}</p>
                      </div>
                      <div>
                        <p className="text-muted-foreground">Properties</p>
                        <p className="font-medium">{band.total_properties}</p>
                      </div>
                      <div>
                        <p className="text-muted-foreground">Sold</p>
                        <p className="font-medium">{band.sold_properties}</p>
                      </div>
                    </div>
                    <div className="bg-muted/30 p-2 rounded text-xs mt-2">
                      <AlertTriangle className="w-3 h-3 inline mr-1" />
                      {band.recommendation}
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        )}

        {seasonalTrends.length > 0 && (
          <Card>
            <CardHeader>
              <CardTitle>Seasonal Market Trends</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {seasonalTrends.map((trend, index) => (
                  <div key={index} className="p-4 border rounded-lg">
                    <div className="flex items-center justify-between mb-2">
                      <h4 className="font-semibold">{trend.month}</h4>
                      <Badge 
                        className={getDemandColor(trend.demand_level)}
                        style={trend.demand_level.toLowerCase() === 'low' ? {backgroundColor: '#2563eb', color: 'white', borderColor: '#2563eb'} : {}}
                      >
                        {trend.demand_level} demand
                      </Badge>
                    </div>
                    <div className="grid grid-cols-3 gap-4 text-sm">
                      <div>
                        <p className="text-muted-foreground">Avg Price</p>
                        <p className="font-medium">{trend.avg_price_formatted}</p>
                      </div>
                      <div>
                        <p className="text-muted-foreground">Activity</p>
                        <p className="font-medium">{trend.activity_count}</p>
                      </div>
                      <div>
                        <p className="text-muted-foreground">Days</p>
                        <p className="font-medium">{trend.avg_days_on_market}</p>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  );
}