import React, { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../components/ui/tabs";
import { Input } from "../components/ui/input";
import { Label } from "../components/ui/label";
import { Textarea } from "../components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../components/ui/select";
import { useToast } from "../hooks/use-toast";
import { 
  TrendingUp, 
  Target, 
  DollarSign, 
  Eye, 
  Users, 
  BarChart3,
  Plus,
  Play,
  Pause,
  Edit,
  Trash2,
  Share2,
  Star,
  Calendar,
  MapPin
} from "lucide-react";

const mockCampaigns = [
  {
    id: "camp-1",
    name: "Luxury Villa Promotion",
    type: "social_media",
    status: "active",
    budget: 2500,
    spent: 1850,
    impressions: 45000,
    clicks: 1200,
    leads: 45,
    startDate: new Date("2024-12-01"),
    endDate: new Date("2024-12-31"),
    properties: ["prop-1", "prop-2"]
  },
  {
    id: "camp-2", 
    name: "Holiday Season Sale",
    type: "google_ads",
    status: "paused",
    budget: 5000,
    spent: 3200,
    impressions: 78000,
    clicks: 2100,
    leads: 89,
    startDate: new Date("2024-11-15"),
    endDate: new Date("2024-12-25"),
    properties: ["prop-3"]
  }
];

const mockPromotions = [
  {
    id: "promo-1",
    propertyId: "prop-1",
    title: "Modern Downtown Apartment",
    type: "featured",
    status: "active",
    views: 2340,
    inquiries: 23,
    cost: 299,
    startDate: new Date("2024-12-10"),
    endDate: new Date("2024-12-24"),
    platforms: ["website", "social"]
  },
  {
    id: "promo-2",
    propertyId: "prop-2", 
    title: "Luxury Beachfront Villa",
    type: "premium",
    status: "scheduled",
    views: 0,
    inquiries: 0,
    cost: 599,
    startDate: new Date("2024-12-20"),
    endDate: new Date("2025-01-05"),
    platforms: ["website", "social", "google"]
  }
];

export default function MarketingTools() {
  const [activeTab, setActiveTab] = useState("campaigns");
  const { toast } = useToast();

  const AdCampaigns = () => {
    const handleCreateCampaign = () => {
      toast({
        title: "Campaign Created",
        description: "Your new ad campaign has been created successfully.",
      });
    };

    const handlePauseCampaign = (campaignId: string) => {
      toast({
        title: "Campaign Paused",
        description: "The campaign has been paused.",
      });
    };

    const handleResumeCampaign = (campaignId: string) => {
      toast({
        title: "Campaign Resumed", 
        description: "The campaign is now active.",
      });
    };

    const getStatusColor = (status: string) => {
      switch (status) {
        case "active":
          return "bg-green-100 text-green-800";
        case "paused":
          return "bg-yellow-100 text-yellow-800";
        case "completed":
          return "bg-blue-100 text-blue-800";
        default:
          return "bg-gray-100 text-gray-800";
      }
    };

    return (
      <div className="space-y-6">
        {/* Stats Cards */}
        <div className="grid grid-cols-2 md:grid-cols-4 gap-3 md:gap-6">
          <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-blue-700 font-medium">Active Campaigns</p>
                  <p className="text-3xl font-bold text-blue-800">
                    {mockCampaigns.filter(c => c.status === "active").length}
                  </p>
                </div>
                <Target className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-green-700 font-medium">Total Leads</p>
                  <p className="text-3xl font-bold text-green-800">
                    {mockCampaigns.reduce((sum, c) => sum + c.leads, 0)}
                  </p>
                </div>
                <Users className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-purple-700 font-medium">Total Spent</p>
                  <p className="text-3xl font-bold text-purple-800">
                    ${mockCampaigns.reduce((sum, c) => sum + c.spent, 0).toLocaleString()}
                  </p>
                </div>
                <DollarSign className="w-8 h-8 text-purple-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-orange-700 font-medium">Impressions</p>
                  <p className="text-3xl font-bold text-orange-800">
                    {(mockCampaigns.reduce((sum, c) => sum + c.impressions, 0) / 1000).toFixed(0)}K
                  </p>
                </div>
                <Eye className="w-8 h-8 text-orange-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Campaigns List */}
        <div className="space-y-4">
          {mockCampaigns.map((campaign) => (
            <Card key={campaign.id}>
              <CardContent className="p-6">
                <div className="flex justify-between items-start mb-4">
                  <div className="flex-1">
                    <div className="flex items-center gap-3 mb-2">
                      <h3 className="text-lg font-semibold">{campaign.name}</h3>
                      <Badge className={getStatusColor(campaign.status)}>
                        {campaign.status.charAt(0).toUpperCase() + campaign.status.slice(1)}
                      </Badge>
                      <Badge variant="outline" className="capitalize">
                        {campaign.type.replace('_', ' ')}
                      </Badge>
                    </div>
                    <p className="text-sm text-muted-foreground">
                      {campaign.startDate.toLocaleDateString()} - {campaign.endDate.toLocaleDateString()}
                    </p>
                  </div>
                  <div className="text-right">
                    <p className="text-2xl font-bold">${campaign.spent.toLocaleString()}</p>
                    <p className="text-sm text-muted-foreground">of ${campaign.budget.toLocaleString()}</p>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-4">
                  <div>
                    <h4 className="font-medium mb-2">Performance</h4>
                    <div className="space-y-1">
                      <p className="text-sm">
                        <span className="text-muted-foreground">Impressions:</span> {campaign.impressions.toLocaleString()}
                      </p>
                      <p className="text-sm">
                        <span className="text-muted-foreground">Clicks:</span> {campaign.clicks.toLocaleString()}
                      </p>
                      <p className="text-sm">
                        <span className="text-muted-foreground">Leads:</span> {campaign.leads}
                      </p>
                    </div>
                  </div>

                  <div>
                    <h4 className="font-medium mb-2">Budget</h4>
                    <div className="space-y-1">
                      <p className="text-sm">
                        <span className="text-muted-foreground">Budget:</span> ${campaign.budget.toLocaleString()}
                      </p>
                      <p className="text-sm">
                        <span className="text-muted-foreground">Spent:</span> ${campaign.spent.toLocaleString()}
                      </p>
                      <p className="text-sm">
                        <span className="text-muted-foreground">Remaining:</span> ${(campaign.budget - campaign.spent).toLocaleString()}
                      </p>
                    </div>
                  </div>

                  <div>
                    <h4 className="font-medium mb-2">Metrics</h4>
                    <div className="space-y-1">
                      <p className="text-sm">
                        <span className="text-muted-foreground">CTR:</span> {((campaign.clicks / campaign.impressions) * 100).toFixed(2)}%
                      </p>
                      <p className="text-sm">
                        <span className="text-muted-foreground">CPC:</span> ${(campaign.spent / campaign.clicks).toFixed(2)}
                      </p>
                      <p className="text-sm">
                        <span className="text-muted-foreground">CPL:</span> ${(campaign.spent / campaign.leads).toFixed(2)}
                      </p>
                    </div>
                  </div>

                  <div>
                    <h4 className="font-medium mb-2">Progress</h4>
                    <div className="w-full bg-gray-200 rounded-full h-2 mb-2">
                      <div 
                        className="bg-blue-600 h-2 rounded-full"
                        style={{ width: `${(campaign.spent / campaign.budget) * 100}%` }}
                      ></div>
                    </div>
                    <p className="text-xs text-muted-foreground">
                      {Math.round((campaign.spent / campaign.budget) * 100)}% of budget used
                    </p>
                  </div>
                </div>

                <div className="flex gap-2 flex-wrap">
                  {campaign.status === "active" ? (
                    <Button size="sm" variant="outline" onClick={() => handlePauseCampaign(campaign.id)}>
                      <Pause className="w-4 h-4 mr-1" />
                      Pause
                    </Button>
                  ) : (
                    <Button size="sm" onClick={() => handleResumeCampaign(campaign.id)}>
                      <Play className="w-4 h-4 mr-1" />
                      Resume
                    </Button>
                  )}
                  <Button size="sm" variant="outline">
                    <Edit className="w-4 h-4 mr-1" />
                    Edit
                  </Button>
                  <Button size="sm" variant="outline">
                    <BarChart3 className="w-4 h-4 mr-1" />
                    Analytics
                  </Button>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  };

  const PromoteProperties = () => {
    const handlePromoteProperty = () => {
      toast({
        title: "Property Promoted",
        description: "Your property has been promoted successfully.",
      });
    };

    const getPromotionTypeColor = (type: string) => {
      switch (type) {
        case "featured":
          return "bg-blue-100 text-blue-800";
        case "premium":
          return "bg-purple-100 text-purple-800";
        case "spotlight":
          return "bg-yellow-100 text-yellow-800";
        default:
          return "bg-gray-100 text-gray-800";
      }
    };

    const getStatusColor = (status: string) => {
      switch (status) {
        case "active":
          return "bg-green-100 text-green-800";
        case "scheduled":
          return "bg-blue-100 text-blue-800";
        case "expired":
          return "bg-red-100 text-red-800";
        default:
          return "bg-gray-100 text-gray-800";
      }
    };

    return (
      <div className="space-y-6">
        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-3 md:gap-6">
          <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-green-700 font-medium">Active Promotions</p>
                  <p className="text-3xl font-bold text-green-800">
                    {mockPromotions.filter(p => p.status === "active").length}
                  </p>
                </div>
                <Star className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-blue-700 font-medium">Total Views</p>
                  <p className="text-3xl font-bold text-blue-800">
                    {mockPromotions.reduce((sum, p) => sum + p.views, 0).toLocaleString()}
                  </p>
                </div>
                <Eye className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-orange-700 font-medium">Total Inquiries</p>
                  <p className="text-3xl font-bold text-orange-800">
                    {mockPromotions.reduce((sum, p) => sum + p.inquiries, 0)}
                  </p>
                </div>
                <Users className="w-8 h-8 text-orange-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Promotions List */}
        <div className="space-y-4">
          {mockPromotions.map((promotion) => (
            <Card key={promotion.id}>
              <CardContent className="p-6">
                <div className="flex justify-between items-start mb-4">
                  <div className="flex-1">
                    <div className="flex items-center gap-3 mb-2">
                      <h3 className="text-lg font-semibold">{promotion.title}</h3>
                      <Badge className={getStatusColor(promotion.status)}>
                        {promotion.status.charAt(0).toUpperCase() + promotion.status.slice(1)}
                      </Badge>
                      <Badge className={getPromotionTypeColor(promotion.type)}>
                        {promotion.type.charAt(0).toUpperCase() + promotion.type.slice(1)}
                      </Badge>
                    </div>
                    <p className="text-sm text-muted-foreground">
                      <Calendar className="inline w-4 h-4 mr-1" />
                      {promotion.startDate.toLocaleDateString()} - {promotion.endDate.toLocaleDateString()}
                    </p>
                  </div>
                  <div className="text-right">
                    <p className="text-2xl font-bold">${promotion.cost}</p>
                    <p className="text-sm text-muted-foreground">promotion cost</p>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-4">
                  <div>
                    <h4 className="font-medium mb-2">Performance</h4>
                    <div className="space-y-1">
                      <p className="text-sm">
                        <span className="text-muted-foreground">Views:</span> {promotion.views.toLocaleString()}
                      </p>
                      <p className="text-sm">
                        <span className="text-muted-foreground">Inquiries:</span> {promotion.inquiries}
                      </p>
                      <p className="text-sm">
                        <span className="text-muted-foreground">Conversion:</span> {promotion.views > 0 ? ((promotion.inquiries / promotion.views) * 100).toFixed(2) : 0}%
                      </p>
                    </div>
                  </div>

                  <div>
                    <h4 className="font-medium mb-2">Platforms</h4>
                    <div className="flex flex-wrap gap-1">
                      {promotion.platforms.map((platform) => (
                        <Badge key={platform} variant="outline" className="text-xs">
                          {platform}
                        </Badge>
                      ))}
                    </div>
                  </div>

                  <div>
                    <h4 className="font-medium mb-2">Duration</h4>
                    <div className="space-y-1">
                      <p className="text-sm">
                        <span className="text-muted-foreground">Days remaining:</span> {Math.max(0, Math.ceil((promotion.endDate.getTime() - new Date().getTime()) / (1000 * 60 * 60 * 24)))}
                      </p>
                    </div>
                  </div>
                </div>

                <div className="flex gap-2 flex-wrap">
                  <Button size="sm" variant="outline">
                    <Edit className="w-4 h-4 mr-1" />
                    Edit
                  </Button>
                  <Button size="sm" variant="outline">
                    <Share2 className="w-4 h-4 mr-1" />
                    Share
                  </Button>
                  <Button size="sm" variant="outline">
                    <BarChart3 className="w-4 h-4 mr-1" />
                    Analytics
                  </Button>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>
      </div>
    );
  };

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Marketing Tools</h1>
          <p className="text-muted-foreground">Manage ad campaigns and property promotions</p>
        </div>
        <Button>
          <Plus className="mr-2 h-4 w-4" />
          Create New
        </Button>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
        <TabsList className="grid w-full grid-cols-1 md:grid-cols-2 h-auto p-1 bg-gray-100 rounded-lg gap-2">
          <TabsTrigger 
            value="campaigns" 
            className={`flex flex-col items-center gap-2 p-4 md:p-5 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
              activeTab === "campaigns" 
                ? "bg-[#0056D2] text-white border-[#0056D2]" 
                : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
            }`}
          >
            <Target className="w-6 h-6 md:w-7 md:h-7" />
            <span className="font-semibold text-sm md:text-base text-center">Ad Campaigns</span>
          </TabsTrigger>
          <TabsTrigger 
            value="promote" 
            className={`flex flex-col items-center gap-2 p-4 md:p-5 rounded-md border shadow-sm transition-all duration-200 cursor-pointer ${
              activeTab === "promote" 
                ? "bg-[#0056D2] text-white border-[#0056D2]" 
                : "bg-white border-gray-200 hover:bg-blue-50 hover:shadow-md"
            }`}
          >
            <TrendingUp className="w-6 h-6 md:w-7 md:h-7" />
            <span className="font-semibold text-sm md:text-base text-center">Promote Properties</span>
          </TabsTrigger>
        </TabsList>

        <TabsContent value="campaigns" className="mt-6">
          <AdCampaigns />
        </TabsContent>

        <TabsContent value="promote" className="mt-6">
          <PromoteProperties />
        </TabsContent>
      </Tabs>
    </div>
  );
}