import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Badge } from "../components/ui/badge";
import { Button } from "../components/ui/button";
import { TrendingUp, TrendingDown, Eye, Heart, MessageCircle, Calendar, Target, Zap, AlertCircle, CheckCircle } from "lucide-react";

export default function PerformanceHub() {
  const propertyPerformance = [
    {
      id: 1,
      title: "Modern Downtown Apartment",
      views: 1247,
      inquiries: 23,
      favorites: 89,
      daysListed: 12,
      performance: "excellent",
      conversionRate: 1.8,
      recommendations: [
        "Add virtual tour to increase engagement",
        "Price is competitive - maintain current pricing"
      ]
    },
    {
      id: 2,
      title: "Suburban Family Home",
      views: 456,
      inquiries: 8,
      favorites: 34,
      daysListed: 28,
      performance: "good",
      conversionRate: 1.7,
      recommendations: [
        "Consider professional photography",
        "Add more interior photos",
        "Highlight family-friendly features"
      ]
    },
    {
      id: 3,
      title: "Luxury Penthouse",
      views: 234,
      inquiries: 3,
      favorites: 12,
      daysListed: 45,
      performance: "needs_attention",
      conversionRate: 1.3,
      recommendations: [
        "Price may be too high - consider 5% reduction",
        "Enhance luxury features in description",
        "Add drone footage of city views"
      ]
    }
  ];

  const marketingInsights = {
    bestPerformingImages: [
      { type: "Exterior shots", engagement: 89 },
      { type: "Kitchen photos", engagement: 76 },
      { type: "Living room", engagement: 68 }
    ],
    optimalPostingTimes: [
      { day: "Saturday", time: "10:00 AM", engagement: 94 },
      { day: "Sunday", time: "2:00 PM", engagement: 87 },
      { day: "Thursday", time: "7:00 PM", engagement: 82 }
    ],
    topTrafficSources: [
      { source: "Organic Search", percentage: 45, trend: "up" },
      { source: "Social Media", percentage: 28, trend: "up" },
      { source: "Direct Traffic", percentage: 18, trend: "stable" },
      { source: "Referrals", percentage: 9, trend: "down" }
    ]
  };

  const getPerformanceColor = (performance: string) => {
    switch (performance) {
      case "excellent":
        return "bg-green-100 text-green-800";
      case "good":
        return "bg-blue-100 text-blue-800";
      case "needs_attention":
        return "bg-red-100 text-red-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getPerformanceIcon = (performance: string) => {
    switch (performance) {
      case "excellent":
        return <CheckCircle className="w-4 h-4" />;
      case "good":
        return <Target className="w-4 h-4" />;
      case "needs_attention":
        return <AlertCircle className="w-4 h-4" />;
      default:
        return <Eye className="w-4 h-4" />;
    }
  };

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Performance Hub</h1>
          <p className="text-muted-foreground">Optimize your property performance with AI-driven insights</p>
        </div>
        <Button>
          <TrendingUp className="mr-2 h-4 w-4" />
          Performance Report
        </Button>
      </div>

      {/* Performance Overview */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-blue-700 font-medium">Total Views</p>
                <p className="text-3xl font-bold text-blue-800">1,937</p>
                <div className="flex items-center mt-2">
                  <TrendingUp className="w-4 h-4 text-green-600 mr-1" />
                  <span className="text-sm text-green-600 font-medium">+15%</span>
                </div>
              </div>
              <Eye className="w-8 h-8 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-green-700 font-medium">Inquiries</p>
                <p className="text-3xl font-bold text-green-800">34</p>
                <div className="flex items-center mt-2">
                  <TrendingUp className="w-4 h-4 text-green-600 mr-1" />
                  <span className="text-sm text-green-600 font-medium">+8%</span>
                </div>
              </div>
              <MessageCircle className="w-8 h-8 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-purple-700 font-medium">Favorites</p>
                <p className="text-3xl font-bold text-purple-800">135</p>
                <div className="flex items-center mt-2">
                  <TrendingUp className="w-4 h-4 text-green-600 mr-1" />
                  <span className="text-sm text-green-600 font-medium">+22%</span>
                </div>
              </div>
              <Heart className="w-8 h-8 text-purple-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-orange-700 font-medium">Avg Conversion</p>
                <p className="text-3xl font-bold text-orange-800">1.6%</p>
                <div className="flex items-center mt-2">
                  <TrendingDown className="w-4 h-4 text-red-600 mr-1" />
                  <span className="text-sm text-red-600 font-medium">-3%</span>
                </div>
              </div>
              <Target className="w-8 h-8 text-orange-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Property Performance Analysis */}
      <Card>
        <CardHeader>
          <CardTitle>Property Performance Analysis</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="space-y-6">
            {propertyPerformance.map((property) => (
              <div key={property.id} className="border rounded-lg p-4">
                <div className="flex items-center justify-between mb-4">
                  <div className="flex items-center gap-3">
                    <h4 className="font-semibold">{property.title}</h4>
                    <Badge className={getPerformanceColor(property.performance)}>
                      {getPerformanceIcon(property.performance)}
                      <span className="ml-1 capitalize">{property.performance.replace('_', ' ')}</span>
                    </Badge>
                  </div>
                  <div className="text-right">
                    <p className="text-sm text-muted-foreground">Listed {property.daysListed} days ago</p>
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-4">
                  <div className="text-center p-3 bg-muted/30 rounded-lg">
                    <Eye className="w-5 h-5 mx-auto mb-1 text-blue-600" />
                    <p className="text-2xl font-bold">{property.views.toLocaleString()}</p>
                    <p className="text-xs text-muted-foreground">Views</p>
                  </div>
                  <div className="text-center p-3 bg-muted/30 rounded-lg">
                    <MessageCircle className="w-5 h-5 mx-auto mb-1 text-green-600" />
                    <p className="text-2xl font-bold">{property.inquiries}</p>
                    <p className="text-xs text-muted-foreground">Inquiries</p>
                  </div>
                  <div className="text-center p-3 bg-muted/30 rounded-lg">
                    <Heart className="w-5 h-5 mx-auto mb-1 text-red-600" />
                    <p className="text-2xl font-bold">{property.favorites}</p>
                    <p className="text-xs text-muted-foreground">Favorites</p>
                  </div>
                  <div className="text-center p-3 bg-muted/30 rounded-lg">
                    <Target className="w-5 h-5 mx-auto mb-1 text-purple-600" />
                    <p className="text-2xl font-bold">{property.conversionRate}%</p>
                    <p className="text-xs text-muted-foreground">Conversion</p>
                  </div>
                </div>

                <div className="bg-blue-50 p-3 rounded-lg">
                  <h5 className="font-medium mb-2 flex items-center">
                    <Zap className="w-4 h-4 mr-1 text-blue-600" />
                    AI Recommendations
                  </h5>
                  <ul className="space-y-1">
                    {property.recommendations.map((rec, index) => (
                      <li key={index} className="text-sm text-blue-800 flex items-start">
                        <span className="w-1 h-1 bg-blue-600 rounded-full mt-2 mr-2 flex-shrink-0"></span>
                        {rec}
                      </li>
                    ))}
                  </ul>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Marketing Insights */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <Card>
          <CardHeader>
            <CardTitle>Best Performing Images</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {marketingInsights.bestPerformingImages.map((image, index) => (
                <div key={index} className="flex items-center justify-between">
                  <span className="text-sm font-medium">{image.type}</span>
                  <div className="flex items-center gap-2">
                    <div className="w-16 bg-gray-200 rounded-full h-2">
                      <div 
                        className="bg-blue-600 h-2 rounded-full" 
                        style={{ width: `${image.engagement}%` }}
                      ></div>
                    </div>
                    <span className="text-sm text-muted-foreground">{image.engagement}%</span>
                  </div>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Optimal Posting Times</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {marketingInsights.optimalPostingTimes.map((time, index) => (
                <div key={index} className="flex items-center justify-between p-2 bg-muted/30 rounded">
                  <div>
                    <p className="font-medium text-sm">{time.day}</p>
                    <p className="text-xs text-muted-foreground">{time.time}</p>
                  </div>
                  <Badge variant="outline">{time.engagement}% engagement</Badge>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardHeader>
            <CardTitle>Traffic Sources</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {marketingInsights.topTrafficSources.map((source, index) => (
                <div key={index} className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <span className="text-sm font-medium">{source.source}</span>
                    {source.trend === "up" && <TrendingUp className="w-3 h-3 text-green-600" />}
                    {source.trend === "down" && <TrendingDown className="w-3 h-3 text-red-600" />}
                  </div>
                  <span className="text-sm font-bold">{source.percentage}%</span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}