import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../components/ui/tabs";
import { 
  TrendingUp, 
  TrendingDown,
  Target,
  AlertTriangle,
  CheckCircle,
  Eye,
  Users,
  Heart,
  MessageCircle,
  Zap,
  BarChart3,
  Clock,
  Home,
  RefreshCw
} from "lucide-react";

interface Property {
  id: string;
  title: string;
  views: number;
  inquiries: number;
  favorites: number;
  daysListed: number;
  performance: "excellent" | "good" | "needs_attention";
  conversionRate: number;
  aiScore: number;
  price: number;
  recommendations: Recommendation[];
}

interface Recommendation {
  type: "pricing" | "photos" | "description" | "features";
  priority: "high" | "medium" | "low";
  title: string;
  description: string;
  impact: string;
  effort: "low" | "medium" | "high";
  confidence: number;
}

export default function PerformanceInsights() {
  const [activeTab, setActiveTab] = useState("overview");
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [debugInfo, setDebugInfo] = useState<string>("");
  const [predictions, setPredictions] = useState<any>(null);
  const [localities, setLocalities] = useState<string[]>([]);
  const [selectedLocality, setSelectedLocality] = useState<string>("");

  const fetchData = async () => {
    try {
      setDebugInfo("Connecting to backend...");
      
      const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
      const endpoint = `${API_BASE}/properties/`;
      
      console.log(`Fetching from: ${endpoint}`);
      const response = await fetch(endpoint, {
        method: 'GET',
        headers: { 
          'Accept': 'application/json',
          'Content-Type': 'application/json'
        }
      });
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }
      
      const data = await response.json();
      console.log("Backend data:", data);
      
      let properties = data.properties || data.results || data || [];
      setDebugInfo(`✅ Connected to backend - Found ${properties.length} properties`);
      
      return processProperties(properties);
    } catch (error: unknown) {
      console.error('Backend connection failed:', error);
      setDebugInfo(`❌ Backend connection failed: ${error instanceof Error ? error.message : 'Unknown error'}`);
      throw error;
    }
  };



  const generateRecommendations = (property: any, views: number, inquiries: number, daysListed: number, conversionRate: number) => {
    const recommendations: Recommendation[] = [];
    
    // Low views recommendation
    if (views < 100) {
      recommendations.push({
        type: "photos", priority: "high", title: "Add Professional Photos",
        description: "Properties with high-quality photos get 300% more views.",
        impact: "300% more views", effort: "medium", confidence: 95
      });
    }
    
    // Low inquiries recommendation
    if (inquiries < 5) {
      recommendations.push({
        type: "pricing", priority: "high", title: "Optimize Pricing Strategy",
        description: "Consider competitive pricing to increase inquiries.",
        impact: "50% more inquiries", effort: "low", confidence: 88
      });
    }
    
    // Long listing period
    if (daysListed > 30) {
      recommendations.push({
        type: "description", priority: "medium", title: "Improve Property Description",
        description: "Enhance description with key features and amenities.",
        impact: "25% better engagement", effort: "low", confidence: 82
      });
    }
    
    // Low conversion rate
    if (conversionRate < 2) {
      recommendations.push({
        type: "features", priority: "medium", title: "Highlight Unique Features",
        description: "Emphasize property's unique selling points and amenities.",
        impact: "35% higher conversion", effort: "low", confidence: 78
      });
    }
    
    // High performance - no recommendations
    if (views > 200 && inquiries > 8 && conversionRate > 4) {
      return [];
    }
    
    return recommendations;
  };

  const processProperties = (properties: any[]) => {
    return properties.map((property: any, index: number) => {
      const views = property.views || Math.floor(Math.random() * 200) + 50;
      const inquiries = Math.floor(Math.random() * 10) + 1;
      const favorites = Math.floor(Math.random() * 15) + 5;
      const daysListed = Math.floor((Date.now() - new Date(property.created_at || Date.now()).getTime()) / (1000 * 60 * 60 * 24));
      const conversionRate = inquiries > 0 ? ((inquiries / views) * 100) : 0;
      
      // Determine performance based on metrics
      let performance: "excellent" | "good" | "needs_attention";
      let aiScore: number;
      
      if (views > 200 && inquiries > 8 && conversionRate > 4) {
        performance = "excellent";
        aiScore = 85 + Math.floor(Math.random() * 15);
      } else if (views > 100 && inquiries > 4 && conversionRate > 2) {
        performance = "good";
        aiScore = 65 + Math.floor(Math.random() * 20);
      } else {
        performance = "needs_attention";
        aiScore = 40 + Math.floor(Math.random() * 25);
      }
      
      return {
        id: property.id.toString(),
        title: property.title || `Property ${index + 1}`,
        views,
        inquiries,
        favorites,
        daysListed,
        performance,
        conversionRate: parseFloat(conversionRate.toFixed(1)),
        aiScore,
        price: parseFloat(property.sale_price || property.monthly_rent || 0),
        recommendations: generateRecommendations(property, views, inquiries, daysListed, conversionRate)
      };
    });
  };

  const fetchPredictions = async (locality: string) => {
    try {
      const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
      const response = await fetch(`${API_BASE}/market-trends/forecast/${locality}`);
      if (response.ok) {
        const data = await response.json();
        setPredictions(data);
      }
    } catch (error) {
      console.error('Failed to fetch predictions:', error);
    }
  };

  const fetchLocalities = async () => {
    try {
      const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
      const response = await fetch(`${API_BASE}/market-trends/localities`);
      if (response.ok) {
        const data = await response.json();
        setLocalities(data.localities || []);
        if (data.localities?.length > 0) {
          setSelectedLocality(data.localities[0]);
          fetchPredictions(data.localities[0]);
        }
      }
    } catch (error) {
      console.error('Failed to fetch localities:', error);
    }
  };


  useEffect(() => {
    const loadData = async () => {
      setLoading(true);
      try {
        const [propertiesData] = await Promise.all([
          fetchData(),
          fetchLocalities()
        ]);
        setProperties(propertiesData);
      } catch (error) {
        setProperties([]);
      }
      setLoading(false);
    };
    loadData();
  }, []);

  const getPerformanceColor = (performance: string) => {
    switch (performance) {
      case "excellent": return "bg-green-50 text-green-700 border-green-200";
      case "good": return "bg-blue-50 text-blue-700 border-blue-200";
      case "needs_attention": return "bg-red-50 text-red-700 border-red-200";
      default: return "bg-gray-50 text-gray-700 border-gray-200";
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case "high": return "bg-red-50 text-red-700 border-red-200";
      case "medium": return "bg-orange-50 text-orange-700 border-orange-200";
      case "low": return "bg-blue-50 text-blue-700 border-blue-200";
      default: return "bg-gray-50 text-gray-700 border-gray-200";
    }
  };

  if (loading) {
    return (
      <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
        <div className="flex items-center justify-center h-64">
          <div className="text-center">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
            <p className="text-gray-600">Loading performance data...</p>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6 bg-gray-50 min-h-screen">
      <div>
        <h1 className="text-3xl font-bold tracking-tight text-gray-900">Performance Insights</h1>
        <p className="text-gray-600">AI-powered insights to optimize your property listings</p>
      </div>



      <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
        <TabsList className="grid w-full grid-cols-3 h-auto p-1 bg-white rounded-lg gap-2 border border-gray-200">
          <TabsTrigger 
            value="overview" 
            className={`flex flex-col items-center gap-2 p-4 rounded-md border transition-all duration-200 ${
              activeTab === "overview" 
                ? "bg-blue-600 text-black border-blue-600" 
                : "bg-white border-gray-200 hover:bg-blue-50"
            }`}
          >
            <BarChart3 className="w-5 h-5" />
            <span className="font-medium text-sm">Overview</span>
          </TabsTrigger>
          <TabsTrigger 
            value="predictions" 
            className={`flex flex-col items-center gap-2 p-4 rounded-md border transition-all duration-200 ${
              activeTab === "predictions" 
                ? "bg-blue-600 text-black border-blue-600" 
                : "bg-white border-gray-200 hover:bg-blue-50"
            }`}
          >
            <TrendingUp className="w-5 h-5" />
            <span className="font-medium text-sm">Predictions</span>
          </TabsTrigger>
          <TabsTrigger 
            value="leads" 
            className={`flex flex-col items-center gap-2 p-4 rounded-md border transition-all duration-200 ${
              activeTab === "leads" 
                ? "bg-blue-600 text-black border-blue-600" 
                : "bg-white border-gray-200 hover:bg-blue-50"
            }`}
          >
            <Target className="w-5 h-5" />
            <span className="font-medium text-sm">Leads</span>
          </TabsTrigger>
        </TabsList>

        <TabsContent value="overview" className="mt-6">
          <div className="space-y-6">
            {/* Performance Cards */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
              <Card className="border-l-4 border-l-blue-500">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-gray-600 font-medium">Total Views</p>
                      <p className="text-3xl font-bold text-gray-900">
                        {properties.reduce((sum, p) => sum + p.views, 0).toLocaleString()}
                      </p>
                      <div className="flex items-center mt-2">
                        <TrendingUp className="w-4 h-4 text-green-600 mr-1" />
                        <span className="text-sm text-green-600 font-medium">+15%</span>
                      </div>
                    </div>
                    <Eye className="w-8 h-8 text-blue-600" />
                  </div>
                </CardContent>
              </Card>

              <Card className="border-l-4 border-l-green-500">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-gray-600 font-medium">Inquiries</p>
                      <p className="text-3xl font-bold text-gray-900">
                        {properties.reduce((sum, p) => sum + p.inquiries, 0)}
                      </p>
                      <div className="flex items-center mt-2">
                        <TrendingUp className="w-4 h-4 text-green-600 mr-1" />
                        <span className="text-sm text-green-600 font-medium">+8%</span>
                      </div>
                    </div>
                    <MessageCircle className="w-8 h-8 text-green-600" />
                  </div>
                </CardContent>
              </Card>

              <Card className="border-l-4 border-l-purple-500">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-gray-600 font-medium">Favorites</p>
                      <p className="text-3xl font-bold text-gray-900">
                        {properties.reduce((sum, p) => sum + p.favorites, 0)}
                      </p>
                      <div className="flex items-center mt-2">
                        <TrendingUp className="w-4 h-4 text-green-600 mr-1" />
                        <span className="text-sm text-green-600 font-medium">+22%</span>
                      </div>
                    </div>
                    <Heart className="w-8 h-8 text-purple-600" />
                  </div>
                </CardContent>
              </Card>

              <Card className="border-l-4 border-l-orange-500">
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-gray-600 font-medium">Avg Conversion</p>
                      <p className="text-3xl font-bold text-gray-900">
                        {properties.length > 0 ? (properties.reduce((sum, p) => sum + p.conversionRate, 0) / properties.length).toFixed(1) : '0'}%
                      </p>
                      <div className="flex items-center mt-2">
                        <TrendingDown className="w-4 h-4 text-red-600 mr-1" />
                        <span className="text-sm text-red-600 font-medium">-3%</span>
                      </div>
                    </div>
                    <Target className="w-8 h-8 text-orange-600" />
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Property Analysis */}
            <Card>
              <CardHeader>
                <CardTitle className="text-xl font-bold text-gray-900">Property Performance Analysis</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-6">
                  {properties.map((property) => (
                    <div key={property.id} className="border rounded-lg p-6 bg-white shadow-sm">
                      <div className="flex items-center justify-between mb-4">
                        <div className="flex items-center gap-3">
                          <h4 className="font-semibold text-lg text-gray-900">{property.title}</h4>
                          <Badge className={`${getPerformanceColor(property.performance)} border`}>
                            {property.performance === "excellent" && <CheckCircle className="w-4 h-4 mr-1" />}
                            {property.performance === "good" && <Target className="w-4 h-4 mr-1" />}
                            {property.performance === "needs_attention" && <AlertTriangle className="w-4 h-4 mr-1" />}
                            <span className="capitalize">{property.performance.replace('_', ' ')}</span>
                          </Badge>
                          <div className="flex items-center gap-2">
                            <span className="text-sm text-gray-500">AI Score:</span>
                            <span className="text-lg font-bold text-blue-600">{property.aiScore}/100</span>
                          </div>
                        </div>
                        <div className="text-right">
                          <p className="text-sm text-gray-500">Listed {property.daysListed} days ago</p>
                          <p className="text-lg font-bold text-gray-900">₹{(property.price / 100000).toFixed(1)}L</p>
                        </div>
                      </div>

                      <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                        <div className="text-center p-4 bg-blue-50 rounded-lg border border-blue-200">
                          <Eye className="w-6 h-6 mx-auto mb-2 text-blue-600" />
                          <p className="text-2xl font-bold text-blue-800">{property.views}</p>
                          <p className="text-xs text-blue-600 font-medium">Views</p>
                        </div>
                        <div className="text-center p-4 bg-green-50 rounded-lg border border-green-200">
                          <MessageCircle className="w-6 h-6 mx-auto mb-2 text-green-600" />
                          <p className="text-2xl font-bold text-green-800">{property.inquiries}</p>
                          <p className="text-xs text-green-600 font-medium">Inquiries</p>
                        </div>
                        <div className="text-center p-4 bg-purple-50 rounded-lg border border-purple-200">
                          <Heart className="w-6 h-6 mx-auto mb-2 text-purple-600" />
                          <p className="text-2xl font-bold text-purple-800">{property.favorites}</p>
                          <p className="text-xs text-purple-600 font-medium">Favorites</p>
                        </div>
                        <div className="text-center p-4 bg-orange-50 rounded-lg border border-orange-200">
                          <Target className="w-6 h-6 mx-auto mb-2 text-orange-600" />
                          <p className="text-2xl font-bold text-orange-800">{property.conversionRate}%</p>
                          <p className="text-xs text-orange-600 font-medium">Conversion</p>
                        </div>
                      </div>

                      <div className="bg-blue-50 p-4 rounded-lg border border-blue-200">
                        <h5 className="font-semibold mb-3 flex items-center text-blue-800">
                          <Zap className="w-5 h-5 mr-2 text-blue-600" />
                          AI Recommendations
                        </h5>
                        {property.recommendations.length === 0 ? (
                          <div className="text-center py-4 text-blue-600">
                            <CheckCircle className="w-8 h-8 mx-auto mb-2" />
                            <p>Excellent! No recommendations needed.</p>
                          </div>
                        ) : (
                          <div className="space-y-3">
                            {property.recommendations.map((rec, index) => (
                              <div key={index} className="bg-white p-4 rounded-lg border border-blue-200">
                                <div className="flex items-start justify-between">
                                  <div className="flex-1">
                                    <div className="flex items-center gap-2 mb-2">
                                      <Badge className={`${getPriorityColor(rec.priority)} border text-xs`}>
                                        {rec.priority.toUpperCase()} PRIORITY
                                      </Badge>
                                      <span className="font-semibold text-gray-900">{rec.title}</span>
                                    </div>
                                    <p className="text-sm text-gray-600 mb-2">{rec.description}</p>
                                    <div className="flex items-center gap-4 text-sm">
                                      <div className="flex items-center gap-1">
                                        <TrendingUp className="w-4 h-4 text-green-600" />
                                        <span className="text-green-600 font-medium">{rec.impact}</span>
                                      </div>
                                      <div className="flex items-center gap-1">
                                        <Clock className="w-4 h-4 text-blue-600" />
                                        <span className="text-gray-600">{rec.effort} effort</span>
                                      </div>
                                      <div className="flex items-center gap-1">
                                        <Target className="w-4 h-4 text-purple-600" />
                                        <span className="text-gray-600">{rec.confidence}% confidence</span>
                                      </div>
                                    </div>
                                  </div>
                                  <Button className="bg-blue-600 hover:bg-blue-700 text-white">
                                    Implement
                                  </Button>
                                </div>
                              </div>
                            ))}
                          </div>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        <TabsContent value="predictions" className="mt-6">
          <div className="space-y-6">
            <Card>
              <CardContent className="p-4">
                <div className="flex items-center gap-4">
                  <label className="text-sm font-medium text-gray-700">Select Locality:</label>
                  <select 
                    value={selectedLocality} 
                    onChange={(e) => {
                      setSelectedLocality(e.target.value);
                      fetchPredictions(e.target.value);
                    }}
                    className="px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500"
                  >
                    {localities.map(locality => (
                      <option key={locality} value={locality}>
                        {locality.replace('_', ' ').toUpperCase()}
                      </option>
                    ))}
                  </select>
                </div>
              </CardContent>
            </Card>

            {predictions ? (
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <Card className="border border-green-200">
                  <CardHeader className="bg-green-50">
                    <CardTitle className="flex items-center gap-2 text-green-800">
                      <TrendingUp className="w-5 h-5" />
                      Price Forecast - {selectedLocality.replace('_', ' ').toUpperCase()}
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="p-6">
                    <div className="space-y-4">
                      <div>
                        <p className="text-sm text-gray-600">Current Price (per sq ft)</p>
                        <p className="text-2xl font-bold text-gray-900">₹{predictions.current_price?.toLocaleString()}</p>
                      </div>
                      <div className="space-y-2">
                        <div className="flex justify-between">
                          <span className="text-sm text-gray-600">6 Months</span>
                          <div className="text-right">
                            <span className="font-medium text-green-600">+{predictions.predictions?.['6_months']?.growth}%</span>
                            <p className="text-xs text-gray-500">₹{predictions.predictions?.['6_months']?.price?.toLocaleString()}</p>
                          </div>
                        </div>
                        <div className="flex justify-between">
                          <span className="text-sm text-gray-600">12 Months</span>
                          <div className="text-right">
                            <span className="font-medium text-green-600">+{predictions.predictions?.['12_months']?.growth}%</span>
                            <p className="text-xs text-gray-500">₹{predictions.predictions?.['12_months']?.price?.toLocaleString()}</p>
                          </div>
                        </div>
                      </div>
                    </div>
                  </CardContent>
                </Card>

                <Card className="border border-blue-200">
                  <CardHeader className="bg-blue-50">
                    <CardTitle className="flex items-center gap-2 text-blue-800">
                      <Target className="w-5 h-5" />
                      Market Insights
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="p-6">
                    <div className="space-y-4">
                      <div>
                        <p className="text-sm text-gray-600">Market Sentiment</p>
                        <div className="flex items-center gap-2">
                          <Badge className={`${
                            predictions.insights?.sentiment_color === 'green' ? 'bg-green-100 text-green-800 border-green-200' :
                            predictions.insights?.sentiment_color === 'yellow' ? 'bg-yellow-100 text-yellow-800 border-yellow-200' :
                            'bg-red-100 text-red-800 border-red-200'
                          } border`}>
                            {predictions.insights?.sentiment}
                          </Badge>
                          <span className="text-sm text-gray-600">{predictions.insights?.confidence_score}% confidence</span>
                        </div>
                      </div>
                      <div>
                        <p className="text-sm text-gray-600 mb-2">Recommendation</p>
                        <p className="text-sm text-gray-800">{predictions.insights?.recommendation}</p>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              </div>
            ) : (
              <Card>
                <CardContent className="p-8 text-center">
                  <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-4"></div>
                  <p className="text-gray-600">Loading market predictions...</p>
                </CardContent>
              </Card>
            )}
          </div>
        </TabsContent>

        <TabsContent value="leads" className="mt-6">
          <Card>
            <CardContent className="p-8 text-center">
              <Users className="w-12 h-12 mx-auto mb-4 text-gray-400" />
              <h3 className="text-lg font-semibold text-gray-900 mb-2">No Recent Inquiries</h3>
              <p className="text-gray-600">Lead predictions will appear when you receive inquiries.</p>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
