import React, { useState, useEffect, useMemo } from "react";
import { useLocation } from "wouter";
import { toast } from "sonner";
import { Property } from "../types";
import { api } from "../utils/api";
import SellerPropertyCard from "../components/SellerPropertyCard";
import PropertyDetailsModal from "../components/PropertyDetailsModal";
import DeleteConfirmModal from "../components/DeleteConfirmModal";
import AddAssetModal from "../components/AddAssetModal";
import { Button } from "../components/ui/button";
import { Input } from "../components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "../components/ui/select";
import {
  Plus,
  Search,
  Filter,
  Grid3X3,
  List,
  SlidersHorizontal,
  Home,
  TrendingUp,
} from "lucide-react";

type ViewMode = "grid" | "list";
type SortField = "title" | "price" | "createdAt" | "views";
type SortOrder = "asc" | "desc";

const PremiumPropertyDashboard: React.FC = () => {
  const [location, navigate] = useLocation();
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [filters, setFilters] = useState({
    search: "",
    propertyType: "all",
    status: "all",
    priceRange: "all",
  });
  const [showFilters, setShowFilters] = useState(false);
  const [viewMode, setViewMode] = useState<ViewMode>("grid");
  const [sortField, setSortField] = useState<SortField>("createdAt");
  const [sortOrder, setSortOrder] = useState<SortOrder>("desc");
  const [selectedProperty, setSelectedProperty] = useState<Property | null>(
    null
  );
  const [showDetailsModal, setShowDetailsModal] = useState(false);
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [showChoiceModal, setShowChoiceModal] = useState(false);
  const [showAssetModal, setShowAssetModal] = useState(false);
  const [isDeleting, setIsDeleting] = useState(false);

  const loadProperties = async (showLoading = true) => {
    try {
      if (showLoading) setLoading(true);
      const data = await api.fetchProperties();
      setProperties(data);
      setLoading(false);
    } catch (error) {
      console.error("Error loading properties:", error);
      setProperties([]);
      setLoading(false);
    }
  };

  useEffect(() => {
    loadProperties();

    // Auto-refresh every 15 seconds
    const interval = setInterval(() => loadProperties(false), 15000);

    // Listen for property added event
    const handlePropertyAdded = (event: CustomEvent) => {
      loadProperties(false);
      if (event.detail) {
        toast.success("Property list updated!", {
          description: "Your new property is now visible.",
          duration: 3000,
        });
      }
    };
    window.addEventListener(
      "propertyAdded",
      handlePropertyAdded as EventListener
    );

    // Listen for focus events to refresh when returning to page
    const handleFocus = () => loadProperties(false);
    window.addEventListener("focus", handleFocus);

    return () => {
      clearInterval(interval);
      window.removeEventListener(
        "propertyAdded",
        handlePropertyAdded as EventListener
      );
      window.removeEventListener("focus", handleFocus);
    };
  }, []);

  const filteredAndSortedProperties = useMemo(() => {
    console.log(
      "Properties with status:",
      properties.map((p) => ({
        id: p.id,
        title: p.title,
        status: p.status,
        price: p.price,
      }))
    );

    let filtered = properties.filter((property) => {
      const matchesSearch =
        !filters.search ||
        property.title.toLowerCase().includes(filters.search.toLowerCase()) ||
        property.address.toLowerCase().includes(filters.search.toLowerCase()) ||
        property.city.toLowerCase().includes(filters.search.toLowerCase());

      const matchesType =
        filters.propertyType === "all" ||
        property.type === filters.propertyType;

      const matchesStatus =
        filters.status === "all" ||
        (filters.status === "not_listed" && property.status === "pending") ||
        (filters.status === "sold" && property.status === "sold") ||
        (filters.status !== "not_listed" &&
          filters.status !== "sold" &&
          filters.status !== "all" &&
          property.status === filters.status);

      console.log(
        `Property ${property.id}: status=${property.status}, matchesStatus=${matchesStatus}, filter=${filters.status}`
      );

      let matchesPrice = true;
      if (filters.priceRange !== "all") {
        const price = property.price;
        switch (filters.priceRange) {
          case "under-50l":
            matchesPrice = price < 5000000;
            break;
          case "50l-1cr":
            matchesPrice = price >= 5000000 && price < 10000000;
            break;
          case "1cr-2cr":
            matchesPrice = price >= 10000000 && price < 20000000;
            break;
          case "above-2cr":
            matchesPrice = price >= 20000000;
            break;
        }
      }

      return matchesSearch && matchesType && matchesStatus && matchesPrice;
    });

    filtered.sort((a, b) => {
      let aValue: any = a[sortField];
      let bValue: any = b[sortField];

      if (sortField === "price" || sortField === "views") {
        aValue = Number(aValue);
        bValue = Number(bValue);
      } else if (sortField === "createdAt") {
        aValue = new Date(aValue);
        bValue = new Date(bValue);
      } else {
        aValue = String(aValue).toLowerCase();
        bValue = String(bValue).toLowerCase();
      }

      return sortOrder === "asc"
        ? aValue > bValue
          ? 1
          : -1
        : aValue < bValue
        ? 1
        : -1;
    });

    return filtered;
  }, [properties, filters, sortField, sortOrder]);

  const handleEdit = (property: Property) => {
    navigate(`/edit-property?id=${property.id}`);
  };

  const handleDelete = (property: Property) => {
    setSelectedProperty(property);
    setShowDeleteModal(true);
  };

  const confirmDelete = async () => {
    if (!selectedProperty) return;

    setIsDeleting(true);
    try {
      await api.deleteProperty(selectedProperty.id);
      setProperties((prev) => prev.filter((p) => p.id !== selectedProperty.id));
      setShowDeleteModal(false);
      setSelectedProperty(null);
    } catch (error: any) {
      console.error("Delete error:", error);
    } finally {
      setIsDeleting(false);
    }
  };

  const handleView = (property: Property) => {
    setSelectedProperty(property);
    setShowDetailsModal(true);
  };

  const handleAddProperty = () => {
    setShowChoiceModal(true);
  };

  const handleAddAsset = () => {
    setShowChoiceModal(false);
    // Show asset modal instead of navigating
    setShowAssetModal(true);
  };

  const handleListForSale = () => {
    setShowChoiceModal(false);
    navigate("/list-property?type=listing");
  };

  const handleListAssetForSale = (property: Property) => {
    navigate(`/list-property?asset=${property.id}`);
  };

  // Force refresh properties when component mounts
  useEffect(() => {
    const handleVisibilityChange = () => {
      if (!document.hidden) {
        loadProperties(false);
      }
    };

    document.addEventListener("visibilitychange", handleVisibilityChange);
    return () =>
      document.removeEventListener("visibilitychange", handleVisibilityChange);
  }, []);

  const clearFilters = () => {
    setFilters({
      search: "",
      propertyType: "all",
      status: "all",
      priceRange: "all",
    });
  };

  // Loading State
  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading properties...</p>
        </div>
      </div>
    );
  }

  // Empty State
  if (!loading && properties.length === 0) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center max-w-md">
          <div className="w-20 h-20 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-6">
            <Home className="w-10 h-10 text-blue-600" />
          </div>
          <h2 className="text-2xl font-bold text-gray-900 mb-3">
            Start Your Property Portfolio
          </h2>
          <p className="text-gray-600 mb-6">
            List your first property and connect with potential buyers.
          </p>
          <button
            onClick={handleAddProperty}
            className="bg-[#0056D2] hover:bg-blue-700 text-white px-6 py-3 rounded-lg font-medium transition-colors shadow-sm"
          >
            Add Your First Property
          </button>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-white border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="py-6">
            {/* Top Section */}
            <div className="flex items-center justify-between mb-6">
              <div>
                <h1 className="text-2xl font-bold text-gray-900">
                  Property Management
                </h1>
                <p className="text-gray-600 mt-1">
                  Manage and track your property listings
                </p>
              </div>
              <button
                onClick={handleAddProperty}
                className="bg-[#0056D2] hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-semibold transition-colors flex items-center gap-2 shadow-md border-2 border-[#0056D2]"
              >
                <Plus className="w-4 h-4" /> Add Property
              </button>
            </div>

            {/* Search and Filters */}
            <div className="flex items-center gap-4">
              {/* Search Bar */}
              <div className="flex-1 max-w-md">
                <Input
                  placeholder="Search properties..."
                  value={filters.search}
                  onChange={(e) =>
                    setFilters({ ...filters, search: e.target.value })
                  }
                  className="h-11 border border-gray-300 focus:border-blue-500 focus:ring-2 focus:ring-blue-200 rounded-xl shadow-sm px-4"
                />
              </div>

              {/* Property Type Filter */}
              <Select
                value={filters.propertyType}
                onValueChange={(value) =>
                  setFilters({ ...filters, propertyType: value })
                }
              >
                <SelectTrigger className="w-40 h-11 border border-gray-300 rounded-xl shadow-sm px-3 focus:border-blue-500 focus:ring-2 focus:ring-blue-200">
                  <SelectValue placeholder="Property Type" />
                </SelectTrigger>
                <SelectContent className="rounded-xl shadow-lg">
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="apartment">Apartment</SelectItem>
                  <SelectItem value="house">House</SelectItem>
                  <SelectItem value="villa">Villa</SelectItem>
                </SelectContent>
              </Select>

              {/* Status Filter */}
              <Select
                value={filters.status}
                onValueChange={(value) =>
                  setFilters({ ...filters, status: value })
                }
              >
                <SelectTrigger className="w-36 h-11 border border-gray-300 rounded-xl shadow-sm px-3 focus:border-blue-500 focus:ring-2 focus:ring-blue-200">
                  <SelectValue placeholder="Status" />
                </SelectTrigger>
                <SelectContent className="rounded-xl shadow-lg">
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="sold">Sold</SelectItem>
                  <SelectItem value="not_listed">Not Listed</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Results Header */}
        <div className="flex items-center justify-between mb-6">
          {/* <p className="text-sm text-gray-600">
            Showing {filteredAndSortedProperties.length} of {properties.length} properties
          </p> */}
          {/* <Select value={`${sortField}-${sortOrder}`} onValueChange={(value) => {
            const [field, order] = value.split('-');
            setSortField(field as SortField);
            setSortOrder(order as SortOrder);
          }}>
            <SelectTrigger className="w-40">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="createdAt-desc">Newest First</SelectItem>
              <SelectItem value="createdAt-asc">Oldest First</SelectItem>
              <SelectItem value="price-desc">Price: High to Low</SelectItem>
              <SelectItem value="price-asc">Price: Low to High</SelectItem>
              <SelectItem value="title-asc">Name: A to Z</SelectItem>
            </SelectContent>
          </Select> */}
        </div>

        {/* Properties Grid */}
        {filteredAndSortedProperties.length === 0 ? (
          <div className="text-center py-12">
            <Search className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">
              No properties found
            </h3>
            <p className="text-gray-600 mb-4">
              Try adjusting your search or filters
            </p>
            <Button variant="outline" onClick={clearFilters}>
              Clear filters
            </Button>
          </div>
        ) : (
          <div
            className={
              viewMode === "grid"
                ? "grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6"
                : "grid grid-cols-1 lg:grid-cols-2 gap-6"
            }
          >
            {filteredAndSortedProperties.map((property) => (
              <SellerPropertyCard
                key={property.id}
                property={property}
                onView={handleView}
                onEdit={handleEdit}
                onDelete={handleDelete}
                onPause={() => {}}
                onListForSale={handleListAssetForSale}
              />
            ))}
          </div>
        )}
      </div>

      {/* Modals */}
      {showChoiceModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 max-w-md w-full mx-4">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">
              Choose Property Action
            </h3>
            <p className="text-gray-600 mb-6">
              Would you like to add this property to your assets or list it for
              sale?
            </p>
            <div className="flex gap-3">
              <button
                onClick={handleAddAsset}
                className="flex-1 bg-gray-100 hover:bg-gray-200 text-gray-800 px-4 py-2 rounded-lg font-medium transition-colors"
              >
                Add to Assets
              </button>
              <button
                onClick={handleListForSale}
                className="flex-1 bg-[#0056D2] hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-medium transition-colors"
              >
                List for Sale/Rent
              </button>
            </div>
            <button
              onClick={() => setShowChoiceModal(false)}
              className="w-full mt-3 text-gray-500 hover:text-gray-700 py-2 text-sm"
            >
              Cancel
            </button>
          </div>
        </div>
      )}

      <PropertyDetailsModal
        property={selectedProperty}
        isOpen={showDetailsModal}
        onClose={() => {
          setShowDetailsModal(false);
          setSelectedProperty(null);
        }}
        onEdit={() => {
          setShowDetailsModal(false);
          setSelectedProperty(null);
          if (selectedProperty) {
            handleEdit(selectedProperty);
          }
        }}
      />

      <DeleteConfirmModal
        property={selectedProperty}
        isOpen={showDeleteModal}
        onClose={() => {
          setShowDeleteModal(false);
          setSelectedProperty(null);
        }}
        onConfirm={confirmDelete}
        isDeleting={isDeleting}
      />

      {/* Add Asset Modal */}
      {showAssetModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4 overflow-y-auto">
          <div className="bg-white rounded-lg w-full max-w-2xl my-8">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900">
                Add Property to Assets
              </h3>
            </div>
            <div className="p-6">
              <AddAssetModal onClose={() => setShowAssetModal(false)} />
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default PremiumPropertyDashboard;
