import React, { useState, useMemo } from 'react';
import { Download, FileText, Calendar, Filter, RefreshCw, TrendingUp, Building, DollarSign, Eye, Users } from 'lucide-react';
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";

const Reports: React.FC = () => {
  const [dateRange, setDateRange] = useState('30');
  const [selectedLocation, setSelectedLocation] = useState('All');
  const [selectedPropertyType, setSelectedPropertyType] = useState('All');
  const [reportType, setReportType] = useState('Property Performance Report');
  const [startDate, setStartDate] = useState('');
  const [endDate, setEndDate] = useState('');
  const [generating, setGenerating] = useState(false);

  // Mock data for seller reports
  const mockData = {
    properties: [
      { id: 1, title: '3BHK Luxury Apartment', type: 'Apartment', location: 'Bandra West', price: 8500000, views: 245, inquiries: 12, status: 'Active', listedDate: '2024-01-15' },
      { id: 2, title: 'Independent Villa', type: 'Villa', location: 'Andheri West', price: 12000000, views: 189, inquiries: 8, status: 'Active', listedDate: '2024-01-20' },
      { id: 3, title: 'Commercial Plot', type: 'Plot', location: 'Whitefield', price: 25000000, views: 156, inquiries: 15, status: 'Sold', listedDate: '2024-01-25' },
    ],
    inquiries: [
      { id: 1, propertyTitle: '3BHK Apartment', buyerName: 'Rajesh Kumar', phone: '+91-9876543210', date: '2024-01-15', status: 'Responded', message: 'Interested in viewing' },
      { id: 2, propertyTitle: '2BHK Villa', buyerName: 'Priya Sharma', phone: '+91-9876543211', date: '2024-01-20', status: 'Pending', message: 'Price negotiation' },
      { id: 3, propertyTitle: 'Commercial Plot', buyerName: 'Amit Patel', phone: '+91-9876543212', date: '2024-01-25', status: 'Responded', message: 'Document verification' },
    ],
    visits: [
      { id: 1, property: '3BHK Apartment', buyer: 'Rajesh Kumar', date: '2024-01-10', time: '10:00 AM', status: 'Completed', feedback: 'Very interested' },
      { id: 2, property: '2BHK Villa', buyer: 'Priya Sharma', date: '2024-01-15', time: '2:00 PM', status: 'Completed', feedback: 'Considering offer' },
      { id: 3, property: 'Commercial Plot', buyer: 'Amit Patel', date: '2024-01-20', time: '11:00 AM', status: 'Scheduled', feedback: 'Pending' },
    ],
    revenue: [
      { id: 1, property: '3BHK Apartment', amount: 8500000, type: 'Sale', status: 'Pending', date: '2024-01-30' },
      { id: 2, property: 'Commercial Plot', amount: 25000000, type: 'Sale', status: 'Completed', date: '2024-02-10' },
    ]
  };

  // Filter data based on selected filters
  const filteredData = useMemo(() => {
    const filterByLocation = (item: any) => selectedLocation === 'All' || item.location === selectedLocation;
    const filterByType = (item: any) => selectedPropertyType === 'All' || item.type === selectedPropertyType;
    const filterByDate = (item: any) => {
      if (dateRange === 'custom' && startDate && endDate) {
        const itemDate = new Date(item.date || item.listedDate);
        return itemDate >= new Date(startDate) && itemDate <= new Date(endDate);
      }
      const days = parseInt(dateRange);
      const cutoffDate = new Date();
      cutoffDate.setDate(cutoffDate.getDate() - days);
      return new Date(item.date || item.listedDate) >= cutoffDate;
    };

    return {
      properties: mockData.properties.filter(item => filterByLocation(item) && filterByType(item) && filterByDate(item)),
      inquiries: mockData.inquiries.filter(item => filterByDate(item)),
      visits: mockData.visits.filter(item => filterByDate(item)),
      revenue: mockData.revenue.filter(item => filterByDate(item))
    };
  }, [dateRange, selectedLocation, selectedPropertyType, startDate, endDate]);

  // Calculate summary statistics
  const summaryStats = useMemo(() => {
    const totalProperties = filteredData.properties.length;
    const activeProperties = filteredData.properties.filter(p => p.status === 'Active').length;
    const totalViews = filteredData.properties.reduce((sum, p) => sum + p.views, 0);
    const totalInquiries = filteredData.inquiries.length;
    const totalRevenue = filteredData.revenue.reduce((sum, r) => sum + (r.status === 'Completed' ? r.amount : 0), 0);
    const avgViewsPerProperty = totalProperties > 0 ? Math.round(totalViews / totalProperties) : 0;

    return {
      totalProperties,
      activeProperties,
      totalViews,
      totalInquiries,
      totalRevenue,
      avgViewsPerProperty
    };
  }, [filteredData]);

  const handleDownloadReport = async () => {
    setGenerating(true);
    
    // Simulate report generation delay
    await new Promise(resolve => setTimeout(resolve, 2000));
    
    const reportData = generateReportData();
    const blob = new Blob([reportData], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `${reportType.replace(/\s+/g, '_')}_${new Date().toISOString().split('T')[0]}.txt`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
    
    setGenerating(false);
  };

  const generateReportData = () => {
    const dateRangeText = dateRange === 'custom' && startDate && endDate 
      ? `${startDate} to ${endDate}` 
      : `Last ${dateRange} days`;

    let reportContent = `
${reportType.toUpperCase()}
Generated on: ${new Date().toLocaleDateString()}
Date Range: ${dateRangeText}
Location: ${selectedLocation}
Property Type: ${selectedPropertyType}

`;

    reportContent += `SUMMARY STATISTICS:\n`;
    reportContent += `- Total Properties: ${summaryStats.totalProperties}\n`;
    reportContent += `- Active Properties: ${summaryStats.activeProperties}\n`;
    reportContent += `- Total Views: ${summaryStats.totalViews}\n`;
    reportContent += `- Total Inquiries: ${summaryStats.totalInquiries}\n`;
    reportContent += `- Total Revenue: ₹${summaryStats.totalRevenue.toLocaleString()}\n`;
    reportContent += `- Avg Views per Property: ${summaryStats.avgViewsPerProperty}\n\n`;

    if (reportType === 'Property Performance Report') {
      reportContent += `PROPERTY DETAILS:\n`;
      filteredData.properties.forEach((property, index) => {
        reportContent += `${index + 1}. ${property.title} - ${property.location} - ₹${property.price.toLocaleString()} - ${property.views} views - ${property.inquiries} inquiries\n`;
      });
    } else if (reportType === 'Inquiry Report') {
      reportContent += `INQUIRY DETAILS:\n`;
      filteredData.inquiries.forEach((inquiry, index) => {
        reportContent += `${index + 1}. ${inquiry.propertyTitle} - ${inquiry.buyerName} - ${inquiry.phone} - ${inquiry.status} - ${inquiry.date}\n`;
      });
    } else if (reportType === 'Revenue Report') {
      reportContent += `REVENUE DETAILS:\n`;
      filteredData.revenue.forEach((revenue, index) => {
        reportContent += `${index + 1}. ${revenue.property} - ₹${revenue.amount.toLocaleString()} - ${revenue.status} - ${revenue.date}\n`;
      });
    } else if (reportType === 'Visit Report') {
      reportContent += `VISIT DETAILS:\n`;
      filteredData.visits.forEach((visit, index) => {
        reportContent += `${index + 1}. ${visit.property} - ${visit.buyer} - ${visit.date} ${visit.time} - ${visit.status}\n`;
      });
    }

    reportContent += `\n\nGenerated by NAL Seller Dashboard`;
    return reportContent;
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">Reports</h2>
          <p className="text-gray-600 mt-1">Generate and download detailed property reports</p>
        </div>
      </div>

      {/* Report Filters */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Generate Report</h3>
        <div className="flex flex-wrap items-end gap-4">
          <div className="flex-1 min-w-[150px]">
            <label className="block text-sm font-medium text-gray-700 mb-2">Report Type</label>
            <select 
              value={reportType}
              onChange={(e) => setReportType(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
              <option>Property Performance Report</option>
              <option>Inquiry Report</option>
              <option>Revenue Report</option>
              <option>Visit Report</option>
            </select>
          </div>
          <div className="flex-1 min-w-[150px]">
            <label className="block text-sm font-medium text-gray-700 mb-2">Date Range</label>
            <select 
              value={dateRange} 
              onChange={(e) => setDateRange(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
              <option value="7">Last 7 days</option>
              <option value="30">Last 30 days</option>
              <option value="90">Last 3 months</option>
              <option value="365">Last year</option>
              <option value="custom">Custom Range</option>
            </select>
          </div>
          {dateRange === 'custom' && (
            <>
              <div className="flex-1 min-w-[120px]">
                <label className="block text-sm font-medium text-gray-700 mb-2">Start Date</label>
                <input
                  type="date"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
              </div>
              <div className="flex-1 min-w-[120px]">
                <label className="block text-sm font-medium text-gray-700 mb-2">End Date</label>
                <input
                  type="date"
                  value={endDate}
                  onChange={(e) => setEndDate(e.target.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                />
              </div>
            </>
          )}
          <div className="flex-1 min-w-[150px]">
            <label className="block text-sm font-medium text-gray-700 mb-2">Location</label>
            <select 
              value={selectedLocation} 
              onChange={(e) => setSelectedLocation(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
              <option value="All">All Locations</option>
              <option value="Bandra West">Bandra West</option>
              <option value="Andheri West">Andheri West</option>
              <option value="Whitefield">Whitefield</option>
              <option value="Electronic City">Electronic City</option>
              <option value="Koramangala">Koramangala</option>
            </select>
          </div>
          <div className="flex-1 min-w-[150px]">
            <label className="block text-sm font-medium text-gray-700 mb-2">Property Type</label>
            <select 
              value={selectedPropertyType} 
              onChange={(e) => setSelectedPropertyType(e.target.value)}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
            >
              <option value="All">All Types</option>
              <option value="Apartment">Apartment</option>
              <option value="Villa">Villa</option>
              <option value="Plot">Plot</option>
              <option value="Commercial">Commercial</option>
            </select>
          </div>
        </div>
        <div className="mt-6">
          <Button 
            onClick={handleDownloadReport}
            disabled={generating}
            className="flex items-center space-x-2 px-6 py-3 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed font-medium shadow-sm"
          >
            {generating ? (
              <>
                <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                <span>Generating...</span>
              </>
            ) : (
              <>
                <Download className="w-4 h-4" />
                <span>Generate Report</span>
              </>
            )}
          </Button>
        </div>
      </div>

      {/* Report Summary */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Report Summary</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          <div className="bg-gradient-to-r from-blue-50 to-blue-100 p-4 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-blue-600 font-medium">Total Properties</p>
                <p className="text-2xl font-bold text-blue-900">{summaryStats.totalProperties}</p>
              </div>
              <Building className="w-8 h-8 text-blue-600" />
            </div>
          </div>
          <div className="bg-gradient-to-r from-green-50 to-green-100 p-4 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-green-600 font-medium">Total Views</p>
                <p className="text-2xl font-bold text-green-900">{summaryStats.totalViews}</p>
              </div>
              <Eye className="w-8 h-8 text-green-600" />
            </div>
          </div>
          <div className="bg-gradient-to-r from-purple-50 to-purple-100 p-4 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-purple-600 font-medium">Total Revenue</p>
                <p className="text-2xl font-bold text-purple-900">₹{(summaryStats.totalRevenue / 100000).toFixed(1)}L</p>
              </div>
              <DollarSign className="w-8 h-8 text-purple-600" />
            </div>
          </div>
        </div>
      </div>

      {/* Detailed Report Data */}
      <div className="bg-white rounded-xl shadow-sm p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">{reportType} Details</h3>
        
        {reportType === 'Property Performance Report' && (
          <div className="overflow-x-auto">
            <table className="w-full text-sm">
              <thead>
                <tr className="border-b border-gray-200">
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Property</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Location</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Price</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Views</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Inquiries</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Status</th>
                </tr>
              </thead>
              <tbody>
                {filteredData.properties.map((property) => (
                  <tr key={property.id} className="border-b border-gray-100 hover:bg-gray-50">
                    <td className="py-3 px-4 font-medium text-gray-900">{property.title}</td>
                    <td className="py-3 px-4 text-gray-600">{property.location}</td>
                    <td className="py-3 px-4 text-gray-600">₹{property.price.toLocaleString()}</td>
                    <td className="py-3 px-4 text-blue-600 font-medium">{property.views}</td>
                    <td className="py-3 px-4 text-green-600 font-medium">{property.inquiries}</td>
                    <td className="py-3 px-4">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        property.status === 'Active' ? 'bg-green-100 text-green-800' :
                        property.status === 'Sold' ? 'bg-blue-100 text-blue-800' :
                        'bg-gray-100 text-gray-800'
                      }`}>
                        {property.status}
                      </span>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}

        {reportType === 'Inquiry Report' && (
          <div className="overflow-x-auto">
            <table className="w-full text-sm">
              <thead>
                <tr className="border-b border-gray-200">
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Property</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Buyer Name</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Phone</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Status</th>
                  <th className="text-left py-3 px-4 font-medium text-gray-900">Date</th>
                </tr>
              </thead>
              <tbody>
                {filteredData.inquiries.map((inquiry) => (
                  <tr key={inquiry.id} className="border-b border-gray-100 hover:bg-gray-50">
                    <td className="py-3 px-4 font-medium text-gray-900">{inquiry.propertyTitle}</td>
                    <td className="py-3 px-4 text-gray-600">{inquiry.buyerName}</td>
                    <td className="py-3 px-4 text-gray-600">{inquiry.phone}</td>
                    <td className="py-3 px-4">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        inquiry.status === 'Responded' ? 'bg-green-100 text-green-800' :
                        'bg-yellow-100 text-yellow-800'
                      }`}>
                        {inquiry.status}
                      </span>
                    </td>
                    <td className="py-3 px-4 text-gray-600">{inquiry.date}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>
    </div>
  );
};

export default Reports;