import { useState } from 'react';
import { useParams, Link } from 'wouter';
import { ArrowLeft, Upload, CheckCircle, Clock, FileText, MessageSquare } from 'lucide-react';

export default function RequestDetails() {
  const params = useParams();
  const id = params.id;
  const [selectedFiles, setSelectedFiles] = useState<{ [key: string]: File | null }>({});

  // Mock data
  const request = {
    id: 1,
    propertyTitle: 'Modern 2BR Apartment - Downtown',
    buyerName: 'Rajesh Kumar',
    buyerEmail: 'rajesh@example.com',
    requestDate: '2024-01-15',
    status: 'pending_documents',
    verificationTeamNotes: 'Please upload clear scanned copies of all documents.'
  };

  const requiredDocuments = [
    { id: 'sale_deed', name: 'Sale Deed', uploaded: true, status: 'verified' },
    { id: 'title_deed', name: 'Title Deed', uploaded: true, status: 'pending_review' },
    { id: 'tax_receipts', name: 'Property Tax Receipts', uploaded: false, status: null },
    { id: 'noc', name: 'NOC Certificates', uploaded: false, status: null },
    { id: 'encumbrance', name: 'Encumbrance Certificate', uploaded: false, status: null }
  ];

  const handleFileSelect = (docId: string, file: File) => {
    setSelectedFiles({ ...selectedFiles, [docId]: file });
  };

  const handleUpload = (docId: string) => {
    console.log('Uploading:', docId, selectedFiles[docId]);
    // API call will be added later
  };

  const getStatusBadge = (status: string | null) => {
    if (!status) return null;
    switch (status) {
      case 'verified':
        return <span className="text-xs px-2 py-1 bg-green-100 text-green-800 rounded-full">Verified</span>;
      case 'pending_review':
        return <span className="text-xs px-2 py-1 bg-yellow-100 text-yellow-800 rounded-full">Pending Review</span>;
      case 'rejected':
        return <span className="text-xs px-2 py-1 bg-red-100 text-red-800 rounded-full">Rejected</span>;
      default:
        return null;
    }
  };

  return (
    <div className="flex-1 p-8 overflow-y-auto">
      <Link href="/legal-health-reports" className="inline-flex items-center gap-2 text-sm text-gray-600 hover:text-gray-900 mb-4">
        <ArrowLeft className="w-4 h-4" />
        Back to Requests
      </Link>

      <div className="mb-6">
        <h1 className="text-2xl font-bold text-gray-900">{request.propertyTitle}</h1>
        <p className="text-sm text-gray-600 mt-1">Legal Health Report Request</p>
      </div>

      <div className="grid grid-cols-3 gap-6">
        {/* Left Column - Request Info */}
        <div className="col-span-2 space-y-6">
          {/* Buyer Info */}
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <h2 className="text-sm font-semibold text-gray-900 mb-3">Buyer Information</h2>
            <div className="space-y-2 text-sm">
              <div className="flex justify-between">
                <span className="text-gray-600">Name:</span>
                <span className="text-gray-900 font-medium">{request.buyerName}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Email:</span>
                <span className="text-gray-900">{request.buyerEmail}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Request Date:</span>
                <span className="text-gray-900">{new Date(request.requestDate).toLocaleDateString()}</span>
              </div>
            </div>
          </div>

          {/* Documents Upload */}
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <h2 className="text-sm font-semibold text-gray-900 mb-3">Required Documents</h2>
            <div className="space-y-3">
              {requiredDocuments.map((doc) => (
                <div key={doc.id} className="border border-gray-200 rounded-lg p-3">
                  <div className="flex items-center justify-between mb-2">
                    <div className="flex items-center gap-2">
                      {doc.uploaded ? (
                        <CheckCircle className="w-4 h-4 text-green-600" />
                      ) : (
                        <Clock className="w-4 h-4 text-gray-400" />
                      )}
                      <span className="text-sm font-medium text-gray-900">{doc.name}</span>
                    </div>
                    {getStatusBadge(doc.status)}
                  </div>
                  
                  {!doc.uploaded && (
                    <div className="flex gap-2">
                      <input
                        type="file"
                        onChange={(e) => e.target.files && handleFileSelect(doc.id, e.target.files[0])}
                        className="text-xs flex-1"
                        accept=".pdf,.jpg,.jpeg,.png"
                      />
                      <button
                        onClick={() => handleUpload(doc.id)}
                        disabled={!selectedFiles[doc.id]}
                        className="px-3 py-1 text-xs bg-blue-600 text-white rounded hover:bg-blue-700 disabled:bg-gray-300 disabled:cursor-not-allowed"
                      >
                        Upload
                      </button>
                    </div>
                  )}
                  
                  {doc.uploaded && (
                    <button className="text-xs text-blue-600 hover:text-blue-800">
                      View Document
                    </button>
                  )}
                </div>
              ))}
            </div>
          </div>

          {/* Verification Team Notes */}
          {request.verificationTeamNotes && (
            <div className="bg-blue-50 p-4 rounded-lg border border-blue-200">
              <div className="flex items-start gap-2">
                <MessageSquare className="w-4 h-4 text-blue-600 mt-0.5" />
                <div>
                  <h3 className="text-sm font-semibold text-blue-900 mb-1">Verification Team Notes</h3>
                  <p className="text-sm text-blue-800">{request.verificationTeamNotes}</p>
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Right Column - Status & Actions */}
        <div className="space-y-6">
          {/* Status Card */}
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <h2 className="text-sm font-semibold text-gray-900 mb-3">Verification Status</h2>
            <div className="space-y-3">
              <div className="flex items-center gap-2">
                <div className="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center">
                  <CheckCircle className="w-4 h-4 text-green-600" />
                </div>
                <div>
                  <div className="text-xs font-medium text-gray-900">Request Received</div>
                  <div className="text-xs text-gray-500">Jan 15, 2024</div>
                </div>
              </div>
              <div className="flex items-center gap-2">
                <div className="w-8 h-8 rounded-full bg-yellow-100 flex items-center justify-center">
                  <Clock className="w-4 h-4 text-yellow-600" />
                </div>
                <div>
                  <div className="text-xs font-medium text-gray-900">Documents Upload</div>
                  <div className="text-xs text-gray-500">In Progress</div>
                </div>
              </div>
              <div className="flex items-center gap-2">
                <div className="w-8 h-8 rounded-full bg-gray-100 flex items-center justify-center">
                  <FileText className="w-4 h-4 text-gray-400" />
                </div>
                <div>
                  <div className="text-xs font-medium text-gray-400">Verification</div>
                  <div className="text-xs text-gray-400">Pending</div>
                </div>
              </div>
            </div>
          </div>

          {/* Quick Actions */}
          <div className="bg-white p-4 rounded-lg border border-gray-200">
            <h2 className="text-sm font-semibold text-gray-900 mb-3">Quick Actions</h2>
            <div className="space-y-2">
              <button className="w-full px-3 py-2 text-sm bg-blue-600 text-white rounded hover:bg-blue-700">
                Contact Verification Team
              </button>
              <button className="w-full px-3 py-2 text-sm bg-white text-gray-700 border border-gray-200 rounded hover:bg-gray-50">
                Download All Documents
              </button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
