import React, { useState, useEffect, useMemo } from 'react';
import { useLocation } from 'wouter';
import { toast } from 'sonner';
import { Property } from '../types';
import { api } from '../utils/api';
import PremiumMobileCard from '../components/PremiumMobileCard';
import PremiumPropertyCard from '../components/PremiumPropertyCard';
import PremiumEmptyState from '../components/PremiumEmptyState';
import PropertyDetailsModal from '../components/PropertyDetailsModal';
import DeleteConfirmModal from '../components/DeleteConfirmModal';
import { Button } from '../components/ui/button';
import { Input } from '../components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../components/ui/select';
import { Plus, Search, Filter, Grid3X3, List } from 'lucide-react';

type ViewMode = 'grid' | 'list';
type SortField = 'title' | 'price' | 'createdAt' | 'views';
type SortOrder = 'asc' | 'desc';

const ResponsiveManageProperty: React.FC = () => {
  const [location, navigate] = useLocation();
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [isMobile, setIsMobile] = useState(window.innerWidth < 768);
  const [filters, setFilters] = useState({
    search: '',
    propertyType: 'all',
    status: 'all',
    priceRange: 'all'
  });
  const [showFilters, setShowFilters] = useState(false);
  const [viewMode, setViewMode] = useState<ViewMode>('grid');
  const [sortField, setSortField] = useState<SortField>('createdAt');
  const [sortOrder, setSortOrder] = useState<SortOrder>('desc');
  const [selectedProperty, setSelectedProperty] = useState<Property | null>(null);
  const [showDetailsModal, setShowDetailsModal] = useState(false);
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [isDeleting, setIsDeleting] = useState(false);

  const loadProperties = async (showLoading = true) => {
    try {
      if (showLoading) setLoading(true);
      const data = await api.fetchProperties();
      setProperties(data);
      setLoading(false);
    } catch (error) {
      console.error('Error loading properties:', error);
      setProperties([]);
      setLoading(false);
    }
  };

  useEffect(() => {
    loadProperties();
    
    const handleResize = () => setIsMobile(window.innerWidth < 768);
    window.addEventListener('resize', handleResize);
    
    // Auto-refresh without loading state
    const interval = setInterval(() => loadProperties(false), 10000);
    const handleFocus = () => loadProperties(false);
    window.addEventListener('focus', handleFocus);
    
    // Listen for property added event
    const handlePropertyAdded = (event: CustomEvent) => {
      loadProperties(false);
      if (event.detail) {
        toast.success('Property list updated!', {
          description: 'Your new property is now visible in the list.',
          duration: 3000,
        });
      }
    };
    window.addEventListener('propertyAdded', handlePropertyAdded as EventListener);
    
    return () => {
      window.removeEventListener('resize', handleResize);
      clearInterval(interval);
      window.removeEventListener('focus', handleFocus);
      window.removeEventListener('propertyAdded', handlePropertyAdded as EventListener);
    };
  }, []);

  const filteredAndSortedProperties = useMemo(() => {
    let filtered = properties.filter((property) => {
      const matchesSearch = !filters.search || 
        property.title.toLowerCase().includes(filters.search.toLowerCase()) ||
        property.address.toLowerCase().includes(filters.search.toLowerCase()) ||
        property.city.toLowerCase().includes(filters.search.toLowerCase());
      
      const matchesType = filters.propertyType === 'all' || property.type === filters.propertyType;
      const matchesStatus = filters.status === 'all' || property.status === filters.status;
      
      let matchesPrice = true;
      if (filters.priceRange !== 'all') {
        const price = property.price;
        switch (filters.priceRange) {
          case 'under-50l': matchesPrice = price < 5000000; break;
          case '50l-1cr': matchesPrice = price >= 5000000 && price < 10000000; break;
          case '1cr-2cr': matchesPrice = price >= 10000000 && price < 20000000; break;
          case 'above-2cr': matchesPrice = price >= 20000000; break;
        }
      }

      return matchesSearch && matchesType && matchesStatus && matchesPrice;
    });

    filtered.sort((a, b) => {
      let aValue: any = a[sortField];
      let bValue: any = b[sortField];

      if (sortField === 'price' || sortField === 'views') {
        aValue = Number(aValue);
        bValue = Number(bValue);
      } else if (sortField === 'createdAt') {
        aValue = new Date(aValue);
        bValue = new Date(bValue);
      } else {
        aValue = String(aValue).toLowerCase();
        bValue = String(bValue).toLowerCase();
      }

      return sortOrder === 'asc' ? (aValue > bValue ? 1 : -1) : (aValue < bValue ? 1 : -1);
    });

    return filtered;
  }, [properties, filters, sortField, sortOrder]);

  const handleEdit = (id: string) => {
    navigate(`/edit-property?id=${id}`);
  };

  const handleDelete = (id: string) => {
    const property = properties.find(p => p.id === id);
    if (property) {
      setSelectedProperty(property);
      setShowDeleteModal(true);
    }
  };

  const confirmDelete = async () => {
    if (!selectedProperty) return;
    
    setIsDeleting(true);
    try {
      await api.deleteProperty(selectedProperty.id);
      setProperties(prev => prev.filter(p => p.id !== selectedProperty.id));
      setShowDeleteModal(false);
      setSelectedProperty(null);
      
      toast.success('Property deleted successfully', {
        description: 'The property has been removed from your listings.',
        duration: 3000,
      });
    } catch (error: any) {
      toast.error('Failed to delete property', {
        description: error.message || 'Please try again later.',
        duration: 4000,
      });
    } finally {
      setIsDeleting(false);
    }
  };

  const handleView = (id: string) => {
    const property = properties.find(p => p.id === id);
    if (property) {
      setSelectedProperty(property);
      setShowDetailsModal(true);
    }
  };

  const handleAddProperty = () => {
    navigate('/list-property');
  };

  const clearFilters = () => {
    setFilters({
      search: '',
      propertyType: 'all',
      status: 'all',
      priceRange: 'all'
    });
  };

  if (loading) {
    return <div className="flex-1 bg-gray-50"></div>;
  }

  // Mobile view
  if (isMobile) {
    if (properties.length === 0) {
      return <PremiumEmptyState onAddProperty={handleAddProperty} isMobile={true} />;
    }
    return (
      <div className="min-h-screen bg-gray-50">
        <div className="bg-white border-b border-gray-200 p-4">
          <h1 className="text-xl font-bold text-gray-900 text-center">My Properties</h1>
        </div>
        <div className="py-4">
          {filteredAndSortedProperties.map((property) => (
            <PremiumMobileCard
              key={property.id}
              property={property}
              onView={handleView}
              onEdit={handleEdit}
              onDelete={handleDelete}
            />
          ))}
        </div>
      </div>
    );
  }

  // Web view - Empty state
  if (properties.length === 0) {
    return <PremiumEmptyState onAddProperty={handleAddProperty} isMobile={false} />;
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-gray-50 to-blue-50">
      {/* Premium Header */}
      <div className="bg-white shadow-sm border-b border-gray-200">
        <div className="max-w-7xl mx-auto px-6 py-6">
          <div className="flex items-center justify-between mb-6">
            <div>
              <h1 className="text-3xl font-bold text-gray-900">Manage Properties</h1>
              <p className="text-gray-600 mt-1">Manage your property listings and track performance</p>
            </div>
            <button 
              onClick={handleAddProperty}
              className="bg-gradient-to-r from-blue-600 to-indigo-600 text-white px-6 py-3 rounded-xl font-semibold hover:shadow-lg transition-all duration-300 flex items-center gap-2"
            >
              <Plus className="w-5 h-5" />
              Add New Property
            </button>
          </div>
          
          {/* Search and Filters */}
          <div className="flex items-center gap-4">
            <Input
              placeholder="Search properties by title, location..."
              value={filters.search}
              onChange={(e) => setFilters({ ...filters, search: e.target.value })}
              className="flex-1 max-w-md h-12 border-gray-300 focus:border-blue-500 focus:ring-2 focus:ring-blue-500/20 rounded-lg"
            />
              
              <Select value={filters.propertyType} onValueChange={(value) => setFilters({ ...filters, propertyType: value })}>
                <SelectTrigger className="w-36 h-11">
                  <SelectValue placeholder="All Types" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="apartment">Apartment</SelectItem>
                  <SelectItem value="house">House</SelectItem>
                  <SelectItem value="villa">Villa</SelectItem>
                  <SelectItem value="plot">Plot</SelectItem>
                </SelectContent>
              </Select>

              <Select value={filters.status} onValueChange={(value) => setFilters({ ...filters, status: value })}>
                <SelectTrigger className="w-32 h-11">
                  <SelectValue placeholder="All Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="inactive">Inactive</SelectItem>
                  <SelectItem value="under_review">Under Review</SelectItem>
                </SelectContent>
              </Select>

              <Select value={filters.priceRange} onValueChange={(value) => setFilters({ ...filters, priceRange: value })}>
                <SelectTrigger className="w-36 h-11">
                  <SelectValue placeholder="All Prices" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Prices</SelectItem>
                  <SelectItem value="under-50l">Under ₹50L</SelectItem>
                  <SelectItem value="50l-1cr">₹50L - ₹1Cr</SelectItem>
                  <SelectItem value="1cr-2cr">₹1Cr - ₹2Cr</SelectItem>
                  <SelectItem value="above-2cr">Above ₹2Cr</SelectItem>
                </SelectContent>
              </Select>

              <Button
                variant="outline"
                onClick={() => setShowFilters(!showFilters)}
                className="h-11 border-2"
              >
                <Filter className="w-4 h-4 mr-2" />
                More Filters
              </Button>
            </div>

            <Button
              variant="outline"
              onClick={() => setShowFilters(!showFilters)}
              className="h-12 px-4 border-gray-300 hover:bg-gray-50"
            >
              <Filter className="w-4 h-4 mr-2" />
              {showFilters ? 'Hide Filters' : 'More Filters'}
            </Button>
          </div>

          {showFilters && (
            <div className="mt-4 pt-4 border-t">
              <div className="flex gap-4">
                <div className="flex border rounded-lg">
                  <Button
                    variant={viewMode === 'grid' ? 'default' : 'ghost'}
                    size="sm"
                    onClick={() => setViewMode('grid')}
                    className="rounded-r-none h-10"
                  >
                    <Grid3X3 className="w-4 h-4" />
                  </Button>
                  <Button
                    variant={viewMode === 'list' ? 'default' : 'ghost'}
                    size="sm"
                    onClick={() => setViewMode('list')}
                    className="rounded-l-none h-10"
                  >
                    <List className="w-4 h-4" />
                  </Button>
                </div>

                <Select value={`${sortField}-${sortOrder}`} onValueChange={(value) => {
                  const [field, order] = value.split('-');
                  setSortField(field as SortField);
                  setSortOrder(order as SortOrder);
                }}>
                  <SelectTrigger className="w-40 h-10">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="createdAt-desc">Newest First</SelectItem>
                    <SelectItem value="createdAt-asc">Oldest First</SelectItem>
                    <SelectItem value="price-desc">Price: High to Low</SelectItem>
                    <SelectItem value="price-asc">Price: Low to High</SelectItem>
                    <SelectItem value="title-asc">Name: A to Z</SelectItem>
                    <SelectItem value="views-desc">Most Viewed</SelectItem>
                  </SelectContent>
                </Select>

                <Button variant="outline" onClick={clearFilters} className="h-10">
                  Clear Filters
                </Button>
              </div>
            </div>
          )}
        </div>

      {/* Properties Content */}
      <div className="max-w-7xl mx-auto px-6 py-8">
        {filteredAndSortedProperties.length === 0 ? (
          <PremiumEmptyState onAddProperty={handleAddProperty} type="no-results" />
        ) : (
          <>
            {/* Results Header */}
            <div className="flex items-center justify-between mb-6">
              <div className="text-sm text-gray-600">
                Showing {filteredAndSortedProperties.length} of {properties.length} properties
              </div>
              <div className="flex items-center gap-3">
                <div className="flex border border-gray-300 rounded-lg overflow-hidden">
                  <button
                    onClick={() => setViewMode('grid')}
                    className={`px-4 py-2 text-sm font-medium transition-colors ${
                      viewMode === 'grid' 
                        ? 'bg-blue-600 text-white' 
                        : 'bg-white text-gray-700 hover:bg-gray-50'
                    }`}
                  >
                    <Grid3X3 className="w-4 h-4" />
                  </button>
                  <button
                    onClick={() => setViewMode('list')}
                    className={`px-4 py-2 text-sm font-medium transition-colors ${
                      viewMode === 'list' 
                        ? 'bg-blue-600 text-white' 
                        : 'bg-white text-gray-700 hover:bg-gray-50'
                    }`}
                  >
                    <List className="w-4 h-4" />
                  </button>
                </div>
              </div>
            </div>
            
            {/* Properties Grid */}
            <div className={viewMode === 'grid' 
              ? "grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-6" 
              : "grid grid-cols-1 lg:grid-cols-2 gap-6"
            }>
              {filteredAndSortedProperties.map((property) => (
                <PremiumPropertyCard
                  key={property.id}
                  property={property}
                  onView={handleView}
                  onEdit={handleEdit}
                  onDelete={handleDelete}
                />
              ))}
            </div>
          </>
        )}
      </div>

      {/* Property Details Modal */}
      <PropertyDetailsModal
        property={selectedProperty}
        isOpen={showDetailsModal}
        onClose={() => {
          setShowDetailsModal(false);
          setSelectedProperty(null);
        }}
        onEdit={(id) => {
          setShowDetailsModal(false);
          setSelectedProperty(null);
          handleEdit(id);
        }}
      />

      {/* Delete Confirmation Modal */}
      <DeleteConfirmModal
        property={selectedProperty}
        isOpen={showDeleteModal}
        onClose={() => {
          setShowDeleteModal(false);
          setSelectedProperty(null);
        }}
        onConfirm={confirmDelete}
        isDeleting={isDeleting}
      />
    </div>
  );
};

export default ResponsiveManageProperty;