import React, { useEffect, useState } from 'react';
import { useParams, useLocation } from 'wouter';
import { transactionApi } from '@/api/transactions';
import { Transaction, TimelineEvent } from '@/types/transaction';
import { TimelineView } from '@/components/transaction-timeline/TimelineView';
import { ArrowLeft } from 'lucide-react';

export const SellerTransactionDetail: React.FC = () => {
  const params = useParams<{ transactionId: string }>();
  const transactionId = params.transactionId;
  const [, setLocation] = useLocation();
  const [transaction, setTransaction] = useState<Transaction | null>(null);
  const [timeline, setTimeline] = useState<TimelineEvent[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    console.log('Current URL:', window.location.href);
    console.log('Current pathname:', window.location.pathname);
    console.log('Transaction ID from params:', transactionId);
    if (transactionId) {
      fetchTransactionDetails();
    } else {
      console.error('No transaction ID provided');
      setLoading(false);
    }
  }, [transactionId]);

  const fetchTransactionDetails = async () => {
    console.log('Fetching transaction details for ID:', transactionId);
    try {
      console.log('Calling transactionApi.getById...');
      const txnData = await transactionApi.getById(transactionId!);
      console.log('Transaction data received:', txnData);
      
      console.log('Calling transactionApi.getTimeline...');
      const timelineData = await transactionApi.getTimeline(transactionId!);
      console.log('Timeline data received:', timelineData);
      
      setTransaction(txnData);
      setTimeline(timelineData);
    } catch (error) {
      console.error('Failed to fetch transaction details:', error);
    } finally {
      console.log('Setting loading to false');
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100 flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-green-600 mx-auto mb-4"></div>
          <p className="text-gray-600 font-medium">Loading transaction details...</p>
        </div>
      </div>
    );
  }

  if (!transaction) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100 flex items-center justify-center">
        <div className="text-center">
          <svg className="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
          </svg>
          <p className="text-gray-600 font-medium text-lg">Transaction not found</p>
        </div>
      </div>
    );
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed': return 'bg-green-100 text-green-800 border-green-200';
      case 'escrow_funded': return 'bg-blue-100 text-blue-800 border-blue-200';
      case 'pending': return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'failed': return 'bg-red-100 text-red-800 border-red-200';
      default: return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-gray-50 to-gray-100">
      <div className="container mx-auto px-4 py-8 max-w-7xl">
        {/* Back Button */}
        <button
          onClick={() => setLocation('/financials')}
          className="flex items-center gap-2 text-gray-600 hover:text-gray-900 mb-4 font-medium"
        >
          <ArrowLeft className="w-4 h-4" />
          Back to Financials
        </button>

        {/* Header */}
        <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-6">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 mb-2">Payment Received</h1>
              <div className="flex items-center gap-3">
                <p className="text-gray-600 font-mono text-sm">{transaction.transaction_number}</p>
                <span className={`px-3 py-1 rounded-full text-xs font-semibold border ${getStatusColor(transaction.transaction_status)}`}>
                  {transaction.transaction_status === 'escrow_funded' ? 'IN ESCROW' : transaction.transaction_status.toUpperCase()}
                </span>
              </div>
            </div>
            <div className="text-right">
              <p className="text-sm text-gray-500 mb-1">Amount Received</p>
              <p className="text-3xl font-bold text-green-600">₹{transaction.total_amount.toLocaleString()}</p>
            </div>
          </div>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-12 gap-6">
          {/* Left Side - Transaction Info */}
          <div className="lg:col-span-4 space-y-6">
            {/* Transaction Information Card */}
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
              <div className="flex items-center gap-2 mb-6 pb-4 border-b">
                <svg className="w-5 h-5 text-green-600" fill="currentColor" viewBox="0 0 20 20">
                  <path d="M9 2a1 1 0 000 2h2a1 1 0 100-2H9z" />
                  <path fillRule="evenodd" d="M4 5a2 2 0 012-2 3 3 0 003 3h2a3 3 0 003-3 2 2 0 012 2v11a2 2 0 01-2 2H6a2 2 0 01-2-2V5zm3 4a1 1 0 000 2h.01a1 1 0 100-2H7zm3 0a1 1 0 000 2h3a1 1 0 100-2h-3zm-3 4a1 1 0 100 2h.01a1 1 0 100-2H7zm3 0a1 1 0 100 2h3a1 1 0 100-2h-3z" clipRule="evenodd" />
                </svg>
                <h2 className="text-lg font-semibold text-gray-900">Payment Information</h2>
              </div>
              
              <div className="space-y-4">
                <div className="bg-gray-50 rounded-lg p-4">
                  <label className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-2 block">Transaction Type</label>
                  <p className="text-base font-semibold text-gray-900">Property Sale</p>
                </div>
                
                <div className="grid grid-cols-2 gap-4">
                  <div className="bg-green-50 rounded-lg p-4">
                    <label className="text-xs font-medium text-green-600 uppercase tracking-wide mb-2 block">Base Amount</label>
                    <p className="text-lg font-bold text-green-900">₹{transaction.amount.toLocaleString()}</p>
                  </div>
                  
                  <div className="bg-blue-50 rounded-lg p-4">
                    <label className="text-xs font-medium text-blue-600 uppercase tracking-wide mb-2 block">Tax & Fees</label>
                    <p className="text-lg font-bold text-blue-900">₹{(transaction.tax_amount + transaction.processing_fee).toLocaleString()}</p>
                  </div>
                </div>
                
                <div className="bg-gray-50 rounded-lg p-4">
                  <label className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-2 block">Payment Method</label>
                  <div className="flex items-center gap-2">
                    <svg className="w-5 h-5 text-gray-600" fill="currentColor" viewBox="0 0 20 20">
                      <path d="M4 4a2 2 0 00-2 2v1h16V6a2 2 0 00-2-2H4z" />
                      <path fillRule="evenodd" d="M18 9H2v5a2 2 0 002 2h12a2 2 0 002-2V9zM4 13a1 1 0 011-1h1a1 1 0 110 2H5a1 1 0 01-1-1zm5-1a1 1 0 100 2h1a1 1 0 100-2H9z" clipRule="evenodd" />
                    </svg>
                    <p className="text-base font-semibold text-gray-900 capitalize">{transaction.payment_method.replace(/_/g, ' ')}</p>
                  </div>
                </div>
                
                <div className="bg-gray-50 rounded-lg p-4">
                  <label className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-2 block">Payment Date</label>
                  <p className="text-sm font-medium text-gray-900">{new Date(transaction.created_at).toLocaleString('en-IN', { dateStyle: 'long', timeStyle: 'short' })}</p>
                </div>

                {transaction.description && (
                  <div className="bg-gray-50 rounded-lg p-4">
                    <label className="text-xs font-medium text-gray-500 uppercase tracking-wide mb-2 block">Description</label>
                    <p className="text-sm text-gray-900">{transaction.description}</p>
                  </div>
                )}
              </div>
            </div>

            {/* Amount Breakdown Card */}
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
              <div className="flex items-center gap-2 mb-6 pb-4 border-b">
                <svg className="w-5 h-5 text-purple-600" fill="currentColor" viewBox="0 0 20 20">
                  <path d="M8.433 7.418c.155-.103.346-.196.567-.267v1.698a2.305 2.305 0 01-.567-.267C8.07 8.34 8 8.114 8 8c0-.114.07-.34.433-.582zM11 12.849v-1.698c.22.071.412.164.567.267.364.243.433.468.433.582 0 .114-.07.34-.433.582a2.305 2.305 0 01-.567.267z" />
                  <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-13a1 1 0 10-2 0v.092a4.535 4.535 0 00-1.676.662C6.602 6.234 6 7.009 6 8c0 .99.602 1.765 1.324 2.246.48.32 1.054.545 1.676.662v1.941c-.391-.127-.68-.317-.843-.504a1 1 0 10-1.51 1.31c.562.649 1.413 1.076 2.353 1.253V15a1 1 0 102 0v-.092a4.535 4.535 0 001.676-.662C13.398 13.766 14 12.991 14 12c0-.99-.602-1.765-1.324-2.246A4.535 4.535 0 0011 9.092V7.151c.391.127.68.317.843.504a1 1 0 101.511-1.31c-.563-.649-1.413-1.076-2.354-1.253V5z" clipRule="evenodd" />
                </svg>
                <h2 className="text-lg font-semibold text-gray-900">Amount Breakdown</h2>
              </div>
              
              <div className="space-y-3">
                <div className="flex justify-between text-sm">
                  <span className="text-gray-600">Base Amount</span>
                  <span className="font-semibold text-gray-900">₹{transaction.amount.toLocaleString()}</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-gray-600">Tax (GST 18%)</span>
                  <span className="font-semibold text-gray-900">₹{transaction.tax_amount.toLocaleString()}</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-gray-600">Processing Fee</span>
                  <span className="font-semibold text-gray-900">₹{transaction.processing_fee.toLocaleString()}</span>
                </div>
                <div className="border-t pt-3 flex justify-between">
                  <span className="font-bold text-gray-900">Total Received</span>
                  <span className="font-bold text-green-600 text-lg">₹{transaction.total_amount.toLocaleString()}</span>
                </div>
              </div>
            </div>
          </div>

          {/* Right Side - Timeline */}
          <div className="lg:col-span-8">
            <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6">
              <div className="flex items-center gap-2 mb-6 pb-4 border-b">
                <svg className="w-5 h-5 text-indigo-600" fill="currentColor" viewBox="0 0 20 20">
                  <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clipRule="evenodd" />
                </svg>
                <h2 className="text-lg font-semibold text-gray-900">Payment Timeline</h2>
              </div>
              <div className="max-h-[600px] overflow-y-auto pr-2">
                <TimelineView events={timeline} />
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};
