import React, { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import { Input } from "../components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "../components/ui/dialog";
import { useToast } from "../hooks/use-toast";
import { 
  FileText, 
  Upload, 
  Download, 
  Share2,
  Eye,
  Edit,
  Trash2,
  Search,
  Filter,
  Calendar,
  Clock,
  CheckCircle,
  AlertTriangle,
  XCircle,
  Shield,
  Lock,
  Unlock,
  Star,
  Tag,
  Users,
  Building,
  DollarSign,
  Award,
  Zap
} from "lucide-react";

// Smart document data with AI categorization
const mockDocuments = [
  {
    id: "doc-1",
    name: "Property_Deed_Downtown_Apt.pdf",
    type: "Legal",
    category: "Property Deed",
    propertyId: "prop-1",
    propertyName: "Modern Downtown Apartment",
    size: "2.4 MB",
    uploadDate: new Date("2024-12-10T09:30:00"),
    lastModified: new Date("2024-12-12T14:20:00"),
    status: "verified",
    aiConfidence: 98,
    tags: ["Legal", "Ownership", "Critical"],
    sharedWith: ["buyer-1", "lawyer-1"],
    expiryDate: null,
    isEncrypted: true,
    accessLevel: "restricted",
    downloadCount: 5,
    viewCount: 12,
    aiSummary: "Property ownership document for downtown apartment. All legal requirements verified.",
    requiredFor: ["Sale", "Mortgage", "Insurance"]
  },
  {
    id: "doc-2",
    name: "Home_Inspection_Report_2024.pdf",
    type: "Inspection",
    category: "Inspection Report",
    propertyId: "prop-1",
    propertyName: "Modern Downtown Apartment",
    size: "5.8 MB",
    uploadDate: new Date("2024-12-08T11:15:00"),
    lastModified: new Date("2024-12-08T11:15:00"),
    status: "pending_review",
    aiConfidence: 95,
    tags: ["Inspection", "Maintenance", "Important"],
    sharedWith: ["buyer-1"],
    expiryDate: new Date("2025-12-08T11:15:00"),
    isEncrypted: false,
    accessLevel: "public",
    downloadCount: 8,
    viewCount: 23,
    aiSummary: "Comprehensive inspection report. Minor issues identified in electrical system.",
    requiredFor: ["Sale", "Insurance"]
  },
  {
    id: "doc-3",
    name: "Property_Valuation_Certificate.pdf",
    type: "Financial",
    category: "Valuation",
    propertyId: "prop-2",
    propertyName: "Luxury Beachfront Villa",
    size: "1.2 MB",
    uploadDate: new Date("2024-12-05T16:45:00"),
    lastModified: new Date("2024-12-05T16:45:00"),
    status: "verified",
    aiConfidence: 99,
    tags: ["Valuation", "Financial", "Critical"],
    sharedWith: ["bank-1", "buyer-2"],
    expiryDate: new Date("2025-06-05T16:45:00"),
    isEncrypted: true,
    accessLevel: "restricted",
    downloadCount: 3,
    viewCount: 7,
    aiSummary: "Professional property valuation at $850,000. Market analysis included.",
    requiredFor: ["Sale", "Mortgage", "Tax"]
  },
  {
    id: "doc-4",
    name: "Insurance_Policy_2024.pdf",
    type: "Insurance",
    category: "Insurance Policy",
    propertyId: "prop-1",
    propertyName: "Modern Downtown Apartment",
    size: "3.1 MB",
    uploadDate: new Date("2024-11-28T13:20:00"),
    lastModified: new Date("2024-11-28T13:20:00"),
    status: "expiring_soon",
    aiConfidence: 92,
    tags: ["Insurance", "Expiring", "Renewal"],
    sharedWith: ["insurance-agent"],
    expiryDate: new Date("2024-12-28T13:20:00"),
    isEncrypted: false,
    accessLevel: "private",
    downloadCount: 2,
    viewCount: 4,
    aiSummary: "Property insurance policy expiring soon. Renewal required by Dec 28.",
    requiredFor: ["Sale", "Mortgage"]
  }
];

export default function SmartDocuments() {
  const [selectedDoc, setSelectedDoc] = useState<any>(null);
  const [isDetailsOpen, setIsDetailsOpen] = useState(false);
  const [filterType, setFilterType] = useState("all");
  const [filterStatus, setFilterStatus] = useState("all");
  const [searchTerm, setSearchTerm] = useState("");
  const [isUploadOpen, setIsUploadOpen] = useState(false);
  const { toast } = useToast();

  const getStatusColor = (status: string) => {
    switch (status) {
      case "verified":
        return "bg-green-100 text-green-800 border-green-200";
      case "pending_review":
        return "bg-yellow-100 text-yellow-800 border-yellow-200";
      case "expiring_soon":
        return "bg-orange-100 text-orange-800 border-orange-200";
      case "expired":
        return "bg-red-100 text-red-800 border-red-200";
      default:
        return "bg-gray-100 text-gray-800 border-gray-200";
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "verified":
        return <CheckCircle className="w-4 h-4" />;
      case "pending_review":
        return <Clock className="w-4 h-4" />;
      case "expiring_soon":
        return <AlertTriangle className="w-4 h-4" />;
      case "expired":
        return <XCircle className="w-4 h-4" />;
      default:
        return <FileText className="w-4 h-4" />;
    }
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case "Legal":
        return <Shield className="w-5 h-5 text-blue-600" />;
      case "Financial":
        return <DollarSign className="w-5 h-5 text-green-600" />;
      case "Inspection":
        return <Eye className="w-5 h-5 text-orange-600" />;
      case "Insurance":
        return <Award className="w-5 h-5 text-purple-600" />;
      default:
        return <FileText className="w-5 h-5 text-gray-600" />;
    }
  };

  const filteredDocuments = mockDocuments.filter(doc => {
    const typeMatch = filterType === "all" || doc.type === filterType;
    const statusMatch = filterStatus === "all" || doc.status === filterStatus;
    const searchMatch = doc.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                       doc.category.toLowerCase().includes(searchTerm.toLowerCase()) ||
                       doc.propertyName.toLowerCase().includes(searchTerm.toLowerCase());
    return typeMatch && statusMatch && searchMatch;
  });

  const handleDownload = (docId: string) => {
    toast({
      title: "Download Started",
      description: "Document download has been initiated.",
    });
  };

  const handleShare = (docId: string) => {
    toast({
      title: "Share Link Generated",
      description: "Secure share link has been copied to clipboard.",
    });
  };

  const handleDelete = (docId: string) => {
    toast({
      title: "Document Deleted",
      description: "Document has been moved to trash.",
      variant: "destructive",
    });
  };

  const openDocDetails = (doc: any) => {
    setSelectedDoc(doc);
    setIsDetailsOpen(true);
  };

  const documentStats = {
    total: mockDocuments.length,
    verified: mockDocuments.filter(d => d.status === "verified").length,
    pending: mockDocuments.filter(d => d.status === "pending_review").length,
    expiring: mockDocuments.filter(d => d.status === "expiring_soon").length,
    avgConfidence: Math.round(mockDocuments.reduce((sum, d) => sum + d.aiConfidence, 0) / mockDocuments.length)
  };

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Smart Document Management</h1>
          <p className="text-muted-foreground">AI-powered document organization and compliance tracking</p>
        </div>
        <Button onClick={() => setIsUploadOpen(true)}>
          <Upload className="mr-2 h-4 w-4" />
          Upload Document
        </Button>
      </div>

      {/* Document Analytics */}
      <div className="grid grid-cols-2 md:grid-cols-5 gap-4">
        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-blue-700 font-medium">Total Documents</p>
                <p className="text-2xl font-bold text-blue-800">{documentStats.total}</p>
              </div>
              <FileText className="w-6 h-6 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-green-700 font-medium">Verified</p>
                <p className="text-2xl font-bold text-green-800">{documentStats.verified}</p>
              </div>
              <CheckCircle className="w-6 h-6 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-yellow-50 to-yellow-100 border-yellow-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-yellow-700 font-medium">Pending Review</p>
                <p className="text-2xl font-bold text-yellow-800">{documentStats.pending}</p>
              </div>
              <Clock className="w-6 h-6 text-yellow-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-orange-700 font-medium">Expiring Soon</p>
                <p className="text-2xl font-bold text-orange-800">{documentStats.expiring}</p>
              </div>
              <AlertTriangle className="w-6 h-6 text-orange-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-purple-700 font-medium">AI Confidence</p>
                <p className="text-2xl font-bold text-purple-800">{documentStats.avgConfidence}%</p>
              </div>
              <Zap className="w-6 h-6 text-purple-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Filters and Search */}
      <div className="flex flex-col md:flex-row gap-4">
        <div className="flex-1">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
            <Input
              placeholder="Search documents by name, category, or property..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-10"
            />
          </div>
        </div>
        <Select value={filterType} onValueChange={setFilterType}>
          <SelectTrigger className="w-48">
            <SelectValue placeholder="Filter by type" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">All Types</SelectItem>
            <SelectItem value="Legal">Legal</SelectItem>
            <SelectItem value="Financial">Financial</SelectItem>
            <SelectItem value="Inspection">Inspection</SelectItem>
            <SelectItem value="Insurance">Insurance</SelectItem>
          </SelectContent>
        </Select>
        <Select value={filterStatus} onValueChange={setFilterStatus}>
          <SelectTrigger className="w-48">
            <SelectValue placeholder="Filter by status" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">All Status</SelectItem>
            <SelectItem value="verified">Verified</SelectItem>
            <SelectItem value="pending_review">Pending Review</SelectItem>
            <SelectItem value="expiring_soon">Expiring Soon</SelectItem>
            <SelectItem value="expired">Expired</SelectItem>
          </SelectContent>
        </Select>
      </div>

      {/* Documents Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {filteredDocuments.map((doc) => (
          <Card key={doc.id} className="hover:shadow-lg transition-shadow">
            <CardContent className="p-6">
              <div className="flex items-start justify-between mb-4">
                <div className="flex items-center gap-3">
                  {getTypeIcon(doc.type)}
                  <div>
                    <h3 className="font-semibold text-sm truncate max-w-[200px]">{doc.name}</h3>
                    <p className="text-xs text-muted-foreground">{doc.category}</p>
                  </div>
                </div>
                <div className="flex items-center gap-1">
                  {doc.isEncrypted && <Lock className="w-4 h-4 text-green-600" />}
                  <Badge className={getStatusColor(doc.status)}>
                    {getStatusIcon(doc.status)}
                  </Badge>
                </div>
              </div>

              <div className="space-y-2 mb-4">
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">Property:</span>
                  <span className="font-medium truncate max-w-[120px]">{doc.propertyName}</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">Size:</span>
                  <span>{doc.size}</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">AI Confidence:</span>
                  <span className="font-medium text-green-600">{doc.aiConfidence}%</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">Views:</span>
                  <span>{doc.viewCount}</span>
                </div>
              </div>

              <div className="flex flex-wrap gap-1 mb-4">
                {doc.tags.slice(0, 3).map((tag) => (
                  <Badge key={tag} variant="outline" className="text-xs">
                    {tag}
                  </Badge>
                ))}
              </div>

              {doc.expiryDate && (
                <div className="mb-4 p-2 bg-orange-50 rounded-lg">
                  <p className="text-xs text-orange-700">
                    Expires: {doc.expiryDate.toLocaleDateString()}
                  </p>
                </div>
              )}

              <div className="flex gap-2">
                <Button size="sm" variant="outline" onClick={() => openDocDetails(doc)}>
                  <Eye className="w-3 h-3 mr-1" />
                  View
                </Button>
                <Button size="sm" variant="outline" onClick={() => handleDownload(doc.id)}>
                  <Download className="w-3 h-3 mr-1" />
                  Download
                </Button>
                <Button size="sm" variant="outline" onClick={() => handleShare(doc.id)}>
                  <Share2 className="w-3 h-3 mr-1" />
                  Share
                </Button>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Document Details Modal */}
      <Dialog open={isDetailsOpen} onOpenChange={setIsDetailsOpen}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Document Details - {selectedDoc?.name}</DialogTitle>
          </DialogHeader>
          {selectedDoc && (
            <div className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <Card>
                  <CardHeader>
                    <CardTitle className="text-lg">Document Information</CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-3">
                    <div>
                      <p className="text-sm text-muted-foreground">Type</p>
                      <div className="flex items-center gap-2">
                        {getTypeIcon(selectedDoc.type)}
                        <span className="font-medium">{selectedDoc.type}</span>
                      </div>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Category</p>
                      <p className="font-medium">{selectedDoc.category}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Size</p>
                      <p className="font-medium">{selectedDoc.size}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">AI Confidence</p>
                      <div className="flex items-center gap-2">
                        <span className="text-lg font-bold text-green-600">{selectedDoc.aiConfidence}%</span>
                        <Star className="w-4 h-4 text-yellow-500" />
                      </div>
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardHeader>
                    <CardTitle className="text-lg">Access & Security</CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-3">
                    <div>
                      <p className="text-sm text-muted-foreground">Access Level</p>
                      <Badge className={selectedDoc.accessLevel === "restricted" ? "bg-red-100 text-red-800" : "bg-green-100 text-green-800"}>
                        {selectedDoc.accessLevel}
                      </Badge>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Encryption</p>
                      <div className="flex items-center gap-2">
                        {selectedDoc.isEncrypted ? <Lock className="w-4 h-4 text-green-600" /> : <Unlock className="w-4 h-4 text-gray-600" />}
                        <span>{selectedDoc.isEncrypted ? "Encrypted" : "Not Encrypted"}</span>
                      </div>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Shared With</p>
                      <p className="font-medium">{selectedDoc.sharedWith.length} users</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Downloads</p>
                      <p className="font-medium">{selectedDoc.downloadCount}</p>
                    </div>
                  </CardContent>
                </Card>
              </div>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">AI Summary</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-sm bg-blue-50 p-4 rounded-lg">{selectedDoc.aiSummary}</p>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Required For</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="flex flex-wrap gap-2">
                    {selectedDoc.requiredFor.map((requirement: string) => (
                      <Badge key={requirement} variant="outline">
                        {requirement}
                      </Badge>
                    ))}
                  </div>
                </CardContent>
              </Card>

              <div className="flex gap-2 pt-4">
                <Button onClick={() => handleDownload(selectedDoc.id)}>
                  <Download className="w-4 h-4 mr-1" />
                  Download
                </Button>
                <Button variant="outline" onClick={() => handleShare(selectedDoc.id)}>
                  <Share2 className="w-4 h-4 mr-1" />
                  Share
                </Button>
                <Button variant="outline">
                  <Edit className="w-4 h-4 mr-1" />
                  Edit
                </Button>
                <Button variant="destructive" onClick={() => handleDelete(selectedDoc.id)}>
                  <Trash2 className="w-4 h-4 mr-1" />
                  Delete
                </Button>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}