import React, { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import { Input } from "../components/ui/input";
import { Textarea } from "../components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "../components/ui/dialog";
import { useToast } from "../hooks/use-toast";
import { 
  Users, 
  Phone, 
  MessageCircle, 
  Mail,
  Star,
  Clock,
  TrendingUp,
  Filter,
  Search,
  Calendar,
  MapPin,
  DollarSign,
  Eye,
  ThumbsUp,
  AlertCircle,
  CheckCircle,
  XCircle,
  Zap,
  Target,
  Award,
  Activity
} from "lucide-react";

// Advanced lead data with AI scoring
const mockLeads = [
  {
    id: "lead-1",
    name: "Sarah Johnson",
    email: "sarah.j@email.com",
    phone: "+1 (555) 123-4567",
    propertyInterest: "Modern Downtown Apartment",
    propertyId: "prop-1",
    leadScore: 92,
    status: "hot",
    source: "Website",
    budget: 450000,
    timeline: "Immediate",
    lastContact: new Date("2024-12-15T10:30:00"),
    interactions: 8,
    viewedProperties: 3,
    inquiryDate: new Date("2024-12-14T14:20:00"),
    notes: "Very interested, pre-approved for mortgage",
    tags: ["Pre-approved", "Urgent", "High-value"],
    engagement: {
      emailOpens: 5,
      linkClicks: 12,
      propertyViews: 15,
      timeOnSite: "8:45"
    },
    preferences: {
      location: "Downtown",
      propertyType: "Apartment",
      bedrooms: "2-3",
      maxBudget: 500000
    }
  },
  {
    id: "lead-2",
    name: "Michael Chen",
    email: "m.chen@email.com", 
    phone: "+1 (555) 987-6543",
    propertyInterest: "Luxury Beachfront Villa",
    propertyId: "prop-2",
    leadScore: 78,
    status: "warm",
    source: "Social Media",
    budget: 800000,
    timeline: "3-6 months",
    lastContact: new Date("2024-12-13T16:45:00"),
    interactions: 5,
    viewedProperties: 2,
    inquiryDate: new Date("2024-12-12T09:15:00"),
    notes: "Interested in luxury properties, needs to sell current home first",
    tags: ["Luxury", "Investor"],
    engagement: {
      emailOpens: 3,
      linkClicks: 7,
      propertyViews: 8,
      timeOnSite: "5:20"
    },
    preferences: {
      location: "Beachfront",
      propertyType: "Villa",
      bedrooms: "4+",
      maxBudget: 1000000
    }
  },
  {
    id: "lead-3",
    name: "Emily Rodriguez",
    email: "emily.r@email.com",
    phone: "+1 (555) 456-7890",
    propertyInterest: "Family Suburban Home",
    propertyId: "prop-3",
    leadScore: 45,
    status: "cold",
    source: "Referral",
    budget: 320000,
    timeline: "6+ months",
    lastContact: new Date("2024-12-10T11:20:00"),
    interactions: 2,
    viewedProperties: 1,
    inquiryDate: new Date("2024-12-09T13:30:00"),
    notes: "First-time buyer, still exploring options",
    tags: ["First-time", "Budget-conscious"],
    engagement: {
      emailOpens: 1,
      linkClicks: 3,
      propertyViews: 4,
      timeOnSite: "2:15"
    },
    preferences: {
      location: "Suburban",
      propertyType: "House",
      bedrooms: "3-4",
      maxBudget: 350000
    }
  }
];

export default function SmartLeads() {
  const [selectedLead, setSelectedLead] = useState<any>(null);
  const [isDetailsOpen, setIsDetailsOpen] = useState(false);
  const [filterStatus, setFilterStatus] = useState("all");
  const [searchTerm, setSearchTerm] = useState("");
  const [followUpNote, setFollowUpNote] = useState("");
  const { toast } = useToast();

  const getStatusColor = (status: string) => {
    switch (status) {
      case "hot":
        return "bg-red-100 text-red-800 border-red-200";
      case "warm":
        return "bg-orange-100 text-orange-800 border-orange-200";
      case "cold":
        return "bg-blue-100 text-blue-800 border-blue-200";
      default:
        return "bg-gray-100 text-gray-800 border-gray-200";
    }
  };

  const getScoreColor = (score: number) => {
    if (score >= 80) return "text-red-600";
    if (score >= 60) return "text-orange-600";
    return "text-blue-600";
  };

  const filteredLeads = mockLeads.filter(lead => {
    const statusMatch = filterStatus === "all" || lead.status === filterStatus;
    const searchMatch = lead.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                       lead.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
                       lead.propertyInterest.toLowerCase().includes(searchTerm.toLowerCase());
    return statusMatch && searchMatch;
  });

  const handleContactLead = (leadId: string, method: string) => {
    toast({
      title: "Contact Initiated",
      description: `${method} contact initiated with lead.`,
    });
  };

  const handleUpdateStatus = (leadId: string, newStatus: string) => {
    toast({
      title: "Status Updated",
      description: `Lead status updated to ${newStatus}.`,
    });
  };

  const handleScheduleFollowUp = () => {
    toast({
      title: "Follow-up Scheduled",
      description: "Follow-up reminder has been set.",
    });
    setFollowUpNote("");
  };

  const openLeadDetails = (lead: any) => {
    setSelectedLead(lead);
    setIsDetailsOpen(true);
  };

  return (
    <div className="flex-1 space-y-6 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold tracking-tight">Smart Lead Management</h1>
          <p className="text-muted-foreground">AI-powered lead scoring and intelligent follow-up system</p>
        </div>
      </div>

      {/* Lead Analytics Overview */}
      <div className="grid grid-cols-2 md:grid-cols-5 gap-4">
        <Card className="bg-gradient-to-br from-red-50 to-red-100 border-red-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-red-700 font-medium">Hot Leads</p>
                <p className="text-2xl font-bold text-red-800">
                  {mockLeads.filter(l => l.status === "hot").length}
                </p>
              </div>
              <Zap className="w-6 h-6 text-red-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-orange-700 font-medium">Warm Leads</p>
                <p className="text-2xl font-bold text-orange-800">
                  {mockLeads.filter(l => l.status === "warm").length}
                </p>
              </div>
              <Target className="w-6 h-6 text-orange-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-blue-700 font-medium">Cold Leads</p>
                <p className="text-2xl font-bold text-blue-800">
                  {mockLeads.filter(l => l.status === "cold").length}
                </p>
              </div>
              <Users className="w-6 h-6 text-blue-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-green-700 font-medium">Avg Score</p>
                <p className="text-2xl font-bold text-green-800">
                  {Math.round(mockLeads.reduce((sum, l) => sum + l.leadScore, 0) / mockLeads.length)}
                </p>
              </div>
              <Award className="w-6 h-6 text-green-600" />
            </div>
          </CardContent>
        </Card>

        <Card className="bg-gradient-to-br from-purple-50 to-purple-100 border-purple-200">
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-purple-700 font-medium">Response Rate</p>
                <p className="text-2xl font-bold text-purple-800">87%</p>
              </div>
              <Activity className="w-6 h-6 text-purple-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Filters and Search */}
      <div className="flex flex-col md:flex-row gap-4">
        <div className="flex-1">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
            <Input
              placeholder="Search leads by name, email, or property..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-10"
            />
          </div>
        </div>
        <Select value={filterStatus} onValueChange={setFilterStatus}>
          <SelectTrigger className="w-48">
            <SelectValue placeholder="Filter by status" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">All Leads</SelectItem>
            <SelectItem value="hot">Hot Leads</SelectItem>
            <SelectItem value="warm">Warm Leads</SelectItem>
            <SelectItem value="cold">Cold Leads</SelectItem>
          </SelectContent>
        </Select>
      </div>

      {/* Leads List */}
      <div className="space-y-4">
        {filteredLeads.map((lead) => (
          <Card key={lead.id} className="hover:shadow-md transition-shadow">
            <CardContent className="p-6">
              <div className="flex justify-between items-start mb-4">
                <div className="flex-1">
                  <div className="flex items-center gap-3 mb-2">
                    <h3 className="text-lg font-semibold">{lead.name}</h3>
                    <Badge className={getStatusColor(lead.status)}>
                      {lead.status.toUpperCase()}
                    </Badge>
                    <div className="flex items-center gap-1">
                      <Star className="w-4 h-4 text-yellow-500" />
                      <span className={`font-semibold ${getScoreColor(lead.leadScore)}`}>
                        {lead.leadScore}
                      </span>
                    </div>
                  </div>
                  <div className="flex flex-wrap gap-2 mb-2">
                    {lead.tags.map((tag) => (
                      <Badge key={tag} variant="outline" className="text-xs">
                        {tag}
                      </Badge>
                    ))}
                  </div>
                  <p className="text-sm text-muted-foreground">{lead.propertyInterest}</p>
                </div>
                <div className="text-right">
                  <p className="text-lg font-bold">${lead.budget.toLocaleString()}</p>
                  <p className="text-sm text-muted-foreground">Budget</p>
                </div>
              </div>

              <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-4">
                <div>
                  <p className="text-xs text-muted-foreground">Contact Info</p>
                  <p className="text-sm font-medium">{lead.email}</p>
                  <p className="text-sm">{lead.phone}</p>
                </div>
                <div>
                  <p className="text-xs text-muted-foreground">Timeline</p>
                  <p className="text-sm font-medium">{lead.timeline}</p>
                  <p className="text-xs text-muted-foreground">Source: {lead.source}</p>
                </div>
                <div>
                  <p className="text-xs text-muted-foreground">Engagement</p>
                  <p className="text-sm">{lead.interactions} interactions</p>
                  <p className="text-sm">{lead.viewedProperties} properties viewed</p>
                </div>
                <div>
                  <p className="text-xs text-muted-foreground">Last Contact</p>
                  <p className="text-sm">{lead.lastContact.toLocaleDateString()}</p>
                  <p className="text-xs text-muted-foreground">
                    {Math.floor((new Date().getTime() - lead.lastContact.getTime()) / (1000 * 60 * 60 * 24))} days ago
                  </p>
                </div>
              </div>

              <div className="flex gap-2 flex-wrap">
                <Button size="sm" onClick={() => handleContactLead(lead.id, "Phone")}>
                  <Phone className="w-4 h-4 mr-1" />
                  Call
                </Button>
                <Button size="sm" variant="outline" onClick={() => handleContactLead(lead.id, "Email")}>
                  <Mail className="w-4 h-4 mr-1" />
                  Email
                </Button>
                <Button size="sm" variant="outline" onClick={() => handleContactLead(lead.id, "SMS")}>
                  <MessageCircle className="w-4 h-4 mr-1" />
                  SMS
                </Button>
                <Button size="sm" variant="outline" onClick={() => openLeadDetails(lead)}>
                  <Eye className="w-4 h-4 mr-1" />
                  Details
                </Button>
                <Select onValueChange={(value) => handleUpdateStatus(lead.id, value)}>
                  <SelectTrigger className="w-32 h-8">
                    <SelectValue placeholder="Update Status" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="hot">Hot</SelectItem>
                    <SelectItem value="warm">Warm</SelectItem>
                    <SelectItem value="cold">Cold</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Lead Details Modal */}
      <Dialog open={isDetailsOpen} onOpenChange={setIsDetailsOpen}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Lead Details - {selectedLead?.name}</DialogTitle>
          </DialogHeader>
          {selectedLead && (
            <div className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <Card>
                  <CardHeader>
                    <CardTitle className="text-lg">Contact Information</CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-3">
                    <div>
                      <p className="text-sm text-muted-foreground">Email</p>
                      <p className="font-medium">{selectedLead.email}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Phone</p>
                      <p className="font-medium">{selectedLead.phone}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Lead Score</p>
                      <div className="flex items-center gap-2">
                        <span className={`text-2xl font-bold ${getScoreColor(selectedLead.leadScore)}`}>
                          {selectedLead.leadScore}
                        </span>
                        <Star className="w-5 h-5 text-yellow-500" />
                      </div>
                    </div>
                  </CardContent>
                </Card>

                <Card>
                  <CardHeader>
                    <CardTitle className="text-lg">Preferences</CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-3">
                    <div>
                      <p className="text-sm text-muted-foreground">Location</p>
                      <p className="font-medium">{selectedLead.preferences.location}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Property Type</p>
                      <p className="font-medium">{selectedLead.preferences.propertyType}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Bedrooms</p>
                      <p className="font-medium">{selectedLead.preferences.bedrooms}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Max Budget</p>
                      <p className="font-medium">${selectedLead.preferences.maxBudget.toLocaleString()}</p>
                    </div>
                  </CardContent>
                </Card>
              </div>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Engagement Analytics</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                    <div className="text-center">
                      <p className="text-2xl font-bold text-blue-600">{selectedLead.engagement.emailOpens}</p>
                      <p className="text-sm text-muted-foreground">Email Opens</p>
                    </div>
                    <div className="text-center">
                      <p className="text-2xl font-bold text-green-600">{selectedLead.engagement.linkClicks}</p>
                      <p className="text-sm text-muted-foreground">Link Clicks</p>
                    </div>
                    <div className="text-center">
                      <p className="text-2xl font-bold text-purple-600">{selectedLead.engagement.propertyViews}</p>
                      <p className="text-sm text-muted-foreground">Property Views</p>
                    </div>
                    <div className="text-center">
                      <p className="text-2xl font-bold text-orange-600">{selectedLead.engagement.timeOnSite}</p>
                      <p className="text-sm text-muted-foreground">Time on Site</p>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-lg">Notes & Follow-up</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <p className="text-sm text-muted-foreground mb-2">Current Notes</p>
                    <p className="p-3 bg-muted/30 rounded-lg">{selectedLead.notes}</p>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground mb-2">Add Follow-up Note</p>
                    <Textarea
                      placeholder="Add notes for follow-up..."
                      value={followUpNote}
                      onChange={(e) => setFollowUpNote(e.target.value)}
                      rows={3}
                    />
                    <Button className="mt-2" onClick={handleScheduleFollowUp}>
                      <Calendar className="w-4 h-4 mr-1" />
                      Schedule Follow-up
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}