import { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../../../ui/card';
import { Button } from '../../../ui/button';
import { Badge } from '../../../ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../../../ui/tabs';
import { Calendar, Clock, Users, Gavel, IndianRupee, TrendingUp, Trash2 } from 'lucide-react';
import { Link } from 'wouter';

interface AuctionSession {
  id: string;
  propertyId: number;
  propertyTitle: string;
  propertyImage?: string;
  propertyLocation: string;
  status: 'active' | 'upcoming' | 'ended' | 'no-auction' | 'scheduled' | 'completed' | 'cancelled';
  startTime: string;
  endTime: string;
  reservePrice: number;
  startingPrice: number;
  bidIncrement: number;
  highestBid: number;
  bidderCount: number;
}

const AuctionBidding: React.FC = () => {
  const [sessions, setSessions] = useState<AuctionSession[]>([]);
  const [loading, setLoading] = useState(true);
  const [showCreateAuction, setShowCreateAuction] = useState(false);
  const [properties, setProperties] = useState<any[]>([]);
  const [auctionForm, setAuctionForm] = useState({
    propertyId: '',
    startingBidPrice: '',
    reservePrice: '',
    minimumIncrement: '50000',
    registrationFee: '25000',
    startTime: '',
    endTime: ''
  });

  useEffect(() => {
    fetchAuctionSessions();
    fetchSellerProperties();
  }, []);

  const fetchSellerProperties = async () => {
    try {
      const sellerId = localStorage.getItem('sellerId');
      const url = sellerId 
        ? `http://localhost:8090/api/properties/?listed_by_id=${sellerId}`
        : `http://localhost:8090/api/properties/`;
      
      const response = await fetch(url);
      if (response.ok) {
        const data = await response.json();
        const allProperties = Array.isArray(data) ? data : (data.properties || []);
        setProperties(allProperties);
      }
    } catch (error) {
      console.error('Error fetching properties:', error);
    }
  };

  const createAuction = async (e: React.FormEvent) => {
    e.preventDefault();
    
    // Validation
    const startingBid = parseFloat(auctionForm.startingBidPrice);
    const reservePrice = parseFloat(auctionForm.reservePrice);
    const startTime = new Date(auctionForm.startTime);
    const endTime = new Date(auctionForm.endTime);
    const now = new Date();
    
    if (startingBid <= 0 || startingBid > 100000000) {
      alert('Starting bid price must be between ₹1 and ₹10 Crores');
      return;
    }
    
    if (reservePrice && (reservePrice <= 0 || reservePrice > 100000000)) {
      alert('Reserve price must be between ₹1 and ₹10 Crores');
      return;
    }
    
    if (reservePrice && reservePrice > 0 && reservePrice < startingBid) {
      alert('Reserve price cannot be less than starting bid price');
      return;
    }
    
    if (startTime <= now) {
      alert('Start time must be in the future');
      return;
    }
    
    if (endTime <= startTime) {
      alert('End time must be after start time');
      return;
    }
    
    const duration = (endTime.getTime() - startTime.getTime()) / (1000 * 60);
    if (duration < 10) {
      alert('Auction duration must be at least 10 minutes');
      return;
    }
    
    if (duration > 10080) { // 7 days = 7 * 24 * 60 = 10080 minutes
      alert('Auction duration cannot exceed 7 days');
      return;
    }
    
    try {
      const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
      console.log('Token found:', token ? 'Yes' : 'No');
      
      if (!token) {
        alert('Please login first - no token found');
        return;
      }
      
      const response = await fetch('http://localhost:8090/api/auction-management/create-session', {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({
          property_id: parseInt(auctionForm.propertyId),
          starting_bid_price: startingBid,
          reserve_price: reservePrice || null,
          minimum_increment: parseFloat(auctionForm.minimumIncrement),
          registration_fee: parseFloat(auctionForm.registrationFee),
          start_time: auctionForm.startTime,
          end_time: auctionForm.endTime
        })
      });
      
      if (response.ok) {
        setShowCreateAuction(false);
        setAuctionForm({
          propertyId: '',
          startingBidPrice: '',
          reservePrice: '',
          minimumIncrement: '50000',
          registrationFee: '25000',
          startTime: '',
          endTime: ''
        });
        // Refresh data immediately and again after delay
        fetchAuctionSessions();
        setTimeout(() => {
          fetchAuctionSessions();
        }, 2000);
      } else {
        const errorText = await response.text();
        console.error('Server response:', response.status, errorText);
        throw new Error(`Failed to create auction: ${response.status} - ${errorText}`);
      }
    } catch (error) {
      console.error('Error creating auction:', error);
      alert(`Failed to create auction: ${error.message}`);
    }
  };

  const fetchAuctionSessions = async () => {
    try {
      const token = localStorage.getItem('sellerToken');
      console.log('Fetching auctions with token:', token ? 'Found' : 'Not found');
      
      // Get all auctions from auction-management endpoint
      const auctionResponse = await fetch('http://localhost:8090/api/auction-management/all-sessions', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      const auctionData = auctionResponse.ok ? await auctionResponse.json() : { sessions: [] };
      
      console.log('API Response:', JSON.stringify(auctionData, null, 2));
      const existingSessions = auctionData.sessions || [];
      
      // Get all properties to get property details
      const propertiesResponse = await fetch(`http://localhost:8090/api/properties/`);
      const propertiesData = propertiesResponse.ok ? await propertiesResponse.json() : [];
      const allProperties = Array.isArray(propertiesData) ? propertiesData : (propertiesData.properties || []);
      
      // Create sessions array from actual auctions only
      const sessions = existingSessions.map((auction: any) => {
        const property = allProperties.find((p: any) => p.id === auction.property_id);
        
        return {
          id: auction.id,
          propertyId: auction.property_id,
          propertyTitle: property?.title || 'Unknown Property',
          propertyLocation: property ? `${property.locality || property.address}, ${property.city}` : 'Unknown Location',
          status: auction.session_status,
          startTime: auction.start_time,
          endTime: auction.end_time,
          reservePrice: auction.reserve_price || auction.starting_bid_price,
          startingPrice: auction.starting_bid_price,
          bidIncrement: auction.minimum_increment,
          highestBid: auction.current_highest_bid || 0,
          bidderCount: auction.registered_bidders || 0
        };
      });
      
      setSessions(sessions);
    } catch (error) {
      console.error('Error fetching auction sessions:', error);
      setSessions([]);
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) return `₹${(amount / 10000000).toFixed(1)} Cr`;
    if (amount >= 100000) return `₹${(amount / 100000).toFixed(1)} L`;
    return `₹${amount.toLocaleString()}`;
  };

  const formatDateTime = (dateString: string) => {
    if (!dateString) return 'Not Set';
    try {
      return new Date(dateString).toLocaleString('en-IN', {
        day: 'numeric',
        month: 'short',
        hour: '2-digit',
        minute: '2-digit'
      });
    } catch {
      return 'Not Set';
    }
  };

  const getStatusBadge = (status: string) => {
    const styles = {
      active: 'bg-green-100 text-green-800 border-green-200',
      scheduled: 'bg-blue-100 text-blue-800 border-blue-200',
      upcoming: 'bg-blue-100 text-blue-800 border-blue-200',
      completed: 'bg-gray-100 text-gray-800 border-gray-200',
      ended: 'bg-gray-100 text-gray-800 border-gray-200',
      cancelled: 'bg-red-100 text-red-800 border-red-200',
      'no-auction': 'bg-yellow-100 text-yellow-800 border-yellow-200'
    };
    return styles[status as keyof typeof styles] || styles.ended;
  };

  const filterAuctions = (status: string) => {
    if (status === 'all') return sessions;
    return sessions.filter(session => session.status === status);
  };

  const deleteAuction = async (sessionId: string) => {
    if (!confirm('Are you sure you want to permanently delete this auction? This action cannot be undone.')) return;
    
    console.log('Deleting auction:', sessionId);
    
    try {
      const token = localStorage.getItem('sellerToken');
      const response = await fetch(`http://localhost:8090/api/auction-management/${sessionId}`, {
        method: 'DELETE',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      
      console.log('Delete response status:', response.status);
      
      if (response.ok) {
        console.log('Auction deleted successfully');
        fetchAuctionSessions(); // Refresh the list
      } else {
        const errorText = await response.text();
        console.error('Delete failed:', response.status, errorText);
        alert('Failed to delete auction');
      }
    } catch (error) {
      console.error('Error deleting auction:', error);
      alert('Failed to delete auction');
    }
  };

  const AuctionCard = ({ session }: { session: AuctionSession }) => (
    <Card className="overflow-hidden hover:shadow-lg transition-shadow">
      <CardHeader className="pb-3">
        <div className="flex justify-between items-start">
          <div>
            <CardTitle className="text-lg">{session.propertyTitle}</CardTitle>
            <p className="text-sm text-gray-600 mt-1">{session.propertyLocation}</p>
          </div>
          <Badge className={`${getStatusBadge(session.status)} border text-xs`}>
            {session.status.charAt(0).toUpperCase() + session.status.slice(1)}
          </Badge>
        </div>
      </CardHeader>
      
      <CardContent className="space-y-4">
        {/* Auction Stats */}
        <div className="grid grid-cols-2 gap-4 text-sm">
          <div>
            <p className="text-gray-600">Current Bid</p>
            <p className="font-semibold text-green-600">{formatCurrency(session.highestBid)}</p>
          </div>
          <div>
            <p className="text-gray-600">Reserve Price</p>
            <p className="font-semibold text-red-600">{formatCurrency(session.reservePrice)}</p>
          </div>
          <div className="flex items-center">
            <Users className="w-4 h-4 mr-1 text-gray-500" />
            <span>{session.bidderCount} Bidders</span>
          </div>
          <div className="flex items-center">
            <TrendingUp className="w-4 h-4 mr-1 text-green-500" />
            <span className={session.highestBid >= session.reservePrice ? 'text-green-600' : 'text-red-600'}>
              {session.highestBid >= session.reservePrice ? 'Above Reserve' : 'Below Reserve'}
            </span>
          </div>
        </div>
        
        {/* Timing */}
        <div className="bg-gray-50 p-3 rounded-lg">
          <div className="flex items-center justify-between text-sm">
            <div className="flex items-center">
              <Calendar className="w-4 h-4 mr-1 text-gray-500" />
              <span>Start: {formatDateTime(session.startTime)}</span>
            </div>
            <div className="flex items-center">
              <Clock className="w-4 h-4 mr-1 text-gray-500" />
              <span>End: {formatDateTime(session.endTime)}</span>
            </div>
          </div>
        </div>
        
        {/* Actions */}
        <div className="flex gap-2">
          <Link href={`/auction-session/${session.propertyId}/${session.id}`} className="flex-1">
            <Button className="w-full" size="sm">
              <Gavel className="w-4 h-4 mr-1" />
              View Bids
            </Button>
          </Link>
          <Button 
            onClick={() => deleteAuction(session.id)}
            variant="destructive" 
            size="sm"
            className="px-3"
          >
            <Trash2 className="w-4 h-4" />
          </Button>
          {session.status === 'upcoming' && (
            <Button variant="outline" size="sm">
              Edit
            </Button>
          )}
        </div>
      </CardContent>
    </Card>
  );

  if (loading) {
    return (
      <div className="flex-1 p-6">
        <div className="flex items-center justify-center h-64">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 overflow-y-auto p-6">
      <div className="bg-white border-b border-gray-200 px-6 py-4 mb-6">
        <div className="flex justify-between items-center">
          <div>
            <h1 className="text-2xl font-semibold text-gray-900">Auction & Bidding</h1>
            <p className="text-sm text-gray-600">Manage your property auctions and monitor bidding activity</p>
          </div>
          <Button 
            onClick={() => setShowCreateAuction(true)} 
            style={{ backgroundColor: '#2563eb', color: 'white' }}
            className="hover:bg-blue-700 text-white px-4 py-2 font-medium"
          >
            Create Auction
          </Button>
        </div>
      </div>
      
      {sessions.length > 0 ? (
        <Tabs defaultValue="all" className="w-full">
          <TabsList className="grid w-full grid-cols-6 mb-6">
            <TabsTrigger value="all">All ({sessions.length})</TabsTrigger>
            <TabsTrigger value="active" className="flex items-center gap-1">
              <div className="w-2 h-2 bg-green-500 rounded-full"></div>
              Active ({filterAuctions('active').length})
            </TabsTrigger>
            <TabsTrigger value="scheduled" className="flex items-center gap-1">
              <div className="w-2 h-2 bg-blue-500 rounded-full"></div>
              Scheduled ({filterAuctions('scheduled').length})
            </TabsTrigger>
            <TabsTrigger value="completed" className="flex items-center gap-1">
              <div className="w-2 h-2 bg-gray-500 rounded-full"></div>
              Completed ({filterAuctions('completed').length})
            </TabsTrigger>
            <TabsTrigger value="cancelled" className="flex items-center gap-1">
              <div className="w-2 h-2 bg-red-500 rounded-full"></div>
              Cancelled ({filterAuctions('cancelled').length})
            </TabsTrigger>
          </TabsList>

          <TabsContent value="all">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {sessions.map((session) => (
                <AuctionCard key={session.id} session={session} />
              ))}
            </div>
          </TabsContent>

          <TabsContent value="active">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {filterAuctions('active').map((session) => (
                <AuctionCard key={session.id} session={session} />
              ))}
            </div>
          </TabsContent>

          <TabsContent value="scheduled">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {filterAuctions('scheduled').map((session) => (
                <AuctionCard key={session.id} session={session} />
              ))}
            </div>
          </TabsContent>

          <TabsContent value="completed">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {filterAuctions('completed').map((session) => (
                <AuctionCard key={session.id} session={session} />
              ))}
            </div>
          </TabsContent>

          <TabsContent value="cancelled">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {filterAuctions('cancelled').map((session) => (
                <AuctionCard key={session.id} session={session} />
              ))}
            </div>
          </TabsContent>
        </Tabs>
      ) : (
        <div className="text-center py-12">
          <Gavel className="w-16 h-16 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">
            No Active Auctions
          </h3>
          <p className="text-gray-600 mb-4">
            You don't have any auction sessions yet. Enable bidding on your properties to start.
          </p>
          <Link href="/bidding-properties">
            <Button>
              <IndianRupee className="w-4 h-4 mr-2" />
              View Properties
            </Button>
          </Link>
        </div>
      )}
      
      {/* Create Auction Modal */}
      {showCreateAuction && (
        <div className="fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center p-4">
          <div className="bg-white rounded-lg p-6 w-full max-w-4xl max-h-[90vh] overflow-y-auto">
            <h2 className="text-xl font-bold mb-4">Create Auction</h2>
            <form onSubmit={createAuction} className="space-y-6">
              <div>
                <label className="block text-sm font-medium mb-1">Property</label>
                <select 
                  value={auctionForm.propertyId} 
                  onChange={(e) => setAuctionForm({...auctionForm, propertyId: e.target.value})}
                  className="w-full p-3 border rounded"
                  required
                >
                  <option value="">Select Property</option>
                  {properties
                    .filter(prop => 
                      prop.listing_type === 'bidding' || 
                      prop.bidding_enabled === true ||
                      prop.auction_enabled === true
                    )
                    .map(prop => (
                      <option key={prop.id} value={prop.id}>{prop.title}</option>
                    ))}
                </select>
              </div>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium mb-2">Starting Bid Price</label>
                    <input 
                      type="number" 
                      value={auctionForm.startingBidPrice}
                      onChange={(e) => setAuctionForm({...auctionForm, startingBidPrice: e.target.value})}
                      className="w-full p-3 border rounded"
                      style={{ MozAppearance: 'textfield' }}
                      onWheel={(e) => e.currentTarget.blur()}
                      min="1"
                      max="100000000"
                      placeholder="Enter amount (max ₹10 Cr)"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-2">Reserve Price (Optional)</label>
                    <input 
                      type="number" 
                      value={auctionForm.reservePrice}
                      onChange={(e) => setAuctionForm({...auctionForm, reservePrice: e.target.value})}
                      className="w-full p-3 border rounded"
                      style={{ MozAppearance: 'textfield' }}
                      onWheel={(e) => e.currentTarget.blur()}
                      min="1"
                      max="100000000"
                      placeholder="Enter amount (max ₹10 Cr, optional)"
                    />
                  </div>
                </div>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-sm font-medium mb-2">Minimum Increment</label>
                    <input 
                      type="number" 
                      value={auctionForm.minimumIncrement}
                      onChange={(e) => setAuctionForm({...auctionForm, minimumIncrement: e.target.value})}
                      className="w-full p-3 border rounded"
                      style={{ MozAppearance: 'textfield' }}
                      onWheel={(e) => e.currentTarget.blur()}
                      min="1000"
                      max="10000000"
                      placeholder="Enter minimum bid increment"
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-sm font-medium mb-2">Registration Fee</label>
                    <input 
                      type="number" 
                      value={auctionForm.registrationFee}
                      onChange={(e) => setAuctionForm({...auctionForm, registrationFee: e.target.value})}
                      className="w-full p-3 border rounded"
                      style={{ MozAppearance: 'textfield' }}
                      onWheel={(e) => e.currentTarget.blur()}
                      min="1000"
                      max="1000000"
                      placeholder="Enter registration fee for bidders"
                      required
                    />
                  </div>
                </div>
              <div>
                <label className="block text-sm font-medium mb-1">Start Time</label>
                <div className="grid grid-cols-4 gap-2">
                  <div>
                    <label className="block text-xs text-gray-500 mb-1">Date</label>
                    <input 
                      type="date" 
                      value={auctionForm.startTime.split('T')[0] || ''}
                      onChange={(e) => {
                        const time = auctionForm.startTime.split('T')[1] || '09:00';
                        setAuctionForm({...auctionForm, startTime: `${e.target.value}T${time}`});
                      }}
                      className="w-full p-2 border rounded text-sm"
                      min={new Date().toISOString().split('T')[0]}
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-xs text-gray-500 mb-1">Hour</label>
                    <select 
                      value={(() => {
                        const hour = parseInt(auctionForm.startTime.split('T')[1]?.split(':')[0] || '09');
                        return hour === 0 ? '12' : hour > 12 ? (hour - 12).toString() : hour.toString();
                      })()} 
                      onChange={(e) => {
                        const date = auctionForm.startTime.split('T')[0] || new Date().toISOString().split('T')[0];
                        const minutes = auctionForm.startTime.split('T')[1]?.split(':')[1] || '00';
                        const currentHour = parseInt(auctionForm.startTime.split('T')[1]?.split(':')[0] || '09');
                        const isPM = currentHour >= 12;
                        let newHour = parseInt(e.target.value);
                        if (newHour === 12) newHour = 0;
                        if (isPM) newHour += 12;
                        setAuctionForm({...auctionForm, startTime: `${date}T${newHour.toString().padStart(2, '0')}:${minutes}`});
                      }}
                      className="w-full p-2 border rounded text-sm"
                      required
                    >
                      {Array.from({length: 12}, (_, i) => i + 1).map(hour => (
                        <option key={hour} value={hour}>{hour}</option>
                      ))}
                    </select>
                  </div>
                  <div>
                    <label className="block text-xs text-gray-500 mb-1">Min</label>
                    <select 
                      value={auctionForm.startTime.split('T')[1]?.split(':')[1] || '00'}
                      onChange={(e) => {
                        const date = auctionForm.startTime.split('T')[0] || new Date().toISOString().split('T')[0];
                        const hour = auctionForm.startTime.split('T')[1]?.split(':')[0] || '09';
                        setAuctionForm({...auctionForm, startTime: `${date}T${hour}:${e.target.value}`});
                      }}
                      className="w-full p-2 border rounded text-sm"
                    >
                      <option value="00">00</option>
                      <option value="15">15</option>
                      <option value="30">30</option>
                      <option value="45">45</option>
                    </select>
                  </div>
                  <div>
                    <label className="block text-xs text-gray-500 mb-1">AM/PM</label>
                    <select 
                      value={(() => {
                        const hour = parseInt(auctionForm.startTime.split('T')[1]?.split(':')[0] || '09');
                        return hour >= 12 ? 'PM' : 'AM';
                      })()} 
                      onChange={(e) => {
                        const date = auctionForm.startTime.split('T')[0] || new Date().toISOString().split('T')[0];
                        const currentHour = parseInt(auctionForm.startTime.split('T')[1]?.split(':')[0] || '09');
                        const minutes = auctionForm.startTime.split('T')[1]?.split(':')[1] || '00';
                        let newHour = currentHour;
                        if (e.target.value === 'PM' && currentHour < 12) newHour += 12;
                        if (e.target.value === 'AM' && currentHour >= 12) newHour -= 12;
                        setAuctionForm({...auctionForm, startTime: `${date}T${newHour.toString().padStart(2, '0')}:${minutes}`});
                      }}
                      className="w-full p-2 border rounded text-sm"
                    >
                      <option value="AM">AM</option>
                      <option value="PM">PM</option>
                    </select>
                  </div>
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium mb-1">End Time</label>
                <div className="grid grid-cols-4 gap-2">
                  <div>
                    <label className="block text-xs text-gray-500 mb-1">Date</label>
                    <input 
                      type="date" 
                      value={auctionForm.endTime.split('T')[0] || ''}
                      onChange={(e) => {
                        const time = auctionForm.endTime.split('T')[1] || '18:00';
                        setAuctionForm({...auctionForm, endTime: `${e.target.value}T${time}`});
                      }}
                      className="w-full p-2 border rounded text-sm"
                      min={auctionForm.startTime.split('T')[0] || new Date().toISOString().split('T')[0]}
                      required
                    />
                  </div>
                  <div>
                    <label className="block text-xs text-gray-500 mb-1">Hour</label>
                    <select 
                      value={(() => {
                        const hour = parseInt(auctionForm.endTime.split('T')[1]?.split(':')[0] || '18');
                        return hour === 0 ? '12' : hour > 12 ? (hour - 12).toString() : hour.toString();
                      })()} 
                      onChange={(e) => {
                        const date = auctionForm.endTime.split('T')[0] || new Date().toISOString().split('T')[0];
                        const minutes = auctionForm.endTime.split('T')[1]?.split(':')[1] || '00';
                        const currentHour = parseInt(auctionForm.endTime.split('T')[1]?.split(':')[0] || '18');
                        const isPM = currentHour >= 12;
                        let newHour = parseInt(e.target.value);
                        if (newHour === 12) newHour = 0;
                        if (isPM) newHour += 12;
                        setAuctionForm({...auctionForm, endTime: `${date}T${newHour.toString().padStart(2, '0')}:${minutes}`});
                      }}
                      className="w-full p-2 border rounded text-sm"
                      required
                    >
                      {Array.from({length: 12}, (_, i) => i + 1).map(hour => (
                        <option key={hour} value={hour}>{hour}</option>
                      ))}
                    </select>
                  </div>
                  <div>
                    <label className="block text-xs text-gray-500 mb-1">Min</label>
                    <select 
                      value={auctionForm.endTime.split('T')[1]?.split(':')[1] || '00'}
                      onChange={(e) => {
                        const date = auctionForm.endTime.split('T')[0] || new Date().toISOString().split('T')[0];
                        const hour = auctionForm.endTime.split('T')[1]?.split(':')[0] || '18';
                        setAuctionForm({...auctionForm, endTime: `${date}T${hour}:${e.target.value}`});
                      }}
                      className="w-full p-2 border rounded text-sm"
                    >
                      <option value="00">00</option>
                      <option value="15">15</option>
                      <option value="30">30</option>
                      <option value="45">45</option>
                    </select>
                  </div>
                  <div>
                    <label className="block text-xs text-gray-500 mb-1">AM/PM</label>
                    <select 
                      value={(() => {
                        const hour = parseInt(auctionForm.endTime.split('T')[1]?.split(':')[0] || '18');
                        return hour >= 12 ? 'PM' : 'AM';
                      })()} 
                      onChange={(e) => {
                        const date = auctionForm.endTime.split('T')[0] || new Date().toISOString().split('T')[0];
                        const currentHour = parseInt(auctionForm.endTime.split('T')[1]?.split(':')[0] || '18');
                        const minutes = auctionForm.endTime.split('T')[1]?.split(':')[1] || '00';
                        let newHour = currentHour;
                        if (e.target.value === 'PM' && currentHour < 12) newHour += 12;
                        if (e.target.value === 'AM' && currentHour >= 12) newHour -= 12;
                        setAuctionForm({...auctionForm, endTime: `${date}T${newHour.toString().padStart(2, '0')}:${minutes}`});
                      }}
                      className="w-full p-2 border rounded text-sm"
                    >
                      <option value="AM">AM</option>
                      <option value="PM">PM</option>
                    </select>
                  </div>
                </div>
              </div>
              <div className="flex gap-2 pt-4">
                <Button type="submit" className="flex-1">Create Auction</Button>
                <Button type="button" onClick={() => setShowCreateAuction(false)} className="flex-1 bg-gray-500">Cancel</Button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  );
};

export default AuctionBidding;