

import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Textarea } from "@/components/ui/textarea";
import { Clock, CheckCircle, XCircle, DollarSign, Calendar, AlertTriangle, MessageCircle, FileText } from "lucide-react";
import { mockProperties, getPropertyById, mockBuyers, getBuyerById } from "../lib/mock-data";
import { useState } from "react";

const mockHolds = [
  {
    id: "hold-1",
    auctionId: "auction-1",
    propertyId: "prop-1",
    buyerId: "buyer-1",
    winningBid: 485000,
    holdPeriod: 48, // hours
    holdStarted: new Date("2024-12-16T18:00:00"),
    holdExpires: new Date("2024-12-18T18:00:00"),
    status: "pending_payment",
    depositRequired: 24250, // 5% of winning bid
    depositPaid: false,
    contractSent: true,
    notes: "Buyer requested extended hold period"
  },
  {
    id: "hold-2",
    auctionId: "auction-2",
    propertyId: "prop-2",
    buyerId: "buyer-2",
    winningBid: 367000,
    holdPeriod: 24,
    holdStarted: new Date("2024-12-15T15:00:00"),
    holdExpires: new Date("2024-12-16T15:00:00"),
    status: "payment_confirmed",
    depositRequired: 18350,
    depositPaid: true,
    contractSent: true,
    paymentConfirmedAt: new Date("2024-12-15T20:30:00"),
    notes: ""
  },
  {
    id: "hold-3",
    auctionId: "auction-3",
    propertyId: "prop-3",
    buyerId: "buyer-3",
    winningBid: 295000,
    holdPeriod: 48,
    holdStarted: new Date("2024-12-14T16:00:00"),
    holdExpires: new Date("2024-12-16T16:00:00"),
    status: "expired",
    depositRequired: 14750,
    depositPaid: false,
    contractSent: true,
    notes: "Buyer did not respond to payment requests"
  }
];

export default function AuctionHold() {
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [selectedHold, setSelectedHold] = useState<any>(null);
  const [isDetailsDialogOpen, setIsDetailsDialogOpen] = useState(false);
  const [notes, setNotes] = useState("");

  const filteredHolds = mockHolds.filter(hold => 
    statusFilter === "all" || hold.status === statusFilter
  );

  const getStatusColor = (status: string) => {
    switch (status) {
      case "pending_payment":
        return "bg-yellow-100 text-yellow-800";
      case "payment_confirmed":
        return "bg-green-100 text-green-800";
      case "expired":
        return "bg-red-100 text-red-800";
      case "cancelled":
        return "bg-gray-100 text-gray-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "pending_payment":
        return <Clock className="w-4 h-4" />;
      case "payment_confirmed":
        return <CheckCircle className="w-4 h-4" />;
      case "expired":
        return <XCircle className="w-4 h-4" />;
      case "cancelled":
        return <XCircle className="w-4 h-4" />;
      default:
        return <Clock className="w-4 h-4" />;
    }
  };

  const formatTimeRemaining = (expireDate: Date) => {
    const now = new Date();
    const diff = expireDate.getTime() - now.getTime();
    
    if (diff <= 0) return "Expired";
    
    const hours = Math.floor(diff / (1000 * 60 * 60));
    const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
    
    if (hours > 24) {
      const days = Math.floor(hours / 24);
      return `${days}d ${hours % 24}h`;
    }
    if (hours > 0) return `${hours}h ${minutes}m`;
    return `${minutes}m`;
  };

  const handleConfirmPayment = (holdId: string) => {
    console.log("Confirming payment for hold:", holdId);
    // In a real app, this would make an API call
  };

  const handleExtendHold = (holdId: string) => {
    console.log("Extending hold:", holdId);
    // In a real app, this would make an API call
  };

  const handleCancelHold = (holdId: string) => {
    console.log("Cancelling hold:", holdId);
    // In a real app, this would make an API call
  };

  const openDetailsDialog = (hold: any) => {
    setSelectedHold(hold);
    setNotes(hold.notes || "");
    setIsDetailsDialogOpen(true);
  };

  return (
    <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Auction Hold & Confirmation</h2>
      </div>
      <div className="space-y-6">

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
          <Card className="bg-gradient-to-br from-yellow-50 to-yellow-100 border-yellow-200">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-yellow-700">
                Pending Confirmations
                <Clock className="w-5 h-5 text-yellow-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-yellow-800">
                {mockHolds.filter(h => h.status === "pending_payment").length}
              </div>
              <p className="text-xs text-yellow-600 mt-1">Awaiting payment</p>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-green-700">
                Confirmed Sales
                <CheckCircle className="w-5 h-5 text-green-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-green-800">
                {mockHolds.filter(h => h.status === "payment_confirmed").length}
              </div>
              <p className="text-xs text-green-600 mt-1">this month</p>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-red-50 to-red-100 border-red-200">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-red-700">
                Expired Holds
                <XCircle className="w-5 h-5 text-red-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-red-800">
                {mockHolds.filter(h => h.status === "expired").length}
              </div>
              <p className="text-xs text-red-600 mt-1">this week</p>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-blue-700">
                Total Hold Value
                <DollarSign className="w-5 h-5 text-blue-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-blue-800">
                ${(mockHolds.reduce((sum, hold) => sum + hold.winningBid, 0) / 1000000).toFixed(1)}M
              </div>
              <p className="text-xs text-blue-600 mt-1">combined value</p>
            </CardContent>
          </Card>
        </div>

        {/* Filter Controls */}
        <div className="flex gap-4">
          <Select value={statusFilter} onValueChange={setStatusFilter}>
            <SelectTrigger className="w-48">
              <SelectValue placeholder="Filter by status" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Status</SelectItem>
              <SelectItem value="pending_payment">Pending Payment</SelectItem>
              <SelectItem value="payment_confirmed">Payment Confirmed</SelectItem>
              <SelectItem value="expired">Expired</SelectItem>
              <SelectItem value="cancelled">Cancelled</SelectItem>
            </SelectContent>
          </Select>
        </div>

        {/* Holds List */}
        <div className="space-y-4">
          {filteredHolds.map((hold) => {
            const property = getPropertyById(hold.propertyId);
            const buyer = getBuyerById(hold.buyerId);
            const isExpiring = hold.status === "pending_payment" && 
              new Date(hold.holdExpires).getTime() - new Date().getTime() < 6 * 60 * 60 * 1000; // 6 hours
            
            return (
              <Card key={hold.id} className={`${isExpiring ? 'border-orange-300 bg-orange-50' : ''}`}>
                <CardContent className="p-6">
                  <div className="flex justify-between items-start mb-4">
                    <div className="flex-1">
                      <div className="flex items-center gap-3 mb-2">
                        <h3 className="text-lg font-semibold">{property?.title}</h3>
                        <Badge className={getStatusColor(hold.status)} variant="outline">
                          {getStatusIcon(hold.status)}
                          <span className="ml-1">{hold.status.replace('_', ' ')}</span>
                        </Badge>
                        {isExpiring && (
                          <Badge variant="outline" className="bg-orange-100 text-orange-800">
                            <AlertTriangle className="w-3 h-3 mr-1" />
                            Expiring Soon
                          </Badge>
                        )}
                      </div>
                      <p className="text-sm text-muted-foreground">{property?.address}</p>
                    </div>
                    <div className="text-right">
                      <p className="text-2xl font-bold">${hold.winningBid.toLocaleString()}</p>
                      <p className="text-sm text-muted-foreground">Winning Bid</p>
                    </div>
                  </div>

                  <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-4">
                    {/* Buyer Information */}
                    <div>
                      <h4 className="font-medium mb-2">Buyer Information</h4>
                      <div className="space-y-1">
                        <p className="text-sm">
                          <span className="font-medium">{buyer?.firstName} {buyer?.lastName}</span>
                        </p>
                        <p className="text-sm text-muted-foreground">{buyer?.email}</p>
                      </div>
                    </div>

                    {/* Hold Details */}
                    <div>
                      <h4 className="font-medium mb-2">Hold Details</h4>
                      <div className="space-y-1">
                        <p className="text-sm">
                          <span className="text-muted-foreground">Started:</span> {hold.holdStarted.toLocaleDateString()}
                        </p>
                        <p className="text-sm">
                          <span className="text-muted-foreground">Expires:</span> {hold.holdExpires.toLocaleDateString()}
                        </p>
                        <p className="text-sm">
                          <span className="text-muted-foreground">Remaining:</span> {formatTimeRemaining(hold.holdExpires)}
                        </p>
                      </div>
                    </div>

                    {/* Payment Status */}
                    <div>
                      <h4 className="font-medium mb-2">Payment Status</h4>
                      <div className="space-y-1">
                        <p className="text-sm">
                          <span className="text-muted-foreground">Deposit Required:</span> ${hold.depositRequired.toLocaleString()}
                        </p>
                        <p className="text-sm">
                          <span className="text-muted-foreground">Deposit Paid:</span> 
                          <Badge variant={hold.depositPaid ? "default" : "secondary"} className="ml-2">
                            {hold.depositPaid ? "Yes" : "No"}
                          </Badge>
                        </p>
                        <p className="text-sm">
                          <span className="text-muted-foreground">Contract Sent:</span> 
                          <Badge variant={hold.contractSent ? "default" : "secondary"} className="ml-2">
                            {hold.contractSent ? "Yes" : "No"}
                          </Badge>
                        </p>
                      </div>
                    </div>
                  </div>

                  {/* Progress Bar for Payment Status */}
                  {hold.status === "pending_payment" && (
                    <div className="mb-4">
                      <div className="flex justify-between text-sm mb-1">
                        <span>Hold Progress</span>
                        <span>{Math.round(((new Date().getTime() - hold.holdStarted.getTime()) / (hold.holdExpires.getTime() - hold.holdStarted.getTime())) * 100)}%</span>
                      </div>
                      <div className="w-full bg-gray-200 rounded-full h-2">
                        <div 
                          className={`h-2 rounded-full ${isExpiring ? 'bg-orange-500' : 'bg-blue-600'}`}
                          style={{ 
                            width: `${Math.min(((new Date().getTime() - hold.holdStarted.getTime()) / (hold.holdExpires.getTime() - hold.holdStarted.getTime())) * 100, 100)}%`
                          }}
                        ></div>
                      </div>
                    </div>
                  )}

                  {/* Notes */}
                  {hold.notes && (
                    <div className="mb-4 p-3 bg-muted/30 rounded-lg">
                      <p className="text-sm"><span className="font-medium">Notes:</span> {hold.notes}</p>
                    </div>
                  )}

                  {/* Action Buttons */}
                  <div className="flex gap-2 flex-wrap">
                    <Button 
                      variant="outline" 
                      size="sm"
                      onClick={() => openDetailsDialog(hold)}
                    >
                      <FileText className="w-4 h-4 mr-1" />
                      View Details
                    </Button>

                    {hold.status === "pending_payment" && (
                      <>
                        <Button 
                          size="sm"
                          onClick={() => handleConfirmPayment(hold.id)}
                        >
                          <CheckCircle className="w-4 h-4 mr-1" />
                          Confirm Payment
                        </Button>
                        <Button 
                          variant="outline" 
                          size="sm"
                          onClick={() => handleExtendHold(hold.id)}
                        >
                          <Calendar className="w-4 h-4 mr-1" />
                          Extend Hold
                        </Button>
                      </>
                    )}

                    {(hold.status === "pending_payment" || hold.status === "payment_confirmed") && (
                      <Button 
                        variant="destructive" 
                        size="sm"
                        onClick={() => handleCancelHold(hold.id)}
                      >
                        <XCircle className="w-4 h-4 mr-1" />
                        Cancel Hold
                      </Button>
                    )}

                    <Button variant="outline" size="sm">
                      <MessageCircle className="w-4 h-4 mr-1" />
                      Contact Buyer
                    </Button>
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {filteredHolds.length === 0 && (
          <Card>
            <CardContent className="text-center py-8">
              <Clock className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-lg font-medium text-muted-foreground mb-2">No holds found</h3>
              <p className="text-sm text-muted-foreground">
                {statusFilter === "all" 
                  ? "No auction holds to display" 
                  : `No holds with status "${statusFilter}"`
                }
              </p>
            </CardContent>
          </Card>
        )}

        {/* Hold Details Dialog */}
        <Dialog open={isDetailsDialogOpen} onOpenChange={setIsDetailsDialogOpen}>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>Hold Details</DialogTitle>
            </DialogHeader>
            {selectedHold && (
              <div className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <h4 className="font-medium mb-2">Property</h4>
                    <p className="text-sm">{getPropertyById(selectedHold.propertyId)?.title}</p>
                  </div>
                  <div>
                    <h4 className="font-medium mb-2">Buyer</h4>
                    <p className="text-sm">{getBuyerById(selectedHold.buyerId)?.firstName} {getBuyerById(selectedHold.buyerId)?.lastName}</p>
                  </div>
                </div>
                
                <div>
                  <h4 className="font-medium mb-2">Timeline</h4>
                  <div className="space-y-2 text-sm">
                    <p>Hold Started: {selectedHold.holdStarted.toLocaleString()}</p>
                    <p>Hold Expires: {selectedHold.holdExpires.toLocaleString()}</p>
                    {selectedHold.paymentConfirmedAt && (
                      <p>Payment Confirmed: {selectedHold.paymentConfirmedAt.toLocaleString()}</p>
                    )}
                  </div>
                </div>

                <div>
                  <h4 className="font-medium mb-2">Notes</h4>
                  <Textarea
                    value={notes}
                    onChange={(e) => setNotes(e.target.value)}
                    placeholder="Add notes about this hold..."
                    rows={3}
                  />
                </div>

                <div className="flex justify-end gap-2">
                  <Button variant="outline" onClick={() => setIsDetailsDialogOpen(false)}>
                    Close
                  </Button>
                  <Button onClick={() => {
                    console.log("Saving notes:", notes);
                    setIsDetailsDialogOpen(false);
                  }}>
                    Save Notes
                  </Button>
                </div>
              </div>
            )}
          </DialogContent>
        </Dialog>
      </div>
    </div>
  );
}
