import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "../components/ui/dialog";
import { Input } from "../components/ui/input";
import { Label } from "../components/ui/label";
import { Calendar, Video, MapPin, Clock, Phone, MessageCircle, Check, X, Plus, CalendarClock } from "lucide-react";
import { getSellerPendingVisits, getSellerVisits, sellerApproveVisit, sellerRejectVisit, VisitSchedule } from "../../../../api/visitSchedules";
import { getUserId } from "../../../../utils/userUtils";
import { toast } from "sonner";

function Bookings() {
  const [statusFilter, setStatusFilter] = useState<string>("all");
  const [typeFilter, setTypeFilter] = useState<string>("all");
  const [rescheduleDialog, setRescheduleDialog] = useState<{ open: boolean; bookingId?: string }>({ open: false });
  const [rescheduleData, setRescheduleData] = useState({ date: "", time: "" });
  const [visits, setVisits] = useState<VisitSchedule[]>([]);
  const [loading, setLoading] = useState(true);
  const [processingAction, setProcessingAction] = useState<{visitId: number, action: 'approve' | 'reject'} | null>(null);
  
  useEffect(() => {
    fetchVisits();
  }, [statusFilter]);

  const fetchVisits = async () => {
    try {
      setLoading(true);
      const sellerId = getUserId();
      if (!sellerId) return;
      
      const statusMap: { [key: string]: string } = {
        'all': '',
        'pending': 'admin_approved',
        'confirmed': 'seller_approved',
        'completed': 'completed',
        'cancelled': 'seller_rejected'
      };
      
      const apiStatus = statusMap[statusFilter] || '';
      const response = await getSellerVisits(sellerId, 1, 50, apiStatus);
      setVisits(response.visits);
    } catch (error) {
      console.error('Error fetching visits:', error);
      toast.error('Failed to fetch visits');
    } finally {
      setLoading(false);
    }
  };

  const handleApprove = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'approve'});
      const sellerId = getUserId();
      if (!sellerId) return;
      
      await sellerApproveVisit(visitId, sellerId);
      toast.success('Visit approved successfully');
      fetchVisits();
    } catch (error) {
      console.error('Error approving visit:', error);
      toast.error('Failed to approve visit');
    } finally {
      setProcessingAction(null);
    }
  };

  const handleReject = async (visitId: number) => {
    try {
      setProcessingAction({visitId, action: 'reject'});
      const sellerId = getUserId();
      if (!sellerId) return;
      
      await sellerRejectVisit(visitId, sellerId);
      toast.success('Visit rejected successfully');
      fetchVisits();
    } catch (error) {
      console.error('Error rejecting visit:', error);
      toast.error('Failed to reject visit');
    } finally {
      setProcessingAction(null);
    }
  };
  
  const getStatusColor = (status: string) => {
    switch (status) {
      case "admin_approved":
      case "pending":
        return "bg-orange-100 text-orange-800";
      case "seller_approved":
      case "confirmed":
        return "bg-green-100 text-green-800";
      case "completed":
        return "bg-blue-100 text-blue-800";
      case "seller_rejected":
      case "admin_rejected":
      case "cancelled":
        return "bg-red-100 text-red-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getTypeIcon = (type: string) => {
    return type === "Virtual Tour" ? <Video className="w-4 h-4" /> : <MapPin className="w-4 h-4" />;
  };

  const getTypeColor = (type: string) => {
    return type === "Virtual Tour" 
      ? "bg-blue-100 text-blue-800" 
      : "bg-green-100 text-green-800";
  };

  const formatDateTime = (dateString: string, timeString?: string) => {
    const date = new Date(dateString);
    const dateStr = date.toLocaleDateString('en-US', {
      month: 'short',
      day: 'numeric',
    });
    return `${dateStr}, ${timeString || 'Time TBD'}`;
  };

  // Filter visits
  const filteredVisits = visits.filter(visit => {
    const typeMatch = typeFilter === "all" || 
      (typeFilter === "virtual" && visit.visit_mode === "Virtual Tour") ||
      (typeFilter === "in-person" && visit.visit_mode === "In-person Visit");
    return typeMatch;
  });

  // Calculate stats
  const todayVisits = visits.filter(v => {
    const today = new Date().toDateString();
    return new Date(v.preferred_date).toDateString() === today;
  }).length;
  
  const pendingVisits = visits.filter(v => v.status === "admin_approved").length;
  const completedVisits = visits.filter(v => v.status === "completed").length;
  const virtualVisits = visits.filter(v => v.visit_mode === "Virtual Tour").length;

  const handleReschedule = (bookingId: string) => {
    setRescheduleDialog({ open: true, bookingId });
    setRescheduleData({ date: "", time: "" });
  };

  const submitReschedule = () => {
    console.log("Rescheduling booking", rescheduleDialog.bookingId, "to", rescheduleData);
    setRescheduleDialog({ open: false });
    setRescheduleData({ date: "", time: "" });
  };

  return (
    <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Appointment Bookings</h2>
      </div>
        <div className="flex items-center justify-between mb-6">
          <div>
            <h1 className="text-2xl font-bold text-foreground">Property Viewings & Appointments</h1>
            <p className="text-muted-foreground">Manage property viewing appointments and virtual tours</p>
          </div>
          <Button>
            <Plus className="mr-2 h-4 w-4" />
            Schedule Viewing
          </Button>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-6">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Today's Viewings</p>
                  <p className="text-2xl font-bold text-foreground">{todayVisits}</p>
                </div>
                <Calendar className="w-8 h-8 text-blue-600" />
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Pending Approval</p>
                  <p className="text-2xl font-bold text-orange-600">{pendingVisits}</p>
                </div>
                <Clock className="w-8 h-8 text-orange-600" />
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Completed</p>
                  <p className="text-2xl font-bold text-green-600">{completedVisits}</p>
                </div>
                <Check className="w-8 h-8 text-green-600" />
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Virtual Tours</p>
                  <p className="text-2xl font-bold text-purple-600">{virtualVisits}</p>
                </div>
                <Video className="w-8 h-8 text-purple-600" />
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Filters */}
        <div className="flex items-center gap-4 mb-6">
          <Select value={statusFilter} onValueChange={setStatusFilter}>
            <SelectTrigger className="w-40" data-testid="filter-status">
              <SelectValue placeholder="All Status" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Status</SelectItem>
              <SelectItem value="pending">Pending</SelectItem>
              <SelectItem value="confirmed">Confirmed</SelectItem>
              <SelectItem value="completed">Completed</SelectItem>
              <SelectItem value="cancelled">Cancelled</SelectItem>
            </SelectContent>
          </Select>
          
          <Select value={typeFilter} onValueChange={setTypeFilter}>
            <SelectTrigger className="w-40" data-testid="filter-type">
              <SelectValue placeholder="All Types" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Types</SelectItem>
              <SelectItem value="in-person">In-Person</SelectItem>
              <SelectItem value="virtual">Virtual</SelectItem>
            </SelectContent>
          </Select>
        </div>

        {/* Bookings Table */}
        <Card>
          <CardContent className="p-0">
            <div className="overflow-hidden">
              <table className="w-full min-w-full">
                <thead className="bg-muted/50 border-b border-border">
                  <tr>
                    <th className="px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider min-w-0">
                      Visitor
                    </th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider min-w-0">
                      Property
                    </th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider min-w-0">
                      Type
                    </th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider min-w-0">
                      Scheduled
                    </th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider min-w-0">
                      Status
                    </th>
                    <th className="px-4 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider min-w-0">
                      Actions
                    </th>
                  </tr>
                </thead>
                <tbody className="divide-y divide-border">
                  {loading ? (
                    <tr>
                      <td colSpan={6} className="px-4 py-8 text-center">
                        <div className="flex justify-center">
                          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-primary"></div>
                        </div>
                      </td>
                    </tr>
                  ) : filteredVisits.length === 0 ? (
                    <tr>
                      <td colSpan={6} className="px-4 py-8 text-center text-muted-foreground">
                        No visits found
                      </td>
                    </tr>
                  ) : (
                    filteredVisits.map((visit) => (
                      <tr key={visit.id} className="hover:bg-muted/50" data-testid={`booking-row-${visit.id}`}>
                        <td className="px-4 py-4">
                          <div className="flex items-center gap-3">
                            <div className="w-8 h-8 bg-primary rounded-full flex items-center justify-center">
                              <span className="text-primary-foreground text-xs font-medium">
                                {visit.full_name.split(' ').map(n => n[0]).join('').slice(0, 2)}
                              </span>
                            </div>
                            <div>
                              <p className="text-sm font-medium text-foreground">
                                {visit.full_name}
                              </p>
                              <p className="text-xs text-muted-foreground">{visit.email}</p>
                            </div>
                          </div>
                        </td>
                        <td className="px-4 py-4">
                          <div className="flex items-center gap-3">
                            <div className="w-10 h-10 bg-muted rounded flex items-center justify-center">
                              <MapPin className="w-5 h-5 text-muted-foreground" />
                            </div>
                            <div>
                              <p className="text-sm font-medium text-foreground">{visit.property_title}</p>
                              <p className="text-xs text-muted-foreground">{visit.property_city}</p>
                            </div>
                          </div>
                        </td>
                        <td className="px-4 py-4">
                          <Badge className={getTypeColor(visit.visit_mode)}>
                            {getTypeIcon(visit.visit_mode)}
                            <span className="ml-1">{visit.visit_mode === "Virtual Tour" ? "virtual" : "visit"}</span>
                          </Badge>
                        </td>
                        <td className="px-4 py-4">
                          <div>
                            <p className="text-sm font-medium text-foreground">
                              {formatDateTime(visit.preferred_date, visit.preferred_time)}
                            </p>
                            <p className="text-xs text-muted-foreground">
                              Duration: 1 hour
                            </p>
                          </div>
                        </td>
                        <td className="px-4 py-4">
                          <Badge className={getStatusColor(visit.status)}>
                            {visit.status === "admin_approved" ? "Pending" : 
                             visit.status === "seller_approved" ? "Confirmed" :
                             visit.status.charAt(0).toUpperCase() + visit.status.slice(1)}
                          </Badge>
                        </td>
                        <td className="px-4 py-4">
                          <div className="flex items-center gap-1">
                            {visit.status === "admin_approved" && (
                              <>
                                <Button 
                                  size="sm" 
                                  onClick={() => handleApprove(visit.id)}
                                  disabled={processingAction?.visitId === visit.id && processingAction?.action === 'approve'}
                                  data-testid={`approve-booking-${visit.id}`}
                                >
                                  <Check className="h-4 w-4" />
                                </Button>
                                <Button 
                                  size="sm" 
                                  variant="destructive" 
                                  onClick={() => handleReject(visit.id)}
                                  disabled={processingAction?.visitId === visit.id && processingAction?.action === 'reject'}
                                  data-testid={`reject-booking-${visit.id}`}
                                >
                                  <X className="h-4 w-4" />
                                </Button>
                              </>
                            )}
                            {visit.status === "seller_approved" && (
                              <>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  data-testid={`call-visitor-${visit.id}`}
                                >
                                  <Phone className="h-4 w-4" />
                                </Button>
                                <Button
                                  variant="outline"
                                  size="sm"
                                  data-testid={`message-visitor-${visit.id}`}
                                >
                                  <MessageCircle className="h-4 w-4" />
                                </Button>
                              </>
                            )}
                            {visit.visit_mode === "Virtual Tour" && visit.status === "seller_approved" && (
                              <Button
                                size="sm"
                                variant="secondary"
                                onClick={() => window.open(`https://meet.jit.si/NAL_Property_${visit.property_id}_Visit_${visit.id}`, '_blank')}
                                data-testid={`join-virtual-${visit.id}`}
                              >
                                <Video className="h-4 w-4 mr-1" />
                                Join Call
                              </Button>
                            )}
                          </div>
                        </td>
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>
          </CardContent>
        </Card>

        {/* Reschedule Dialog */}
        <Dialog open={rescheduleDialog.open} onOpenChange={(open) => setRescheduleDialog({ open })}>
          <DialogContent className="max-w-md">
            <DialogHeader>
              <DialogTitle>Reschedule Appointment</DialogTitle>
            </DialogHeader>
            <div className="space-y-4">
              <div>
                <Label htmlFor="reschedule-date">New Date</Label>
                <Input
                  id="reschedule-date"
                  type="date"
                  value={rescheduleData.date}
                  onChange={(e) => setRescheduleData(prev => ({ ...prev, date: e.target.value }))}
                  min={new Date().toISOString().split('T')[0]}
                />
              </div>
              <div>
                <Label htmlFor="reschedule-time">New Time</Label>
                <Input
                  id="reschedule-time"
                  type="time"
                  value={rescheduleData.time}
                  onChange={(e) => setRescheduleData(prev => ({ ...prev, time: e.target.value }))}
                />
              </div>
              <div className="flex justify-end gap-2 pt-4">
                <Button 
                  variant="outline" 
                  onClick={() => setRescheduleDialog({ open: false })}
                >
                  Cancel
                </Button>
                <Button 
                  onClick={submitReschedule}
                  disabled={!rescheduleData.date || !rescheduleData.time}
                >
                  Reschedule
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
    </div>
  );
}

export default Bookings;