import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../components/ui/select";
import { Plus, Megaphone, MousePointer, TrendingUp, BarChart3, Pause, Edit, Loader2 } from "lucide-react";
import { getCampaignsByUserId } from "../../../../api/campaigns";
import { getCurrentUserUuid } from "../../../../utils/userUtils";
import { useNavigate } from "react-router-dom";
import { toast } from "sonner";

interface Campaign {
  id: string;
  campaignName: string;
  campaignObjective: string;
  propertyId: string;
  startDate: string;
  endDate: string;
  budgetAmount: number;
  budgetType: string;
  propertyType: string;
  propertyLocation: string;
  propertyPrice: number;
  status: string;
  created_at: string;
}

export default function Campaigns() {
  const navigate = useNavigate();
  const [campaigns, setCampaigns] = useState<Campaign[]>([]);
  const [loading, setLoading] = useState(true);
  const [statusFilter, setStatusFilter] = useState<string>("all");

  useEffect(() => {
    fetchCampaigns();
  }, [statusFilter]);

  const fetchCampaigns = async () => {
    try {
      setLoading(true);
      
      // Get user ID from localStorage
      const currentUserId = getCurrentUserUuid();
      
      if (!currentUserId) {
        toast.error('Please login to view campaigns');
        return;
      }

      const params = statusFilter !== 'all' ? { status: statusFilter } : {};
      const response = await getCampaignsByUserId(currentUserId, params);
      setCampaigns(response.campaigns || []);
    } catch (error: any) {
      console.error('Error fetching campaigns:', error);
      toast.error('Failed to fetch campaigns');
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    if (!status) return "bg-gray-100 text-gray-800";
    
    switch (status.toLowerCase()) {
      case "active":
        return "bg-green-100 text-green-800";
      case "paused":
        return "bg-yellow-100 text-yellow-800";
      case "approved":
        return "bg-blue-100 text-blue-800";
      case "draft":
        return "bg-gray-100 text-gray-800";
      case "pending":
        return "bg-orange-100 text-orange-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const formatCurrency = (amount: number) => {
    if (typeof amount !== 'number' || isNaN(amount)) {
      return '₹0';
    }
    return `₹${amount.toLocaleString()}`;
  };

  const formatDateRange = (startDate: string, endDate: string) => {
    const s = new Date(startDate);
    const e = new Date(endDate);
    const formatOptions: Intl.DateTimeFormatOptions = { month: 'short', day: 'numeric' };
    return `${s.toLocaleDateString('en-US', formatOptions)} - ${e.toLocaleDateString('en-US', formatOptions)}`;
  };

  // Calculate overview stats
  const activeCampaigns = campaigns.filter(c => c.status?.toLowerCase() === "active").length;
  const approvedCampaigns = campaigns.filter(c => c.status?.toLowerCase() === "approved").length;
  const totalBudget = campaigns.reduce((sum, c) => sum + (c.budgetAmount || 0), 0);
  const draftCampaigns = campaigns.filter(c => c.status?.toLowerCase() === "draft").length;

  const handleCreateCampaign = () => {
    navigate('/campaign-form');
  };

  if (loading) {
    return (
      <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
        <div className="flex items-center justify-center h-64">
          <Loader2 className="h-8 w-8 animate-spin" />
          <span className="ml-2">Loading campaigns...</span>
        </div>
      </div>
    );
  }

  return (
    <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Ad Campaigns</h2>
      </div>
      {approvedCampaigns > 0 && (
        <div className="rounded-lg border border-green-200 bg-green-50 p-4 text-sm text-green-800">
          You have {approvedCampaigns} approved campaign{approvedCampaigns > 1 ? 's' : ''}. Start tracking performance now.
        </div>
      )}
        <div className="flex items-center justify-between mb-6">
          <div className="flex items-center gap-4">
            <Button onClick={handleCreateCampaign} data-testid="create-campaign-button">
              <Plus className="mr-2 h-4 w-4" />
              Create Campaign
            </Button>
            
            <Select value={statusFilter} onValueChange={setStatusFilter}>
              <SelectTrigger className="w-48" data-testid="filter-campaigns">
                <SelectValue placeholder="All Campaigns" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All Campaigns</SelectItem>
                <SelectItem value="active">Active</SelectItem>
                <SelectItem value="draft">Draft</SelectItem>
                <SelectItem value="pending">Pending</SelectItem>
                <SelectItem value="approved">Approved</SelectItem>
              </SelectContent>
            </Select>
          </div>
          
          <div className="text-sm text-muted-foreground">
            Total budget: <span className="font-medium text-foreground">{formatCurrency(totalBudget)}</span>
          </div>
        </div>

        {/* Campaign Performance Overview */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <Card data-testid="stat-active-campaigns">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Active Campaigns</p>
                  <p className="text-3xl font-bold text-foreground">{activeCampaigns}</p>
                </div>
                <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center">
                  <Megaphone className="text-primary text-xl" />
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card data-testid="stat-total-campaigns">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Total Campaigns</p>
                  <p className="text-3xl font-bold text-foreground">{campaigns.length}</p>
                </div>
                <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                  <BarChart3 className="text-blue-600 text-xl" />
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card data-testid="stat-draft-campaigns">
            <CardContent className="p-6">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Draft Campaigns</p>
                  <p className="text-3xl font-bold text-orange-600">{draftCampaigns}</p>
                </div>
                <div className="w-12 h-12 bg-orange-100 rounded-lg flex items-center justify-center">
                  <Edit className="text-orange-600 text-xl" />
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Campaigns Table */}
        <Card>
          <CardContent className="p-0">
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead className="bg-muted/50 border-b border-border">
                  <tr>
                    <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                      Campaign
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                      Property
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                      Budget
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                      Duration
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                      Performance
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                      Status
                    </th>
                    <th className="px-6 py-3 text-left text-xs font-medium text-muted-foreground uppercase tracking-wider">
                      Actions
                    </th>
                  </tr>
                </thead>
                <tbody className="divide-y divide-border">
                  {campaigns.length === 0 ? (
                    <tr>
                      <td colSpan={7} className="px-6 py-12 text-center">
                        <div className="flex flex-col items-center justify-center">
                          <Megaphone className="h-12 w-12 text-gray-400 mb-4" />
                          <h3 className="text-lg font-medium text-gray-900 mb-2">No campaigns found</h3>
                          <p className="text-gray-500 mb-4">Get started by creating your first campaign</p>
                          <Button onClick={handleCreateCampaign}>
                            <Plus className="mr-2 h-4 w-4" />
                            Create Campaign
                          </Button>
                        </div>
                      </td>
                    </tr>
                  ) : (
                    campaigns.map((campaign) => (
                      <tr key={campaign.id} className="hover:bg-muted/50" data-testid={`campaign-row-${campaign.id}`}>
                        <td className="px-6 py-4">
                          <div>
                            <p className="text-sm font-medium text-foreground">{campaign.campaignName}</p>
                            <p className="text-xs text-muted-foreground">{campaign.campaignObjective} campaign</p>
                          </div>
                        </td>
                        <td className="px-6 py-4">
                          <div>
                            <p className="text-sm text-foreground">{campaign.propertyType}</p>
                            <p className="text-xs text-muted-foreground">{campaign.propertyLocation}</p>
                          </div>
                        </td>
                        <td className="px-6 py-4">
                          <div>
                            <p className="text-sm font-medium text-foreground">{formatCurrency(campaign.budgetAmount)}</p>
                            <p className="text-xs text-muted-foreground">{campaign.budgetType} budget</p>
                          </div>
                        </td>
                        <td className="px-6 py-4 text-sm text-foreground">
                          {formatDateRange(campaign.startDate, campaign.endDate)}
                        </td>
                        <td className="px-6 py-4">
                          <div>
                            <p className="text-sm font-medium text-foreground">Property: {formatCurrency(campaign.propertyPrice)}</p>
                            <p className="text-xs text-muted-foreground">Created: {new Date(campaign.created_at).toLocaleDateString()}</p>
                          </div>
                        </td>
                        <td className="px-6 py-4">
                          <Badge className={getStatusColor(campaign.status)}>
                            {campaign.status.charAt(0).toUpperCase() + campaign.status.slice(1)}
                          </Badge>
                        </td>
                        <td className="px-6 py-4">
                          <div className="flex items-center gap-2">
                            <Button variant="ghost" size="sm" data-testid={`view-campaign-${campaign.id}`}>
                              <BarChart3 className="h-4 w-4" />
                            </Button>
                            <Button variant="ghost" size="sm" data-testid={`edit-campaign-${campaign.id}`}>
                              <Edit className="h-4 w-4" />
                            </Button>
                          </div>
                        </td>
                      </tr>
                    ))
                  )}
                </tbody>
              </table>
            </div>
          </CardContent>
        </Card>
    </div>
  );
}
