import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
} from "../components/ui/card";
import { Badge } from "../components/ui/badge";
import { Button } from "../components/ui/button";
import {
  Building,
  Users,
  TrendingUp,
  IndianRupee,
  Eye,
  MessageCircle,
  Plus,
  X,
  MapPin,
  BarChart3,
  Target,
} from "lucide-react";
import { ListingForm } from "../../../../pages/ListingForm";
import { useState, useEffect } from "react";
import { toast } from "sonner";
import { sellerDashboardAPI } from "../../../../api/seller/sellerDashboard";
import { getSellerProfileCompletionStatus } from "../../../../api/sellerProfile";
import { useNavigate } from "react-router-dom";
import { Dialog, DialogContent } from "../components/ui/dialog";
import AddPropertyChoiceModal from "../components/AddPropertyChoiceModal";
import AddAssetModal from "../components/AddAssetModal";
import { reportService } from "../../../../services/reportService";

export default function Dashboard() {
  const navigate = useNavigate();
  const [showAddProperty, setShowAddProperty] = useState(false);
  const [showProfileBanner, setShowProfileBanner] = useState(false);
  const [isAddChoiceModalOpen, setIsAddChoiceModalOpen] = useState(false);
  const [isAddAssetModalOpen, setIsAddAssetModalOpen] = useState(false);
  const [completionStatus, setCompletionStatus] = useState(null);
  const [realData, setRealData] = useState({
    activeProperties: 0,
    totalViews: 0,
    totalLeads: 0,
    totalRevenue: 0,
    conversionRate: 0,
    avgResponseTime: 0,
  });
  const [reportLoading, setReportLoading] = useState(false);

  const handleGenerateReport = async () => {
    if (reportLoading) return; // Prevent double execution
    try {
      setReportLoading(true);
      const filename = await reportService.downloadDashboardReport();
      toast.success(`Report downloaded: ${filename}`, {
        description: "Your dashboard report has been generated successfully.",
      });
    } catch (error) {
      console.error("Failed to generate report:", error);
      toast.error(
        error.message || "Failed to generate report. Please try again.",
      );
    } finally {
      setReportLoading(false);
    }
  };

  useEffect(() => {
    const checkProfileCompletion = async () => {
      try {
        const status = await getSellerProfileCompletionStatus();
        setCompletionStatus(status);
        if (status.completion_percentage < 80) {
          setShowProfileBanner(true);
        }
      } catch (error) {
        console.error("Error checking profile completion:", error);
      }
    };

    const fetchDashboardData = async () => {
      try {
        // Try both 'user' and 'seller' keys from localStorage
        const user = JSON.parse(localStorage.getItem("user") || "{}");
        const seller = JSON.parse(localStorage.getItem("seller") || "{}");
        const agentData = JSON.parse(localStorage.getItem("agentData") || "{}");

        // Try multiple extraction methods
        const userId =
          user.user_id ||
          user.id ||
          seller.user_id ||
          seller.id ||
          agentData.user_id ||
          agentData.id ||
          localStorage.getItem("userId") ||
          localStorage.getItem("sellerId");

        console.log("Dashboard - User from localStorage:", user);
        console.log("Dashboard - Seller from localStorage:", seller);
        console.log("Dashboard - AgentData from localStorage:", agentData);
        console.log(
          "Dashboard - All localStorage keys:",
          Object.keys(localStorage),
        );
        console.log("Dashboard - Using userId:", userId);

        if (!userId) {
          console.log("Dashboard - No userId found");
          return;
        }

        // Fetch properties directly to get real counts
        const timestamp = new Date().getTime();
        const token =
          localStorage.getItem("sellerToken") || localStorage.getItem("token");
        const headers = {
          "Cache-Control": "no-cache",
          Pragma: "no-cache",
          "Content-Type": "application/json",
        };
        if (token) {
          headers["Authorization"] = `Bearer ${token}`;
        }

        const propertiesResponse = await fetch(
          `http://localhost:8090/api/properties/?listed_by_id=${userId}&_t=${timestamp}`,
          { cache: "no-store", headers },
        );

        if (propertiesResponse.ok) {
          const propertiesData = await propertiesResponse.json();
          const allProperties =
            propertiesData.properties ||
            propertiesData.results ||
            propertiesData ||
            [];

          const activeProperties = allProperties.filter(
            (p) => p.status === "active",
          );
          const totalViews = activeProperties.reduce(
            (sum, p) => sum + (p.views || 0),
            0,
          );

          console.log(
            "Dashboard - Active properties:",
            activeProperties.length,
          );

          setRealData({
            activeProperties: activeProperties.length,
            totalViews,
            totalLeads: 0,
            totalRevenue: 0,
            conversionRate: 0,
            avgResponseTime: 0,
          });
        } else {
          console.log("Dashboard - Failed to fetch properties");
        }
      } catch (error) {
        console.error("Error fetching dashboard data:", error);
      }
    };

    checkProfileCompletion();
    fetchDashboardData();
  }, []);

  // Bangalore market data (mock for UI demonstration)
  const bangaloreMarketData = {
    avgPropertyPrice: 8500000, // ₹85 Lakhs
    marketGrowth: 12.5, // 12.5% YoY
    totalListings: 15420,
    avgDaysOnMarket: 45,
    hotAreas: ["Whitefield", "Electronic City", "Koramangala", "Indiranagar"],
  };

  if (showAddProperty) {
    return (
      <div className="flex-1 overflow-y-auto p-6 space-y-6">
        <div className="flex items-center justify-between mb-8">
          <div>
            <h1 className="text-3xl font-bold text-foreground">
              Add New Property
            </h1>
            <p className="text-muted-foreground mt-1">
              Fill in the details to list your property
            </p>
          </div>
          <Button variant="outline" onClick={() => setShowAddProperty(false)}>
            <X className="mr-2 h-4 w-4" />
            Back to Dashboard
          </Button>
        </div>
        <ListingForm />
      </div>
    );
  }

  return (
    <div className="flex-1 overflow-y-auto p-6 space-y-6 bg-gray-50">
      {/* Profile Completion Banner */}
      {showProfileBanner && completionStatus && (
        <div className="bg-gradient-to-r from-blue-50 to-blue-100 border border-blue-200 rounded-xl p-6 mb-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center">
                <Building className="w-6 h-6 text-white" />
              </div>
              <div>
                <h3 className="text-lg font-semibold text-gray-900">
                  Complete Your Seller Profile
                </h3>
                <p className="text-gray-600">
                  Your profile is {completionStatus.completion_percentage}%
                  complete. Complete it to get better leads!
                </p>
                <div className="w-64 bg-gray-200 rounded-full h-2 mt-2">
                  <div
                    className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                    style={{
                      width: `${completionStatus.completion_percentage}%`,
                    }}
                  ></div>
                </div>
              </div>
            </div>
            <div className="flex items-center space-x-3">
              <Button
                onClick={() => navigate("/seller/profile-completion")}
                className="bg-blue-600 hover:bg-blue-700 text-white"
              >
                Complete Profile
              </Button>
              <Button
                variant="outline"
                onClick={() => setShowProfileBanner(false)}
                size="sm"
              >
                <X className="w-4 h-4" />
              </Button>
            </div>
          </div>
        </div>
      )}

      {/* Header */}
      <div className="bg-white rounded-xl shadow-sm border p-6">
        <div className="flex items-center justify-between">
          <div>
            <div className="flex items-center gap-2">
              <MapPin className="w-4 h-4 text-blue-600" />
              <span className="text-gray-600">Bangalore, Karnataka</span>
              <Badge variant="outline" className="ml-2">
                Live Market Data
              </Badge>
            </div>
          </div>
          <div className="flex gap-3">
            <Button
              onClick={handleGenerateReport}
              disabled={reportLoading}
              variant="outline"
              className="border-blue-500 text-blue-600 hover:bg-blue-50 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              <BarChart3 className="mr-2 h-4 w-4" />
              {reportLoading ? "Generating..." : "Generate Report"}
            </Button>
            <Button
              onClick={() => setShowAddProperty(true)}
              className="bg-blue-600 hover:bg-blue-700 text-white"
            >
              <Plus className="mr-2 h-4 w-4" />
              Add Property
            </Button>
          </div>
        </div>
      </div>

      {/* Key Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        <Card className="bg-white shadow-sm border">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">
                  Active Properties
                </p>
                <p className="text-3xl font-bold text-gray-900">
                  {realData.activeProperties}
                </p>
                <p className="text-xs text-gray-500 mt-1">Real Data</p>
              </div>
              <div className="p-3 bg-blue-100 rounded-lg">
                <Building className="w-6 h-6 text-blue-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-white shadow-sm border">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">
                  Total Revenue
                </p>
                <p className="text-3xl font-bold text-gray-900">
                  ₹{realData.totalRevenue}
                </p>
                <p className="text-xs text-gray-500 mt-1">Real Data</p>
              </div>
              <div className="p-3 bg-green-100 rounded-lg">
                <IndianRupee className="w-6 h-6 text-green-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-white shadow-sm border">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">Total Leads</p>
                <p className="text-3xl font-bold text-gray-900">
                  {realData.totalLeads}
                </p>
                <p className="text-xs text-gray-500 mt-1">Real Data</p>
              </div>
              <div className="p-3 bg-orange-100 rounded-lg">
                <Users className="w-6 h-6 text-orange-600" />
              </div>
            </div>
          </CardContent>
        </Card>

        <Card className="bg-white shadow-sm border">
          <CardContent className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">
                  Property Views
                </p>
                <p className="text-3xl font-bold text-gray-900">
                  {realData.totalViews}
                </p>
                <p className="text-xs text-gray-500 mt-1">Real Data</p>
              </div>
              <div className="p-3 bg-purple-100 rounded-lg">
                <Eye className="w-6 h-6 text-purple-600" />
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Bangalore Market Insights */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card className="bg-white shadow-sm border">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <BarChart3 className="w-5 h-5 text-blue-600" />
              Bangalore Market Overview
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex justify-between items-center">
              <span className="text-sm text-gray-600">Avg Property Price</span>
              <span className="font-semibold">
                ₹{(bangaloreMarketData.avgPropertyPrice / 100000).toFixed(0)}L
              </span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-sm text-gray-600">Market Growth (YoY)</span>
              <span className="font-semibold text-green-600">
                +{bangaloreMarketData.marketGrowth}%
              </span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-sm text-gray-600">Total Listings</span>
              <span className="font-semibold">
                {bangaloreMarketData.totalListings.toLocaleString()}
              </span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-sm text-gray-600">Avg Days on Market</span>
              <span className="font-semibold">
                {bangaloreMarketData.avgDaysOnMarket} days
              </span>
            </div>
            <p className="text-xs text-gray-500 mt-2">Mock Data for UI Demo</p>
          </CardContent>
        </Card>

        <Card className="bg-white shadow-sm border">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Target className="w-5 h-5 text-orange-600" />
              Performance Metrics
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex justify-between items-center">
              <span className="text-sm text-gray-600">Conversion Rate</span>
              <span className="font-semibold">{realData.conversionRate}%</span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-sm text-gray-600">Avg Response Time</span>
              <span className="font-semibold">{realData.avgResponseTime}h</span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-sm text-gray-600">Lead Quality Score</span>
              <span className="font-semibold">0/10</span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-sm text-gray-600">Portfolio Value</span>
              <span className="font-semibold">₹0</span>
            </div>
            <p className="text-xs text-gray-500 mt-2">Real Data</p>
          </CardContent>
        </Card>
      </div>

      {/* Hot Areas in Bangalore */}
      <Card className="bg-white shadow-sm border">
        <CardHeader>
          <CardTitle>Hot Areas in Bangalore</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            {bangaloreMarketData.hotAreas.map((area, index) => (
              <div
                key={area}
                className="p-4 bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg border border-blue-200"
              >
                <h3 className="font-semibold text-gray-900">{area}</h3>
                <p className="text-sm text-gray-600 mt-1">High Demand</p>
                <p className="text-xs text-blue-600 font-medium mt-2">
                  ₹{8000 + index * 500}K/sq ft
                </p>
              </div>
            ))}
          </div>
          <p className="text-xs text-gray-500 mt-4">Mock Data for UI Demo</p>
        </CardContent>
      </Card>

      {/* Quick Actions */}
      <Card className="bg-white shadow-sm border">
        <CardHeader>
          <CardTitle>Quick Actions</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <Button
              variant="outline"
              className="h-16 flex flex-col items-center justify-center space-y-2"
            >
              <Building className="w-5 h-5" />
              <span>Manage Properties</span>
            </Button>
            <Button
              variant="outline"
              className="h-16 flex flex-col items-center justify-center space-y-2"
            >
              <Users className="w-5 h-5" />
              <span>View Leads</span>
            </Button>
            <Button
              variant="outline"
              className="h-16 flex flex-col items-center justify-center space-y-2"
            >
              <BarChart3 className="w-5 h-5" />
              <span>Analytics</span>
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Add Property Choice Modal */}
      <Dialog
        open={isAddChoiceModalOpen}
        onOpenChange={setIsAddChoiceModalOpen}
      >
        <DialogContent className="max-w-4xl">
          <AddPropertyChoiceModal
            onClose={() => setIsAddChoiceModalOpen(false)}
            onAddAsset={() => {
              setIsAddChoiceModalOpen(false);
              setIsAddAssetModalOpen(true);
            }}
            onListProperty={() => {
              setIsAddChoiceModalOpen(false);
              setShowAddProperty(true);
            }}
          />
        </DialogContent>
      </Dialog>

      {/* Add Asset Modal */}
      <Dialog open={isAddAssetModalOpen} onOpenChange={setIsAddAssetModalOpen}>
        <DialogContent className="max-w-3xl">
          <AddAssetModal onClose={() => setIsAddAssetModalOpen(false)} />
        </DialogContent>
      </Dialog>
    </div>
  );
}
