import { useState, useEffect } from "react";

export default function Documents() {
  const [search, setSearch] = useState("");
  const [selectedProperty, setSelectedProperty] = useState("");
  const [properties, setProperties] = useState([]);
  const [documents, setDocuments] = useState([]);

  // Fetch properties from backend
  useEffect(() => {
    const fetchProperties = async () => {
      try {
        const response = await fetch('http://localhost:8000/api/listings/');
        if (response.ok) {
          const data = await response.json();
          console.log('Properties API response:', data);
          
          // Handle both paginated and direct array responses
          const propertiesList = data.results || data || [];
          setProperties(propertiesList);
          
          if (propertiesList.length > 0) {
            setSelectedProperty(propertiesList[0].id);
          }
        } else {
          console.error('Failed to fetch properties:', response.status);
          // Use mock data as fallback
          const mockProperties = [
            { id: 1, title: "Modern 2BR Apartment - Downtown" },
            { id: 2, title: "3BR Villa with Garden - Suburbs" },
            { id: 3, title: "Luxury Villa - Whitefield" }
          ];
          setProperties(mockProperties);
          setSelectedProperty(mockProperties[0].id);
        }
      } catch (error) {
        console.error('Error fetching properties:', error);
        // Use mock data as fallback
        const mockProperties = [
          { id: 1, title: "Modern 2BR Apartment - Downtown" },
          { id: 2, title: "3BR Villa with Garden - Suburbs" },
          { id: 3, title: "Luxury Villa - Whitefield" }
        ];
        setProperties(mockProperties);
        setSelectedProperty(mockProperties[0].id);
      }
    };
    
    fetchProperties();
  }, []);

  // Fetch documents for selected property
  useEffect(() => {
    if (!selectedProperty) return;
    
    const fetchDocuments = async () => {
      try {
        console.log('Fetching documents for property:', selectedProperty);
        const response = await fetch(`http://localhost:8000/api/documents/?listing=${selectedProperty}`);
        console.log('Documents API response status:', response.status);
        
        if (response.ok) {
          const data = await response.json();
          console.log('Documents API response data:', data);
          
          // Since we're filtering by listing, expect direct array
          const documentsList = Array.isArray(data) ? data : [];
          console.log('Processed documents list:', documentsList);
          setDocuments(documentsList);
        } else {
          console.error('Failed to fetch documents:', response.status);
          setDocuments([]);
        }
      } catch (error) {
        console.error('Error fetching documents:', error);
        setDocuments([]);
      }
    };
    
    fetchDocuments();
  }, [selectedProperty]);

  const formatDate = (dateString) => {
    return new Date(dateString).toLocaleDateString('en-IN', {
      year: 'numeric',
      month: '2-digit', 
      day: '2-digit'
    });
  };

  const getStatusConfig = (status) => {
    switch (status) {
      case 'verified':
        return 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-300';
      case 'pending_review':
        return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-300';
      case 'rejected':
        return 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-300';
      default:
        return 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300';
    }
  };

  const getStatusLabel = (status) => {
    switch (status) {
      case 'verified': return 'Verified';
      case 'pending_review': return 'Pending';
      case 'rejected': return 'Rejected';
      default: return 'Uploaded';
    }
  };

  return (
    <div className="flex-1 p-8 overflow-y-auto">
      <div className="flex justify-between items-center mb-8">
        <h2 className="text-3xl font-bold text-gray-900 dark:text-white">Documents</h2>
      </div>
      
      <div className="bg-white dark:bg-gray-800 p-6 rounded-lg shadow-sm">
        {/* Property Selection */}
        <div className="mb-6">
          <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
            Select Property
          </label>
          <select
            value={selectedProperty}
            onChange={(e) => setSelectedProperty(e.target.value)}
            className="w-full max-w-md px-3 py-2 border border-gray-200 dark:border-gray-700 rounded-lg bg-white dark:bg-gray-800 focus:ring-blue-500 focus:border-blue-500"
          >
            {properties.map((property) => (
              <option key={property.id} value={property.id}>
                {property.title}
              </option>
            ))}
          </select>
        </div>

        {/* Search
        <div className="relative mb-6">
          <input 
            className="w-full pl-4 pr-4 py-3 rounded-lg border border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-800 focus:ring-blue-500 focus:border-blue-500" 
            placeholder="Search documents..." 
            type="text"
            value={search}
            onChange={(e) => setSearch(e.target.value)}
          />
        </div> */}
        
        <div className="overflow-x-auto">
          <table className="w-full text-sm text-left text-gray-500 dark:text-gray-400">
            <thead className="text-xs text-gray-700 dark:text-gray-400 uppercase bg-gray-50 dark:bg-gray-700/50">
              <tr>
                <th className="px-6 py-3" scope="col">Name</th>
                <th className="px-6 py-3" scope="col">Upload Date</th>
                <th className="px-6 py-3" scope="col">Type</th>
                <th className="px-6 py-3" scope="col">Status</th>
                <th className="px-6 py-3" scope="col">Actions</th>
              </tr>
            </thead>
            <tbody>
              {documents.length === 0 ? (
                <tr className="bg-white dark:bg-gray-800">
                  <td colSpan="5" className="px-6 py-8 text-center text-gray-500">
                    No documents found for this property
                  </td>
                </tr>
              ) : (
                documents
                  .filter(doc => 
                    search === "" || 
                    doc.original_filename.toLowerCase().includes(search.toLowerCase()) ||
                    doc.display_name.toLowerCase().includes(search.toLowerCase())
                  )
                  .map((doc) => (
                    <tr key={doc.id} className="bg-white dark:bg-gray-800 border-b dark:border-gray-800">
                      <th className="px-6 py-4 font-medium text-gray-900 dark:text-white whitespace-nowrap" scope="row">
                        <div>
                          <div className="font-medium">{doc.original_filename}</div>
                          <div className="text-xs text-gray-500">{doc.file_size_mb || (doc.file_size ? (doc.file_size / (1024 * 1024)).toFixed(2) : '0')} MB</div>
                        </div>
                      </th>
                      <td className="px-6 py-4">{formatDate(doc.uploaded_at)}</td>
                      <td className="px-6 py-4">{doc.display_name}</td>
                      <td className="px-6 py-4">
                        <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusConfig(doc.status)}`}>
                          {getStatusLabel(doc.status)}
                        </span>
                      </td>
                      <td className="px-6 py-4">
                       


<button
  className="px-3 py-1 text-xs font-medium text-blue-600 border border-blue-600 bg-white 
             hover:bg-blue-50 hover:text-blue-800 hover:border-blue-800 
             rounded transition-colors"
  onClick={() => doc.document && window.open(doc.document, "_blank")}
>
  View Document
</button>





                      </td>
                    </tr>
                  ))
              )}
            </tbody>
          </table>
        </div>
      </div>
    </div>
  );
}