import { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../components/ui/tabs";
import { Button } from "../components/ui/button";
import { PricingRecommendationCard } from "../../../../components/pricing/PricingRecommendationCard";
import { BuyerInsightsCard } from "../../../../components/pricing/BuyerInsightsCard";
import { Loader2, TrendingUp, IndianRupee, Target, Sparkles, AlertCircle } from "lucide-react";
import axios from 'axios';

const API_BASE = 'http://localhost:8090';

export default function DynamicPricing() {
  const sellerToken = localStorage.getItem('sellerToken');
  const sellerId = localStorage.getItem('sellerId');
  const [generating, setGenerating] = useState(false);
  const [loading, setLoading] = useState(true);
  const [message, setMessage] = useState('');
  const [recommendations, setRecommendations] = useState<any[]>([]);

  const fetchRecommendations = async () => {
    if (!sellerId || !sellerToken) return;
    try {
      console.log('Fetching recommendations for seller:', sellerId);
      const response = await axios.get(`${API_BASE}/api/pricing/seller/${sellerId}/recommendations`, {
        headers: { 'Authorization': `Bearer ${sellerToken}` }
      });
      console.log('API Response:', response.data);
      const recs = response.data?.data || [];
      console.log('Extracted recommendations:', recs);
      setRecommendations(recs);
    } catch (error: any) {
      console.error('Failed to fetch recommendations:', error.response?.data || error.message);
      setRecommendations([]);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchRecommendations();
  }, [sellerId, sellerToken]);

  const handleGenerateForAll = async () => {
    setGenerating(true);
    setMessage('');
    try {
      const response = await axios.post(
        `${API_BASE}/api/pricing/seller/${sellerId}/generate-all`,
        {},
        { headers: { 'Authorization': `Bearer ${sellerToken}` } }
      );
      
      const result = response.data?.data;
      console.log('Generate result:', result);
      
      if (result.success) {
        const { total, eligible, generated, excluded } = result;
        
        if (generated > 0) {
          setMessage(`✅ Generated ${generated} of ${total} pricing recommendations successfully!`);
        } else if (excluded && excluded.length > 0) {
          const reasons = excluded.map(e => `${e.title}: ${e.reason}`).join('; ');
          setMessage(`⚠️ No recommendations generated. ${reasons}`);
        } else {
          setMessage(`⚠️ No eligible properties found (${total} total properties)`);
        }
        
        setTimeout(() => fetchRecommendations(), 1000);
      } else {
        setMessage(`❌ ${result.message || 'Failed to generate recommendations'}`);
      }
    } catch (error: any) {
      console.error('Generate error:', error);
      setMessage(`❌ ${error.response?.data?.detail || 'Failed to generate recommendations'}`);
    } finally {
      setGenerating(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-96">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  const activeRecs = recommendations?.filter(r => r.status === 'pending') || [];
  const acceptedRecs = recommendations?.filter(r => r.status === 'accepted') || [];
  const rejectedRecs = recommendations?.filter(r => r.status === 'rejected') || [];

  return (
    <div className="space-y-6 p-6">
      <div>
        <h1 className="text-3xl font-bold">Dynamic Pricing Intelligence</h1>
        <p className="text-muted-foreground mt-2">
          AI-powered pricing recommendations using 7 models including ML valuation & market prediction
        </p>
      </div>

      <div className="flex justify-between items-center mb-4">
        <div>
          {message && (
            <div className={`flex items-center gap-2 px-4 py-2 rounded-lg ${
              message.includes('✅') ? 'bg-green-50 text-green-700' : 'bg-red-50 text-red-700'
            }`}>
              <AlertCircle className="h-4 w-4" />
              <span className="text-sm font-medium">{message}</span>
            </div>
          )}
        </div>
        <div className="flex gap-2">
          <Button 
            onClick={fetchRecommendations}
            variant="outline"
            disabled={loading}
          >
            {loading ? <Loader2 className="h-4 w-4 mr-2 animate-spin" /> : null}
            Refresh
          </Button>
          <Button 
            onClick={handleGenerateForAll}
            disabled={generating}
            className="bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700"
          >
            {generating ? (
              <Loader2 className="h-4 w-4 mr-2 animate-spin" />
            ) : (
              <Sparkles className="h-4 w-4 mr-2" />
            )}
            Generate Recommendations Now
          </Button>
        </div>
      </div>

      <div className="grid gap-4 md:grid-cols-3">
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Active Recommendations</CardTitle>
            <Target className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{activeRecs.length}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Accepted</CardTitle>
            <TrendingUp className="h-4 w-4 text-green-600" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{acceptedRecs.length}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Avg Confidence</CardTitle>
            <IndianRupee className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">
              {activeRecs.length > 0
                ? Math.round(
                    activeRecs.reduce((sum, r) => sum + (r.confidence_level === 'HIGH' ? 90 : r.confidence_level === 'MEDIUM' ? 70 : 50), 0) /
                      activeRecs.length
                  ) + '%'
                : 'N/A'}
            </div>
            <p className="text-xs text-muted-foreground mt-1">ML-Enhanced</p>
          </CardContent>
        </Card>
      </div>

      <Tabs defaultValue="active" className="space-y-4">
        <TabsList>
          <TabsTrigger value="active">Active ({activeRecs.length})</TabsTrigger>
          <TabsTrigger value="accepted">Accepted ({acceptedRecs.length})</TabsTrigger>
          <TabsTrigger value="rejected">Rejected ({rejectedRecs.length})</TabsTrigger>
        </TabsList>

        <TabsContent value="active" className="space-y-4">
          {activeRecs.length === 0 ? (
            <Card>
              <CardContent className="pt-6">
                <p className="text-center text-muted-foreground">
                  No active pricing recommendations. New recommendations are generated daily at 2 AM.
                </p>
              </CardContent>
            </Card>
          ) : (
            activeRecs.map((rec) => (
              <div key={rec.id} className="grid gap-4 md:grid-cols-2">
                <PricingRecommendationCard recommendation={rec} />
                <BuyerInsightsCard propertyId={rec.property_id} />
              </div>
            ))
          )}
        </TabsContent>

        <TabsContent value="accepted" className="space-y-4">
          {acceptedRecs.length === 0 ? (
            <Card>
              <CardContent className="pt-6">
                <p className="text-center text-muted-foreground">No accepted recommendations yet.</p>
              </CardContent>
            </Card>
          ) : (
            acceptedRecs.map((rec) => (
              <PricingRecommendationCard key={rec.id} recommendation={rec} />
            ))
          )}
        </TabsContent>

        <TabsContent value="rejected" className="space-y-4">
          {rejectedRecs.length === 0 ? (
            <Card>
              <CardContent className="pt-6">
                <p className="text-center text-muted-foreground">No rejected recommendations.</p>
              </CardContent>
            </Card>
          ) : (
            rejectedRecs.map((rec) => (
              <PricingRecommendationCard key={rec.id} recommendation={rec} />
            ))
          )}
        </TabsContent>
      </Tabs>
    </div>
  );
}
