

import { Card, CardContent, CardHeader, CardTitle } from "../../../ui/card";
import { Button } from "../../../ui/button";
import { Badge } from "../../../ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../../../ui/select";
import { Video, Clock, Users, Upload, Settings, Play, Calendar, Eye, Share2 } from "lucide-react";
import { mockBookings, mockBuyers, mockProperties, getBuyerById, getPropertyById } from "../lib/mock-data";
import { useState } from "react";

export default function Hosting() {
  const [sessionFilter, setSessionFilter] = useState<string>("all");
  const [propertyFilter, setPropertyFilter] = useState<string>("all");
  
  const getStatusColor = (status: string) => {
    switch (status) {
      case "confirmed":
        return "bg-green-100 text-green-800";
      case "scheduled":
        return "bg-blue-100 text-blue-800";
      case "completed":
        return "bg-gray-100 text-gray-800";
      case "cancelled":
        return "bg-red-100 text-red-800";
      case "live":
        return "bg-red-100 text-red-800 animate-pulse";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const mockVirtualTours = [
    {
      id: "tour-1",
      propertyId: "prop-1", 
      title: "Downtown Apartment Virtual Tour",
      scheduledDate: new Date("2024-12-20T14:00:00"),
      duration: "30 minutes",
      attendees: 5,
      maxAttendees: 10,
      status: "scheduled",
      tourLink: "https://tour.example.com/abc123",
      recordingEnabled: true
    },
    {
      id: "tour-2",
      propertyId: "prop-2",
      title: "Luxury Villa Showcase", 
      scheduledDate: new Date("2024-12-18T16:30:00"),
      duration: "45 minutes",
      attendees: 3,
      maxAttendees: 8,
      status: "live",
      tourLink: "https://tour.example.com/def456",
      recordingEnabled: false
    }
  ];

  const virtualBookings = mockBookings.filter(b => b.type === "virtual");

  const formatDateTime = (date: Date) => {
    return new Intl.DateTimeFormat('en-US', {
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    }).format(date);
  };

  return (
    <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Virtual Tours & Hosting</h2>
      </div>
        <div className="flex items-center justify-between mb-6">
          <div>
            <h1 className="text-2xl font-bold text-foreground">Virtual Tours & Hosting</h1>
            <p className="text-muted-foreground">Manage virtual property tours and online viewings</p>
          </div>
          <div className="flex gap-2">
            <Button variant="outline">
              <Upload className="mr-2 h-4 w-4" />
              Upload 360° Tour
            </Button>
            <Button>
              <Video className="mr-2 h-4 w-4" />
              Schedule Tour
            </Button>
          </div>
        </div>

        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-6">
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Live Tours</p>
                  <p className="text-2xl font-bold text-red-600">
                    {mockVirtualTours.filter(t => t.status === "live").length}
                  </p>
                </div>
                <div className="w-8 h-8 bg-red-100 rounded-lg flex items-center justify-center">
                  <Video className="w-5 h-5 text-red-600" />
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Scheduled</p>
                  <p className="text-2xl font-bold text-blue-600">
                    {mockVirtualTours.filter(t => t.status === "scheduled").length}
                  </p>
                </div>
                <div className="w-8 h-8 bg-blue-100 rounded-lg flex items-center justify-center">
                  <Calendar className="w-5 h-5 text-blue-600" />
                </div>
              </div>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Total Views</p>
                  <p className="text-2xl font-bold text-green-600">1,248</p>
                </div>
                <div className="w-8 h-8 bg-green-100 rounded-lg flex items-center justify-center">
                  <Eye className="w-5 h-5 text-green-600" />
                </div>
              </div>
              <p className="text-xs text-green-600 mt-1">+23% this week</p>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-4">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-muted-foreground">Conversion Rate</p>
                  <p className="text-2xl font-bold text-purple-600">18%</p>
                </div>
                <div className="w-8 h-8 bg-purple-100 rounded-lg flex items-center justify-center">
                  <Users className="w-5 h-5 text-purple-600" />
                </div>
              </div>
              <p className="text-xs text-muted-foreground mt-1">Tours to inquiries</p>
            </CardContent>
          </Card>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Virtual Tours */}
          <Card>
            <CardHeader className="flex flex-row items-center justify-between">
              <CardTitle>Virtual Tours</CardTitle>
              <Button size="sm">
                <Video className="mr-2 h-4 w-4" />
                New Tour
              </Button>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {mockVirtualTours.map((tour) => {
                  const property = getPropertyById(tour.propertyId);
                  if (!property) return null;
                  
                  return (
                    <div key={tour.id} className="border border-border rounded-lg p-4">
                      <div className="flex items-start justify-between mb-3">
                        <div className="flex items-center gap-3">
                          <img 
                            src={property.images?.[0] || "https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=40&h=40"}
                            alt={property.title}
                            className="w-10 h-10 rounded object-cover"
                          />
                          <div>
                            <h4 className="font-semibold text-foreground">{tour.title}</h4>
                            <p className="text-sm text-muted-foreground">{property.location}</p>
                          </div>
                        </div>
                        <Badge className={getStatusColor(tour.status)}>
                          {tour.status === "live" && <div className="w-2 h-2 bg-red-600 rounded-full mr-1 animate-pulse" />}
                          {tour.status.charAt(0).toUpperCase() + tour.status.slice(1)}
                        </Badge>
                      </div>
                      
                      <div className="grid grid-cols-2 gap-4 mb-3 text-sm">
                        <div>
                          <span className="text-muted-foreground">Scheduled:</span>
                          <p className="font-medium">{formatDateTime(tour.scheduledDate)}</p>
                        </div>
                        <div>
                          <span className="text-muted-foreground">Duration:</span>
                          <p className="font-medium">{tour.duration}</p>
                        </div>
                        <div>
                          <span className="text-muted-foreground">Attendees:</span>
                          <p className="font-medium">{tour.attendees}/{tour.maxAttendees}</p>
                        </div>
                        <div>
                          <span className="text-muted-foreground">Recording:</span>
                          <p className="font-medium">{tour.recordingEnabled ? "Enabled" : "Disabled"}</p>
                        </div>
                      </div>
                      
                      <div className="flex gap-2">
                        {tour.status === "live" ? (
                          <Button size="sm" className="bg-red-600 hover:bg-red-700">
                            <Video className="mr-2 h-4 w-4" />
                            Join Live
                          </Button>
                        ) : tour.status === "scheduled" ? (
                          <Button size="sm">
                            <Play className="mr-2 h-4 w-4" />
                            Start Tour
                          </Button>
                        ) : (
                          <Button size="sm" variant="outline">
                            <Eye className="mr-2 h-4 w-4" />
                            View Recording
                          </Button>
                        )}
                        <Button size="sm" variant="outline">
                          <Share2 className="mr-2 h-4 w-4" />
                          Share
                        </Button>
                        <Button size="sm" variant="outline">
                          <Settings className="mr-2 h-4 w-4" />
                          Settings
                        </Button>
                      </div>
                    </div>
                  );
                })}
              </div>
            </CardContent>
          </Card>

          {/* Virtual Bookings */}
          <Card>
            <CardHeader>
              <CardTitle>Upcoming Virtual Viewings</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {virtualBookings.slice(0, 5).map((booking) => {
                  const buyer = getBuyerById(booking.buyerId);
                  const property = getPropertyById(booking.propertyId);
                  
                  if (!buyer || !property) return null;
                  
                  return (
                    <div key={booking.id} className="flex items-center justify-between p-3 bg-muted/30 rounded-lg">
                      <div className="flex items-center gap-3">
                        <div className="w-8 h-8 bg-primary rounded-full flex items-center justify-center">
                          <span className="text-primary-foreground text-xs font-medium">
                            {buyer.firstName?.[0]}{buyer.lastName?.[0]}
                          </span>
                        </div>
                        <div>
                          <p className="text-sm font-medium text-foreground">
                            {buyer.firstName} {buyer.lastName}
                          </p>
                          <p className="text-xs text-muted-foreground">
                            {formatDateTime(booking.scheduledDate)}
                          </p>
                        </div>
                      </div>
                      <div className="text-right">
                        <p className="text-sm font-medium text-foreground">{property.title}</p>
                        <Badge className={getStatusColor(booking.status)} variant="outline">
                          {booking.status}
                        </Badge>
                      </div>
                    </div>
                  );
                })}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Tour Analytics */}
        <div className="mt-6">
          <Card>
            <CardHeader>
              <CardTitle>Tour Performance Analytics</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div className="p-4 bg-blue-50 rounded-lg border border-blue-200">
                  <h4 className="font-semibold text-blue-900 mb-2">Average Tour Duration</h4>
                  <p className="text-2xl font-bold text-blue-800">12m 34s</p>
                  <p className="text-xs text-blue-600 mt-1">+2m vs last month</p>
                </div>
                
                <div className="p-4 bg-green-50 rounded-lg border border-green-200">
                  <h4 className="font-semibold text-green-900 mb-2">Engagement Rate</h4>
                  <p className="text-2xl font-bold text-green-800">78%</p>
                  <p className="text-xs text-green-600 mt-1">Viewers staying &gt;50% of tour</p>
                </div>
                
                <div className="p-4 bg-purple-50 rounded-lg border border-purple-200">
                  <h4 className="font-semibold text-purple-900 mb-2">Lead Generation</h4>
                  <p className="text-2xl font-bold text-purple-800">24</p>
                  <p className="text-xs text-purple-600 mt-1">Leads from virtual tours</p>
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
    </div>
  );
}
