import { Card, CardContent, CardHeader } from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../components/ui/tabs";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "../components/ui/select";
import { Input } from "../components/ui/input";
import { Avatar, AvatarFallback } from "../components/ui/avatar";
import { ScrollArea } from "../components/ui/scroll-area";
import {
  Phone,
  MessageCircle,
  Search,
  ChevronDown,
  ChevronLeft,
  ChevronRight,
  Archive,
} from "lucide-react";
import { useState, useEffect } from "react";

interface Lead {
  id: string;
  name: string; // API returns 'name', not 'buyer_name'
  phone: string; // API returns 'phone', not 'buyer_phone'
  email?: string; // API returns 'email', not 'buyer_email'
  buyer_name?: string;
  buyer_email?: string;
  buyer_phone?: string;
  property_id: number;
  property_title?: string;
  message?: string;
  status: string;
  created_at: string;
  score?: number;
  ai_category?: string;
  budget?: string;
}

export default function Leads() {
  const [rentalLeads, setRentalLeads] = useState<Lead[]>([]);
  const [saleLeads, setSaleLeads] = useState<Lead[]>([]);
  const [loading, setLoading] = useState(true);
  const [activeEnquiryType, setActiveEnquiryType] = useState("rental");
  const [activeTab, setActiveTab] = useState("all");
  const [searchTerm, setSearchTerm] = useState("");
  const [filterType, setFilterType] = useState("all");
  const [sortOrder, setSortOrder] = useState("recent");
  const [currentPage, setCurrentPage] = useState(1);
  const leadsPerPage = 10;

  useEffect(() => {
    fetchLeads();
    const interval = setInterval(fetchLeads, 30000);
    return () => clearInterval(interval);
  }, [activeEnquiryType]);

  const fetchLeads = async () => {
    try {
      setLoading(true);

      // Try multiple localStorage keys for user data
      let userStr =
        localStorage.getItem("seller") ||
        localStorage.getItem("user") ||
        localStorage.getItem("currentUser") ||
        localStorage.getItem("agent");

      let userId = null;

      if (userStr) {
        try {
          const user = JSON.parse(userStr);
          console.log("Seller Dashboard - Current user:", user);
          userId = user.id || user.user_id || user.userId || user.user?.id;
        } catch (e) {
          console.error("Error parsing user data:", e);
        }
      }

      // If no user found in JSON format, check for direct ID storage
      if (!userId) {
        userId =
          localStorage.getItem("sellerId") ||
          localStorage.getItem("user_id") ||
          localStorage.getItem("agentId");
      }

      // If no user found or no user ID, use fallback seller ID
      if (!userId) {
        console.log(
          "Seller Dashboard - No user found in localStorage, using fallback seller ID"
        );
        userId = "e7aec4cf-8249-4a5c-ad71-b95011819f80";
      }

      console.log("Seller Dashboard - Using user ID:", userId);

      const API_BASE =
        import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
      const token =
        localStorage.getItem("access_token") ||
        localStorage.getItem("token") ||
        localStorage.getItem("agentToken") ||
        localStorage.getItem("sellerToken");
      
      // First, debug what leads exist
      const debugUrl = `${API_BASE}/leads-management/debug/seller/${userId}/leads`;
      console.log("Debug API URL:", debugUrl);
      
      try {
        const debugResponse = await fetch(debugUrl, {
          headers: {
            "Content-Type": "application/json",
          },
        });
        
        if (debugResponse.ok) {
          const debugData = await debugResponse.json();
          console.log("DEBUG - All leads for seller:", debugData);
          console.log("DEBUG - Lead details:", debugData.leads?.map(lead => ({
            id: lead.id,
            lead_type: lead.lead_type,
            lead_type_debug: lead.lead_type_debug,
            status: lead.status,
            buyer_name: lead.buyer_name
          })));
        }
      } catch (debugError) {
        console.error("Debug request failed:", debugError);
      }
      
      // Fetch all leads first, then filter client-side
      const apiUrl = `${API_BASE}/leads-management/leads/seller/${userId}`;
      console.log("Seller Dashboard - API URL:", apiUrl);

      const response = await fetch(apiUrl, {
        headers: {
          "Content-Type": "application/json",
        },
      });

      console.log("Seller Dashboard - Response status:", response.status);
      if (!response.ok) {
        const errorText = await response.text();
        console.error("Seller Dashboard - API Error:", errorText);
        throw new Error(`HTTP ${response.status}: ${errorText}`);
      }

      const allLeads = await response.json();
      console.log("Seller Dashboard - All leads:", allLeads);
      
      // Filter leads client-side based on lead_type
      const filteredLeads = allLeads.filter(lead => {
        const leadType = lead.lead_type || 'sale'; // Default to sale for existing leads
        console.log(`Lead ${lead.id}: lead_type='${lead.lead_type}', treating as '${leadType}', activeEnquiryType='${activeEnquiryType}'`);
        
        if (activeEnquiryType === "rental") {
          const isRental = leadType === 'rental' || leadType === 'RENTAL';
          console.log(`Lead ${lead.id} rental check: ${isRental}`);
          return isRental;
        } else {
          // Treat 'interested', null, empty, or 'sale' as sale leads
          const isSale = leadType === 'sale' || leadType === 'SALE' || leadType === 'interested' || leadType === null || leadType === '' || !leadType;
          console.log(`Lead ${lead.id} sale check: ${isSale}`);
          return isSale;
        }
      });
      
      console.log(
        "Seller Dashboard - Filtered leads:",
        filteredLeads.length,
        "leads found for", activeEnquiryType
      );
      
      if (activeEnquiryType === "rental") {
        setRentalLeads(filteredLeads);
      } else {
        setSaleLeads(filteredLeads);
      }
    } catch (error) {
      console.error("Seller Dashboard - Error fetching leads:", error);
      if (activeEnquiryType === "rental") {
        setRentalLeads([]);
      } else {
        setSaleLeads([]);
      }
    } finally {
      setLoading(false);
    }
  };

  const getTimeLabel = (dateString: string) => {
    const now = new Date();
    const date = new Date(dateString);
    const diffMs = now.getTime() - date.getTime();
    const diffMins = Math.floor(diffMs / (1000 * 60));
    const diffHours = Math.floor(diffMs / (1000 * 60 * 60));
    const diffDays = Math.floor(diffMs / (1000 * 60 * 60 * 24));

    if (diffMins < 1) return "Just now";
    if (diffMins < 60) return `${diffMins} mins ago`;
    if (diffHours < 24) return `${diffHours} hours ago`;
    if (diffDays === 1) return "Yesterday";
    if (diffDays < 7) return `${diffDays} days ago`;
    return date.toLocaleDateString();
  };

  const isHotLead = (lead: Lead) => {
    return (
      lead.ai_category?.toLowerCase() === "hot" ||
      (lead.score && lead.score >= 75) ||
      lead.budget?.toLowerCase().includes("crore")
    );
  };

  const currentLeads = activeEnquiryType === "rental" ? rentalLeads : saleLeads;
  console.log(`Current leads for ${activeEnquiryType}:`, currentLeads.length, currentLeads);

  const filteredLeads = currentLeads.filter((lead) => {
    console.log(`Filtering lead ${lead.id}: status=${lead.status}, activeTab=${activeTab}`);
    
    // Tab filter - different statuses for rental vs sale
    const validStatuses = activeEnquiryType === "rental" 
      ? ["NEW", "CONTACTED", "CONVERTED", "ARCHIVED"]
      : ["NEW", "CONTACTED", "PROSPECT", "ARCHIVED"];
    
    // Convert status to uppercase for comparison
    const leadStatus = (lead.status || '').toUpperCase();
    
    if (activeTab === "new" && leadStatus !== "NEW") {
      console.log(`Lead ${lead.id} filtered out: not NEW (${leadStatus})`);
      return false;
    }
    if (activeTab === "contacted" && leadStatus !== "CONTACTED") {
      console.log(`Lead ${lead.id} filtered out: not CONTACTED (${leadStatus})`);
      return false;
    }
    if (activeTab === "prospects" && leadStatus !== "PROSPECT") {
      console.log(`Lead ${lead.id} filtered out: not PROSPECT (${leadStatus})`);
      return false;
    }
    if (activeTab === "converted" && leadStatus !== "CONVERTED") {
      console.log(`Lead ${lead.id} filtered out: not CONVERTED (${leadStatus})`);
      return false;
    }
    if (activeTab === "archived" && leadStatus !== "ARCHIVED") {
      console.log(`Lead ${lead.id} filtered out: not ARCHIVED (${leadStatus})`);
      return false;
    }

    // Search filter
    const leadName = lead.buyer_name || lead.name || "";
    const leadMessage = lead.message || "";
    if (
      searchTerm &&
      !leadName.toLowerCase().includes(searchTerm.toLowerCase()) &&
      !leadMessage.toLowerCase().includes(searchTerm.toLowerCase())
    ) {
      console.log(`Lead ${lead.id} filtered out: search term`);
      return false;
    }

    // Type filter
    if (filterType === "hot" && !isHotLead(lead)) {
      console.log(`Lead ${lead.id} filtered out: not hot lead`);
      return false;
    }

    console.log(`Lead ${lead.id} passed all filters`);
    return true;
  });

  const sortedLeads = [...filteredLeads].sort((a, b) => {
    const dateA = new Date(a.created_at).getTime();
    const dateB = new Date(b.created_at).getTime();
    return sortOrder === "recent" ? dateB - dateA : dateA - dateB;
  });

  const totalPages = Math.ceil(sortedLeads.length / leadsPerPage);
  const startIndex = (currentPage - 1) * leadsPerPage;
  const paginatedLeads = sortedLeads.slice(
    startIndex,
    startIndex + leadsPerPage
  );

  const getTabs = () => {
    if (activeEnquiryType === "rental") {
      return [
        { id: "all", label: "All", count: rentalLeads.length },
        { id: "new", label: "New", count: rentalLeads.filter((l) => (l.status || '').toUpperCase() === "NEW").length },
        { id: "contacted", label: "Contacted", count: rentalLeads.filter((l) => (l.status || '').toUpperCase() === "CONTACTED").length },
        { id: "converted", label: "Converted", count: rentalLeads.filter((l) => (l.status || '').toUpperCase() === "CONVERTED").length },
        { id: "archived", label: "Archived", count: rentalLeads.filter((l) => (l.status || '').toUpperCase() === "ARCHIVED").length },
      ];
    } else {
      return [
        { id: "all", label: "All", count: saleLeads.length },
        { id: "new", label: "New", count: saleLeads.filter((l) => (l.status || '').toUpperCase() === "NEW").length },
        { id: "contacted", label: "Contacted", count: saleLeads.filter((l) => (l.status || '').toUpperCase() === "CONTACTED").length },
        { id: "prospects", label: "Prospects", count: saleLeads.filter((l) => (l.status || '').toUpperCase() === "PROSPECT").length },
        { id: "archived", label: "Archived", count: saleLeads.filter((l) => (l.status || '').toUpperCase() === "ARCHIVED").length },
      ];
    }
  };

  const tabs = getTabs();

  const convertRentalEnquiry = async (leadId: string) => {
    try {
      const API_BASE = import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
      
      // Get lead details first
      const lead = rentalLeads.find(l => l.id === leadId);
      if (!lead) {
        alert("Lead not found");
        return;
      }
      
      console.log('Converting lead to tenancy:', lead);
      
      // Create tenancy record
      const response = await fetch(
        `${API_BASE}/property-management/tenancies`,
        {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify({
            property_id: lead.property_id,
            tenant_name: lead.buyer_name || lead.name || 'Unknown Tenant',
            tenant_email: lead.buyer_email || lead.email,
            tenant_phone: lead.buyer_phone || lead.phone,
            tenancy_type: "RENT",
            start_date: new Date().toISOString().split('T')[0],
            status: "ACTIVE"
          })
        }
      );

      if (response.ok) {
        const result = await response.json();
        console.log('Tenancy created:', result);
        // Update lead status to CONVERTED
        await updateLeadStatus(leadId, "CONVERTED");
        alert(`Rental enquiry converted to active tenancy successfully! Tenancy ID: ${result.tenancy_id}`);
      } else {
        const errorText = await response.text();
        console.error('Tenancy creation failed:', errorText);
        alert(`Failed to convert enquiry: ${errorText}`);
      }
    } catch (error) {
      console.error("Error converting rental enquiry:", error);
      alert("Failed to convert enquiry. Please try again.");
    }
  };

  const updateLeadStatus = async (leadId: string, status: string) => {
    try {
      console.log(`Updating lead ${leadId} to status ${status}`);
      const API_BASE = import.meta.env.VITE_API_BASE || "http://localhost:8090/api";
      
      let userStr = localStorage.getItem("seller") || localStorage.getItem("user");
      let userId = "e7aec4cf-8249-4a5c-ad71-b95011819f80";
      
      if (userStr) {
        try {
          const user = JSON.parse(userStr);
          userId = user.id || user.user_id || userId;
        } catch (e) {
          console.error("Error parsing user:", e);
        }
      }

      const apiUrl = `${API_BASE}/leads-management/leads/${leadId}/status?status=${encodeURIComponent(status.toUpperCase())}&user_id=${encodeURIComponent(userId)}`;
      console.log(`Making API call to: ${apiUrl}`);
      
      const response = await fetch(apiUrl, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        }
      });

      console.log(`Status update response: ${response.status}`);
      
      if (response.ok) {
        const result = await response.json();
        console.log(`Successfully updated lead ${leadId} to ${status}:`, result);
        // Update local state
        if (activeEnquiryType === "rental") {
          setRentalLeads((prevLeads) =>
            prevLeads.map((lead) =>
              lead.id === leadId ? { ...lead, status: status.toUpperCase() } : lead
            )
          );
        } else {
          setSaleLeads((prevLeads) =>
            prevLeads.map((lead) =>
              lead.id === leadId ? { ...lead, status: status.toUpperCase() } : lead
            )
          );
        }
        alert(`Lead status updated to ${status}`);
      } else {
        const responseText = await response.text();
        console.error(`API Error Response (${response.status}):`, responseText);
        
        let errorMessage = responseText || `Server error: ${response.status}`;
        
        if (responseText && response.headers.get('content-type')?.includes('application/json')) {
          try {
            const errorData = JSON.parse(responseText);
            if (errorData?.detail) {
              errorMessage = errorData.detail;
            } else if (errorData?.message) {
              errorMessage = errorData.message;
            } else if (errorData?.error) {
              errorMessage = errorData.error;
            }
          } catch (parseError) {
            console.error('Failed to parse error response as JSON:', parseError);
          }
        }
        
        console.error(`Failed to update lead status: ${response.status} - ${errorMessage}`);
        alert(`Failed to update status: ${errorMessage}`);
      }
    } catch (error) {
      console.error("Error updating lead status:", error);
      alert(`Error updating status: ${error.message}`);
    }
  };

  const handleCall = async (lead: Lead) => {
    if ((lead.status || '').toUpperCase() === "ARCHIVED") {
      alert("Cannot contact archived leads");
      return;
    }
    
    // Don't change status if already converted
    if ((lead.status || '').toUpperCase() === "CONVERTED") {
      const phone = lead.buyer_phone || lead.phone;
      if (phone) {
        window.open(`tel:${phone}`, "_self");
      }
      return;
    }

    const phone = lead.buyer_phone || lead.phone;
    if (phone) {
      window.open(`tel:${phone}`, "_self");
      // Only update to CONTACTED if not already converted
      updateLeadStatus(lead.id, "CONTACTED");
    }
  };

  const handleWhatsApp = async (lead: Lead) => {
    if ((lead.status || '').toUpperCase() === "ARCHIVED") {
      alert("Cannot contact archived leads");
      return;
    }
    
    // Don't change status if already converted
    if ((lead.status || '').toUpperCase() === "CONVERTED") {
      const phone = lead.buyer_phone || lead.phone;
      if (phone) {
        const cleanPhone = phone.replace(/[^+\d]/g, "");
        window.open(`https://wa.me/${cleanPhone}`, "_blank");
      }
      return;
    }

    const phone = lead.buyer_phone || lead.phone;
    if (phone) {
      const cleanPhone = phone.replace(/[^+\d]/g, "");
      window.open(`https://wa.me/${cleanPhone}`, "_blank");
      // Only update to CONTACTED if not already converted
      updateLeadStatus(lead.id, "CONTACTED");
    }
  };

  const handleArchive = (lead: Lead) => {
    updateLeadStatus(lead.id, "ARCHIVED");
  };

  const renderEnquiriesContent = () => {
    return (
      <>
        {/* Search and Filters */}
        <div className="flex flex-col sm:flex-row gap-4 items-start sm:items-center justify-between">
          <div className="relative flex-1 max-w-md">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
            <Input
              type="text"
              placeholder="Search Enquiries..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-10"
            />
          </div>

          <div className="flex gap-4 items-center">
            <div className="flex items-center gap-2">
              <span className="text-sm text-gray-600">Sort:</span>
              <div className="relative">
                <Select value={sortOrder} onValueChange={setSortOrder}>
                  <SelectTrigger className="w-24">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="recent">Recent</SelectItem>
                    <SelectItem value="oldest">Oldest</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>
          </div>
        </div>

        {/* Enquiries List */}
        {loading ? (
          <div className="flex justify-center py-12">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
          </div>
        ) : paginatedLeads.length === 0 ? (
          <div className="text-center py-12">
            <p className="text-gray-500 text-lg">No {activeEnquiryType} enquiries found</p>
            <p className="text-gray-400 text-sm mt-2">
              {searchTerm
                ? "Try adjusting your search terms"
                : `New ${activeEnquiryType} enquiries will appear here`}
            </p>
          </div>
        ) : (
          <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-6">
            {paginatedLeads.map((lead) => (
              <Card
                key={lead.id}
                className="hover:shadow-lg transition-shadow bg-blue-50 border-blue-100"
              >
                <CardHeader className="pb-3">
                  <div className="flex items-start justify-between">
                    <div className="flex items-center gap-3">
                      <div className="w-12 h-12 bg-gray-200 rounded-full flex items-center justify-center">
                        <span className="text-lg font-medium text-gray-600">
                          {(lead.buyer_name || lead.name || "No Name")
                            .charAt(0)
                            .toUpperCase()}
                        </span>
                      </div>
                      <div>
                        <h3 className="text-lg font-medium text-gray-900">
                          {lead.buyer_name || lead.name || "No Name"}
                        </h3>
                        <p className="text-sm text-gray-500">
                          {getTimeLabel(lead.created_at)}
                        </p>
                      </div>
                    </div>
                    {isHotLead(lead) && (
                      <Badge className="bg-red-100 text-red-800 hover:bg-red-100">
                        Hot Lead
                      </Badge>
                    )}
                  </div>
                </CardHeader>

                <CardContent className="space-y-4">
                  {/* Message */}
                  <div>
                    <p className="text-gray-600 text-sm line-clamp-3">
                      {lead.message || "No message provided"}
                    </p>
                  </div>

                  {/* Contact Info */}
                  <div className="space-y-2">
                    <div className="flex items-center justify-between text-sm">
                      <span className="font-medium text-gray-700">Phone:</span>
                      <span className="text-gray-600">
                        {lead.buyer_phone || lead.phone || "Not provided"}
                      </span>
                    </div>
                    <div className="flex items-center justify-between text-sm">
                      <span className="font-medium text-gray-700">Email:</span>
                      <span className="text-gray-600 truncate ml-2">
                        {lead.buyer_email || lead.email || "Not provided"}
                      </span>
                    </div>
                  </div>

                  {/* Property & Budget */}
                  {(lead.property_title || lead.budget) && (
                    <div className="space-y-2">
                      {lead.property_title && (
                        <div className="flex items-center justify-between text-sm">
                          <span className="font-medium text-gray-700">
                            Property:
                          </span>
                          <span className="text-gray-600 truncate ml-2">
                            {lead.property_title}
                          </span>
                        </div>
                      )}
                      {lead.budget && (
                        <div className="flex items-center justify-between text-sm">
                          <span className="font-medium text-gray-700">
                            {activeEnquiryType === "rental" ? "Duration:" : "Budget:"}
                          </span>
                          <span className="text-gray-600">{lead.budget}</span>
                        </div>
                      )}
                    </div>
                  )}

                  {/* Status & Score */}
                  <div className="flex items-center justify-between">
                    <Badge
                      className={`${
                        lead.status === "NEW"
                          ? "bg-green-100 text-green-800 hover:bg-green-100"
                          : lead.status === "CONTACTED"
                          ? "bg-blue-100 text-blue-800 hover:bg-blue-100"
                          : lead.status === "PROSPECT" || lead.status === "CONVERTED"
                          ? "bg-yellow-100 text-yellow-800 hover:bg-yellow-100"
                          : "bg-gray-100 text-gray-800 hover:bg-gray-100"
                      }`}
                    >
                      {lead.status}
                    </Badge>
                    {lead.score && (
                      <span className="text-sm text-gray-600">
                        Score: {lead.score}/100
                      </span>
                    )}
                  </div>

                  {/* Action Buttons */}
                  <div className="flex flex-wrap gap-2 pt-2">
                    {(lead.buyer_phone || lead.phone) && (
                      <>
                        <Button
                          onClick={() => handleWhatsApp(lead)}
                          size="sm"
                          className="flex-1 bg-green-500 hover:bg-green-600 text-white"
                        >
                          <svg className="w-4 h-4 mr-1" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893A11.821 11.821 0 0020.465 3.488"/>
                          </svg>
                          WhatsApp
                        </Button>
                        <Button
                          onClick={() => handleCall(lead)}
                          size="sm"
                          className="flex-1 bg-blue-500 hover:bg-blue-600 text-white"
                        >
                          <Phone className="w-4 h-4 mr-1" />
                          Call
                        </Button>
                      </>
                    )}
                  </div>

                  <div className="flex gap-2">
                    {/* Status update buttons based on enquiry type */}
                    {activeEnquiryType === "rental" ? (
                      // Rental: NEW → CONTACTED → CONVERTED → ARCHIVED
                      <>
                        {(lead.status || '').toUpperCase() === "NEW" && (
                          <Button
                            onClick={() => updateLeadStatus(lead.id, "CONTACTED")}
                            variant="outline"
                            size="sm"
                            className="flex-1 bg-blue-50 hover:bg-blue-100 text-blue-700 border-blue-200"
                          >
                            Mark Contacted
                          </Button>
                        )}
                        {(lead.status || '').toUpperCase() === "CONTACTED" && (
                          <Button
                            onClick={() => convertRentalEnquiry(lead.id)}
                            variant="outline"
                            size="sm"
                            className="flex-1 bg-green-50 hover:bg-green-100 text-green-700 border-green-200"
                          >
                            Convert to Tenancy
                          </Button>
                        )}
                      </>
                    ) : (
                      // Sale: NEW → CONTACTED → PROSPECT → ARCHIVED
                      <>
                        {(lead.status || '').toUpperCase() === "NEW" && (
                          <Button
                            onClick={() => updateLeadStatus(lead.id, "CONTACTED")}
                            variant="outline"
                            size="sm"
                            className="flex-1 bg-blue-50 hover:bg-blue-100 text-blue-700 border-blue-200"
                          >
                            Mark Contacted
                          </Button>
                        )}
                        {(lead.status || '').toUpperCase() === "CONTACTED" && (
                          <Button
                            onClick={() => updateLeadStatus(lead.id, "PROSPECT")}
                            variant="outline"
                            size="sm"
                            className="flex-1 bg-yellow-50 hover:bg-yellow-100 text-yellow-700 border-yellow-200"
                          >
                            Mark Prospect
                          </Button>
                        )}
                      </>
                    )}

                    <Button
                      onClick={() => updateLeadStatus(lead.id, "ARCHIVED")}
                      variant="outline"
                      size="sm"
                      className="bg-gray-50 hover:bg-gray-100"
                    >
                      <Archive className="w-4 h-4" />
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        )}

        {/* Pagination */}
        {totalPages > 1 && (
          <div className="flex items-center justify-between mt-6 px-4 py-4 bg-white rounded-lg shadow border border-gray-200">
            <div className="text-sm text-gray-700">
              Showing {startIndex + 1} to{" "}
              {Math.min(startIndex + leadsPerPage, sortedLeads.length)} of{" "}
              {sortedLeads.length} results
            </div>

            <div className="flex items-center gap-2">
              <Button
                onClick={() => setCurrentPage((prev) => Math.max(prev - 1, 1))}
                disabled={currentPage === 1}
                variant="outline"
                size="sm"
                className="flex items-center gap-1"
              >
                <ChevronLeft className="w-4 h-4" />
                Prev
              </Button>

              <span className="px-3 py-2 text-sm font-medium text-gray-900 bg-blue-50 border border-blue-200 rounded-lg">
                {currentPage}
              </span>

              <Button
                onClick={() =>
                  setCurrentPage((prev) => Math.min(prev + 1, totalPages))
                }
                disabled={currentPage === totalPages}
                variant="outline"
                size="sm"
                className="flex items-center gap-1"
              >
                Next
                <ChevronRight className="w-4 h-4" />
              </Button>
            </div>
          </div>
        )}
      </>
    );
  };

  return (
    <div className="space-y-6 p-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold text-gray-900">Leads & Enquiries</h1>
      </div>

      {/* Main Tabs for Rental vs Sale */}
      <Tabs value={activeEnquiryType} onValueChange={setActiveEnquiryType} className="w-full">
        <TabsList className="grid w-full grid-cols-2">
          <TabsTrigger value="rental">Rental Enquiries</TabsTrigger>
          <TabsTrigger value="sale">Sale Enquiries</TabsTrigger>
        </TabsList>

        <TabsContent value="rental" className="space-y-6">
          {/* Status Tabs for Rental */}
          <div className="border-b border-gray-200">
            <div className="flex gap-6">
              {tabs.map((tab) => (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id)}
                  className={`pb-3 px-1 font-medium text-sm border-b-2 transition-colors ${
                    activeTab === tab.id
                      ? "border-blue-600 text-blue-600"
                      : "border-transparent text-gray-600 hover:text-gray-900"
                  }`}
                >
                  {tab.label}
                  {tab.count > 0 && (
                    <span
                      className={`ml-2 px-2 py-1 text-xs rounded-full ${
                        activeTab === tab.id
                          ? "bg-blue-100 text-blue-600"
                          : "bg-gray-100 text-gray-600"
                      }`}
                    >
                      {tab.count}
                    </span>
                  )}
                </button>
              ))}
            </div>
          </div>
          {renderEnquiriesContent()}
        </TabsContent>

        <TabsContent value="sale" className="space-y-6">
          {/* Status Tabs for Sale */}
          <div className="border-b border-gray-200">
            <div className="flex gap-6">
              {tabs.map((tab) => (
                <button
                  key={tab.id}
                  onClick={() => setActiveTab(tab.id)}
                  className={`pb-3 px-1 font-medium text-sm border-b-2 transition-colors ${
                    activeTab === tab.id
                      ? "border-blue-600 text-blue-600"
                      : "border-transparent text-gray-600 hover:text-gray-900"
                  }`}
                >
                  {tab.label}
                  {tab.count > 0 && (
                    <span
                      className={`ml-2 px-2 py-1 text-xs rounded-full ${
                        activeTab === tab.id
                          ? "bg-blue-100 text-blue-600"
                          : "bg-gray-100 text-gray-600"
                      }`}
                    >
                      {tab.count}
                    </span>
                  )}
                </button>
              ))}
            </div>
          </div>
          {renderEnquiriesContent()}
        </TabsContent>
      </Tabs>
    </div>
  );
}