import { useEffect, useState } from 'react';
import { useRoute, useLocation } from 'wouter';
import { negotiationApi } from '../../../../api/negotiation.api';
import AudioCall from '../../../negotiation/AudioCall';

export default function NegotiationDetail() {
  const [, params] = useRoute('/negotiation/:id');
  const [, setLocation] = useLocation();
  const [timeline, setTimeline] = useState<any>(null);
  const [loading, setLoading] = useState(true);
  const [processing, setProcessing] = useState(false);

  useEffect(() => {
    if (params?.id) {
      loadNegotiation(params.id);
    }
  }, [params?.id]);

  const loadNegotiation = async (id: string) => {
    try {
      const data = await negotiationApi.getTimeline(id);
      setTimeline(data);
    } catch (error) {
      console.error('Failed to load negotiation:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleAccept = async (eventId: string) => {
    if (!confirm('Accept this offer?')) return;
    
    setProcessing(true);
    try {
      await negotiationApi.acceptOffer(eventId);
      await loadNegotiation(params!.id);
      alert('Offer accepted! Escrow process initiated.');
    } catch (error: any) {
      alert(error.response?.data?.detail || 'Failed to accept offer');
    } finally {
      setProcessing(false);
    }
  };

  const handleReject = async (eventId: string) => {
    const reason = prompt('Reason for rejection (optional):');
    
    setProcessing(true);
    try {
      await negotiationApi.rejectOffer(eventId, reason ? { reason } : undefined);
      await loadNegotiation(params!.id);
      alert('Offer rejected');
    } catch (error: any) {
      alert(error.response?.data?.detail || 'Failed to reject offer');
    } finally {
      setProcessing(false);
    }
  };

  const handleCounter = async () => {
    const amount = prompt('Enter counter offer amount (e.g., 5000000 for ₹50,00,000):');
    if (!amount) return;

    const numericAmount = parseFloat(amount.replace(/,/g, ''));
    if (isNaN(numericAmount) || numericAmount <= 0) {
      alert('Please enter a valid amount');
      return;
    }

    const message = prompt('Message (optional):');
    
    setProcessing(true);
    try {
      await negotiationApi.submitOffer(params!.id, {
        amount: numericAmount,
        message: message || undefined,
      });
      await loadNegotiation(params!.id);
      alert('Counter offer sent');
    } catch (error: any) {
      alert(error.response?.data?.detail || 'Failed to send counter offer');
    } finally {
      setProcessing(false);
    }
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-1/4"></div>
          <div className="h-64 bg-gray-200 rounded"></div>
        </div>
      </div>
    );
  }

  if (!timeline) {
    return (
      <div className="p-6">
        <div className="bg-red-50 border border-red-200 rounded-lg p-4">
          <p className="text-red-600">Negotiation not found</p>
        </div>
      </div>
    );
  }

  const activeOffer = timeline.events.find((e: any) => e.is_active && e.amount);
  
  console.log('Active Offer:', activeOffer);
  console.log('Timeline Status:', timeline.status);
  console.log('Offered By:', activeOffer?.offered_by);
  
  const canRespond = activeOffer && activeOffer.offered_by === 'BUYER' && 
                     !['ACCEPTED', 'REJECTED', 'EXPIRED'].includes(timeline.status);
  
  console.log('Can Respond:', canRespond);

  // Get current user ID for call functionality
  const getCurrentUserId = () => {
    try {
      const token = localStorage.getItem('sellerToken') || sessionStorage.getItem('sellerToken');
      if (token) {
        const payload = JSON.parse(atob(token.split('.')[1]));
        return payload.user_id;
      }
    } catch (error) {
      console.error('Failed to get user ID from token:', error);
    }
    return null;
  };

  const currentUserId = getCurrentUserId();
  
  // Check if calls are allowed (active negotiation)
  const canMakeCall = () => {
    return !['ACCEPTED', 'REJECTED', 'EXPIRED'].includes(timeline.status);
  };

  return (
    <div className="p-6 max-w-4xl mx-auto">
      <button
        onClick={() => setLocation('/negotiations')}
        className="mb-4 text-blue-600 hover:text-blue-700 flex items-center gap-2"
      >
        ← Back to Negotiations
      </button>

      <div className="bg-white rounded-lg border p-6 mb-6">
        <div className="flex justify-between items-start mb-4">
          <div>
            <h1 className="text-2xl font-bold text-gray-900">Negotiation Details</h1>
            <p className="text-sm text-gray-600 mt-1">ID: {timeline.negotiation_id}</p>
          </div>
          <span className={`px-3 py-1 rounded-full text-sm font-medium ${
            timeline.status === 'ACCEPTED' ? 'bg-green-100 text-green-800' :
            timeline.status === 'REJECTED' ? 'bg-red-100 text-red-800' :
            timeline.status === 'OFFER_SENT' ? 'bg-yellow-100 text-yellow-800' :
            'bg-blue-100 text-blue-800'
          }`}>
            {timeline.status.replace('_', ' ')}
          </span>
        </div>

        {timeline.ai_insight && (
          <div className="bg-blue-50 rounded-lg p-4 mb-4">
            <h3 className="font-semibold text-blue-900 mb-2">Market Analysis</h3>
            <div className="grid grid-cols-3 gap-4 text-sm">
              <div>
                <p className="text-blue-700">Market Price</p>
                <p className="font-bold text-blue-900">₹{timeline.ai_insight.market_price.toLocaleString('en-IN')}</p>
              </div>
              <div>
                <p className="text-blue-700">Fair Range</p>
                <p className="font-bold text-blue-900">
                  ₹{timeline.ai_insight.suggested_range.low.toLocaleString('en-IN')} - 
                  ₹{timeline.ai_insight.suggested_range.high.toLocaleString('en-IN')}
                </p>
              </div>
              <div>
                <p className="text-blue-700">Classification</p>
                <p className="font-bold text-blue-900">{timeline.ai_insight.classification}</p>
              </div>
            </div>
          </div>
        )}
      </div>

      {activeOffer && (
        <div className="bg-white rounded-lg border p-6 mb-6">
          <h2 className="text-xl font-semibold text-gray-900 mb-4">Current Offer</h2>
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
            <div className="flex justify-between items-start mb-3">
              <div>
                <p className="text-3xl font-bold text-gray-900">₹{activeOffer.amount.toLocaleString('en-IN')}</p>
                <p className="text-sm text-gray-600 mt-1">From: {activeOffer.offered_by}</p>
              </div>
              {activeOffer.expires_at && (
                <p className="text-sm text-orange-600">
                  Expires: {new Date(activeOffer.expires_at).toLocaleString('en-IN')}
                </p>
              )}
            </div>
            {activeOffer.message && (
              <div className="bg-white rounded p-3 mb-3">
                <p className="text-sm text-gray-700">"{activeOffer.message}"</p>
              </div>
            )}
            
            {activeOffer && (
              <div className="flex gap-3 mt-4">
                <button
                  onClick={() => handleAccept(activeOffer.id)}
                  disabled={processing || !canRespond}
                  style={{ backgroundColor: '#2563eb', color: '#ffffff', padding: '8px 16px', borderRadius: '6px', fontWeight: '600' }}
                  className="flex-1 shadow-md disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {processing ? 'Processing...' : 'Accept Offer'}
                </button>
                <button
                  onClick={() => handleCounter()}
                  disabled={processing || !canRespond}
                  style={{ backgroundColor: '#2563eb', color: '#ffffff', padding: '8px 16px', borderRadius: '6px', fontWeight: '600' }}
                  className="flex-1 shadow-md disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  Counter Offer
                </button>
                <button
                  onClick={() => handleReject(activeOffer.id)}
                  disabled={processing || !canRespond}
                  style={{ backgroundColor: '#2563eb', color: '#ffffff', padding: '8px 16px', borderRadius: '6px', fontWeight: '600' }}
                  className="flex-1 shadow-md disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  Reject
                </button>
              </div>
            )}
          </div>
        </div>
      )}

      {/* Audio Call Component */}
      {currentUserId && (
        <div className="bg-white rounded-lg border p-6 mb-6">
          <AudioCall
            negotiationId={timeline.negotiation_id}
            userId={currentUserId}
            canMakeCall={canMakeCall()}
            disabled={processing}
          />
        </div>
      )}

      <div className="bg-white rounded-lg border p-6">
        <h2 className="text-xl font-semibold text-gray-900 mb-4">Timeline</h2>
        <div className="space-y-4">
          {timeline.events.map((event: any) => (
            <div key={event.id} className={`border-l-4 pl-4 py-2 ${
              event.is_active ? 'border-blue-500 bg-blue-50' : 'border-gray-300'
            }`}>
              <div className="flex justify-between items-start">
                <div>
                  <p className="font-semibold text-gray-900">{event.event_type.replace('_', ' ')}</p>
                  <p className="text-sm text-gray-600">by {event.offered_by}</p>
                  {event.amount && (
                    <p className="text-lg font-bold text-gray-900 mt-1">₹{event.amount.toLocaleString('en-IN')}</p>
                  )}
                  {event.message && (
                    <p className="text-sm text-gray-700 mt-2 bg-white rounded p-2">"{event.message}"</p>
                  )}
                </div>
                <p className="text-xs text-gray-500">{new Date(event.created_at).toLocaleString('en-IN')}</p>
              </div>
            </div>
          ))}
        </div>
      </div>
    </div>
  );
}
