import { useEffect, useState } from 'react';
import { useLocation } from 'wouter';
import { negotiationApi } from '../../../../api/negotiation.api';

export default function Negotiations() {
  const [, setLocation] = useLocation();
  const [negotiations, setNegotiations] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadNegotiations();
  }, []);

  const loadNegotiations = async () => {
    try {
      const data = await negotiationApi.listNegotiations();
      setNegotiations(data);
    } catch (error) {
      console.error('Failed to load negotiations:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="p-6">
        <div className="animate-pulse space-y-4">
          <div className="h-8 bg-gray-200 rounded w-1/4"></div>
          <div className="h-32 bg-gray-200 rounded"></div>
          <div className="h-32 bg-gray-200 rounded"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="p-6">
      <div className="mb-6">
        <h1 className="text-2xl font-bold text-gray-900">Negotiations</h1>
        <p className="text-gray-600 mt-1">Manage all property negotiations</p>
      </div>

      {negotiations.length === 0 ? (
        <div className="bg-white rounded-lg border p-12 text-center">
          <svg className="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z" />
          </svg>
          <h3 className="text-lg font-medium text-gray-900 mb-2">No Negotiations Yet</h3>
          <p className="text-gray-600">Negotiations will appear here when buyers make offers on your properties</p>
        </div>
      ) : (
        <div className="space-y-4">
          {negotiations.map((neg) => (
            <div key={neg.negotiation_id} className="bg-white rounded-lg border p-6 hover:shadow-md transition-shadow">
              <div className="flex justify-between items-start mb-4">
                <div>
                  <h3 className="text-lg font-semibold text-gray-900">{neg.property_title}</h3>
                  <p className="text-sm text-gray-600 mt-1">Property Price: ₹{neg.property_price?.toLocaleString('en-IN')}</p>
                </div>
                <span className={`px-3 py-1 rounded-full text-sm font-medium ${
                  neg.negotiation_status === 'ACCEPTED' ? 'bg-green-100 text-green-800' :
                  neg.negotiation_status === 'REJECTED' ? 'bg-red-100 text-red-800' :
                  neg.negotiation_status === 'OFFER_SENT' ? 'bg-yellow-100 text-yellow-800' :
                  'bg-blue-100 text-blue-800'
                }`}>
                  {neg.negotiation_status.replace('_', ' ')}
                </span>
              </div>

              <div className="flex items-center justify-between">
                <div className="text-sm text-gray-600">
                  <span className="font-medium">Your Role:</span> {neg.user_role}
                </div>
                <button
                  onClick={() => setLocation(`/negotiation/${neg.negotiation_id}`)}
                  style={{ backgroundColor: '#2563eb', color: '#ffffff', padding: '8px 16px', borderRadius: '6px', fontWeight: '600' }}
                  className="shadow-md hover:opacity-90"
                >
                  View Details
                </button>
              </div>
            </div>
          ))}
        </div>
      )}
    </div>
  );
}
