import React, { useState, useEffect } from 'react';
import { Clock, CheckCircle, AlertTriangle, FileText, MessageSquare, Upload, Eye, Calendar, X, Send, Phone, Mail, MapPin, User, Building, DollarSign, Shield, Gavel, CreditCard } from 'lucide-react';
import { toast } from 'sonner';

interface NRIRequest {
  id: string;
  property_id: string;
  property_title: string;
  property_address: string;
  buyer_name: string;
  buyer_email: string;
  buyer_phone: string;
  assistance_category: string;
  description: string;
  priority_level: 'high' | 'medium' | 'low';
  status: 'pending_acknowledgment' | 'acknowledged' | 'documents_pending' | 'in_progress' | 'completed' | 'declined';
  created_at: string;
  approved_at: string;
}

interface ResponseTemplate {
  category: string;
  icon: any;
  color: string;
  templates: {
    title: string;
    content: string;
  }[];
}

export default function NRIAssistance() {
  const [requests, setRequests] = useState<NRIRequest[]>([]);
  const [selectedRequest, setSelectedRequest] = useState<NRIRequest | null>(null);
  const [loading, setLoading] = useState(true);
  const [showDetailsModal, setShowDetailsModal] = useState(false);
  const [showMessageModal, setShowMessageModal] = useState(false);
  const [showUploadModal, setShowUploadModal] = useState(false);
  const [showResponseModal, setShowResponseModal] = useState(false);
  const [messageText, setMessageText] = useState('');
  const [selectedTemplate, setSelectedTemplate] = useState('');

  const responseTemplates: ResponseTemplate[] = [
    {
      category: 'Property',
      icon: Building,
      color: 'blue',
      templates: [
        {
          title: 'Market Analysis',
          content: 'I can provide you with a comprehensive market analysis for this property, including recent comparable sales, price trends, and investment potential in this area.'
        },
        {
          title: 'Property Documentation',
          content: 'I will share all relevant property documents including title deed, approved plans, NOC certificates, and property tax receipts for your review.'
        }
      ]
    },
    {
      category: 'Legal',
      icon: Gavel,
      color: 'purple',
      templates: [
        {
          title: 'Legal Compliance',
          content: 'I can guide you through the legal requirements for NRI property purchase, including FEMA compliance, required documentation, and legal procedures.'
        },
        {
          title: 'Documentation Support',
          content: 'I will help you with Power of Attorney preparation, legal document verification, and connect you with trusted legal advisors.'
        }
      ]
    },
    {
      category: 'Banking',
      icon: CreditCard,
      color: 'green',
      templates: [
        {
          title: 'Loan Assistance',
          content: 'I can connect you with banks offering NRI home loans, help with documentation, and guide you through the loan approval process.'
        },
        {
          title: 'Banking Setup',
          content: 'I will assist you with NRE/NRO account setup, remittance procedures, and tax implications for your property investment.'
        }
      ]
    },
    {
      category: 'Verification',
      icon: Shield,
      color: 'orange',
      templates: [
        {
          title: 'Property Verification',
          content: 'I will conduct thorough verification of property documents, ownership history, legal clearances, and provide you with a detailed verification report.'
        },
        {
          title: 'Due Diligence',
          content: 'I can arrange for professional due diligence including site inspection, document verification, and legal status confirmation.'
        }
      ]
    }
  ];

  useEffect(() => {
    fetchNRIRequests();
  }, []);

  const fetchNRIRequests = async () => {
    try {
      const user = JSON.parse(localStorage.getItem('user') || '{}');
      const accessToken = localStorage.getItem('access_token');
      const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
      
      if (!accessToken) {
        setLoading(false);
        return;
      }

      let userId = user.id || user.user_id || localStorage.getItem('user_id');
      if (!userId) {
        setLoading(false);
        return;
      }

      const response = await fetch(`${API_BASE}/nri-requests/my-properties/?userid=${userId}`, {
        headers: {
          'Authorization': `Bearer ${accessToken}`,
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        const data = await response.json();
        const transformedRequests = data.requests?.map((req: any) => ({
          id: req.id.toString(),
          property_id: req.property_id,
          property_title: req.property_title || 'Property',
          property_address: req.property_address || '',
          buyer_name: req.buyer_name,
          buyer_email: req.buyer_email,
          buyer_phone: req.buyer_phone,
          assistance_category: req.assistance_category,
          description: req.description,
          priority_level: req.priority_level?.toLowerCase() || 'medium',
          status: 'pending_acknowledgment',
          created_at: req.created_at,
          approved_at: req.approved_at
        })) || [];
        setRequests(transformedRequests);
      }
    } catch (error) {
      console.error('Error fetching NRI requests:', error);
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'pending_acknowledgment': return 'bg-red-100 text-red-800 border-red-200';
      case 'acknowledged': return 'bg-blue-100 text-blue-800 border-blue-200';
      case 'documents_pending': return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'in_progress': return 'bg-purple-100 text-purple-800 border-purple-200';
      case 'completed': return 'bg-green-100 text-green-800 border-green-200';
      default: return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'high': return 'bg-red-500';
      case 'medium': return 'bg-yellow-500';
      case 'low': return 'bg-green-500';
      default: return 'bg-gray-500';
    }
  };

  const getTimeRemaining = (deadline: string) => {
    const now = new Date();
    const end = new Date(deadline);
    const diff = end.getTime() - now.getTime();
    const hours = Math.floor(diff / (1000 * 60 * 60));
    return hours > 0 ? `${hours}h remaining` : 'Overdue';
  };

  const handleAcknowledge = (requestId: string, action: 'accept' | 'decline') => {
    setRequests(prev => prev.map(req => 
      req.id === requestId 
        ? { ...req, status: action === 'accept' ? 'acknowledged' : 'completed', current_step: action === 'accept' ? 2 : 5 }
        : req
    ));
    toast.success(`Request ${action}ed successfully`);
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-white rounded-lg shadow-sm border p-6">
        <h1 className="text-2xl font-bold text-gray-900 mb-2">NRI Assistance Requests</h1>
        <p className="text-gray-600">Manage assistance requests from NRI buyers for your properties</p>
      </div>

      {/* Active Requests */}
      <div className="grid gap-6">
        {requests.map((request) => (
          <div key={request.id} className="bg-white rounded-lg shadow-sm border overflow-hidden">
            {/* Request Header */}
            <div className="p-6 border-b bg-gray-50">
              <div className="flex items-center justify-between">
                <div className="flex items-center space-x-4">
                  <div className={`w-3 h-3 rounded-full ${getPriorityColor(request.priority)}`}></div>
                  <div>
                    <h3 className="font-semibold text-gray-900">{request.property_title}</h3>
                    <p className="text-sm text-gray-600">Request from {request.nri_name}</p>
                  </div>
                </div>
                <div className="text-right">
                  <span className={`inline-flex px-3 py-1 text-xs font-medium rounded-full border ${getStatusColor(request.status)}`}>
                    {request.status.replace('_', ' ').toUpperCase()}
                  </span>
                  <p className="text-xs text-gray-500 mt-1">{getTimeRemaining(request.deadline)}</p>
                </div>
              </div>
            </div>

            {/* Progress Bar */}
            <div className="px-6 py-3 bg-gray-50 border-b">
              <div className="flex items-center justify-between mb-2">
                <span className="text-sm font-medium text-gray-700">Progress</span>
                <span className="text-sm text-gray-500">{request.current_step}/{request.total_steps}</span>
              </div>
              <div className="w-full bg-gray-200 rounded-full h-2">
                <div 
                  className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                  style={{ width: `${(request.current_step / request.total_steps) * 100}%` }}
                ></div>
              </div>
            </div>

            {/* Request Content */}
            <div className="p-6">
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <h4 className="font-medium text-gray-900 mb-3">Request Details</h4>
                  <div className="space-y-2 text-sm">
                    <div className="flex justify-between">
                      <span className="text-gray-600">Property ID:</span>
                      <span className="font-medium">{request.property_id}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Assistance Type:</span>
                      <span className="font-medium">{request.assistance_type}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Priority:</span>
                      <span className={`font-medium capitalize ${request.priority === 'high' ? 'text-red-600' : request.priority === 'medium' ? 'text-yellow-600' : 'text-green-600'}`}>
                        {request.priority}
                      </span>
                    </div>
                  </div>
                </div>

                <div>
                  <h4 className="font-medium text-gray-900 mb-3">Next Actions</h4>
                  {request.status === 'pending_acknowledgment' && (
                    <div className="space-y-3">
                      <div className="bg-red-50 border border-red-200 rounded-lg p-3">
                        <div className="flex items-center text-red-800 text-sm">
                          <Clock className="w-4 h-4 mr-2" />
                          Response required within {getTimeRemaining(request.deadline)}
                        </div>
                      </div>
                      <div className="flex space-x-3">
                        <button
                          onClick={() => handleAcknowledge(request.id, 'accept')}
                          className="flex-1 bg-green-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-green-600 transition-colors"
                        >
                          Accept Request
                        </button>
                        <button
                          onClick={() => handleAcknowledge(request.id, 'decline')}
                          className="flex-1 bg-red-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-red-600 transition-colors"
                        >
                          Decline Request
                        </button>
                      </div>
                    </div>
                  )}

                  {request.status === 'acknowledged' && (
                    <div className="space-y-3">
                      <button className="w-full bg-blue-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-blue-600 transition-colors flex items-center justify-center">
                        <Upload className="w-4 h-4 mr-2" />
                        Upload Documents
                      </button>
                      <button className="w-full bg-orange-500 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-orange-600 transition-colors flex items-center justify-center">
                        <MessageSquare className="w-4 h-4 mr-2" />
                        Message NRI Buyer
                      </button>
                    </div>
                  )}

                  {(request.status === 'documents_pending' || request.status === 'in_progress') && (
                    <div className="space-y-3">
                      <button className="w-full bg-purple-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-purple-700 transition-colors flex items-center justify-center">
                        <FileText className="w-4 h-4 mr-2" />
                        Update Status
                      </button>
                      <button className="w-full bg-blue-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-blue-700 transition-colors flex items-center justify-center">
                        <MessageSquare className="w-4 h-4 mr-2" />
                        Send Message
                      </button>
                    </div>
                  )}

                  {request.status === 'completed' && (
                    <div className="bg-green-50 border border-green-200 rounded-lg p-3">
                      <div className="flex items-center text-green-800 text-sm">
                        <CheckCircle className="w-4 h-4 mr-2" />
                        Request completed successfully
                      </div>
                    </div>
                  )}
                </div>
              </div>

              {/* Quick Actions */}
              <div className="mt-6 pt-4 border-t flex items-center justify-between">
                <div className="flex items-center space-x-4 text-sm text-gray-600">
                  <span className="flex items-center">
                    <Calendar className="w-4 h-4 mr-1" />
                    Created {new Date(request.created_at).toLocaleDateString()}
                  </span>
                </div>
                <button 
                  onClick={() => setSelectedRequest(request)}
                  className="text-blue-600 hover:text-blue-700 text-sm font-medium flex items-center"
                >
                  <Eye className="w-4 h-4 mr-1" />
                  View Details
                </button>
              </div>
            </div>
          </div>
        ))}
      </div>

      {requests.length === 0 && (
        <div className="bg-white rounded-lg shadow-sm border p-12 text-center">
          <FileText className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No NRI Assistance Requests</h3>
          <p className="text-gray-600">You don't have any active NRI assistance requests at the moment.</p>
        </div>
      )}

      {/* Upload Documents Modal */}
      {showUploadModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 max-w-md w-full mx-4">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-lg font-semibold">Upload Documents</h3>
              <button onClick={() => setShowUploadModal(false)}>
                <X className="w-5 h-5" />
              </button>
            </div>
            <div className="border-2 border-dashed border-gray-300 rounded-lg p-8 text-center">
              <Upload className="w-12 h-12 text-gray-400 mx-auto mb-4" />
              <p className="text-gray-600 mb-4">Drag and drop files here or click to browse</p>
              <input type="file" multiple className="hidden" id="file-upload" />
              <label htmlFor="file-upload" className="bg-blue-500 text-white px-4 py-2 rounded-lg cursor-pointer hover:bg-blue-600">
                Choose Files
              </label>
            </div>
            <div className="mt-4 flex space-x-3">
              <button 
                onClick={() => { setShowUploadModal(false); toast.success('Documents uploaded successfully'); }}
                className="flex-1 bg-green-500 text-white px-4 py-2 rounded-lg hover:bg-green-600"
              >
                Upload
              </button>
              <button 
                onClick={() => setShowUploadModal(false)}
                className="flex-1 bg-gray-500 text-white px-4 py-2 rounded-lg hover:bg-gray-600"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Message Modal */}
      {showMessageModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 max-w-md w-full mx-4">
            <div className="flex justify-between items-center mb-4">
              <h3 className="text-lg font-semibold">Message NRI Buyer</h3>
              <button onClick={() => setShowMessageModal(false)}>
                <X className="w-5 h-5" />
              </button>
            </div>
            <textarea 
              className="w-full border border-gray-300 rounded-lg p-3 h-32 resize-none"
              placeholder="Type your message here..."
            ></textarea>
            <div className="mt-4 flex space-x-3">
              <button 
                onClick={() => { setShowMessageModal(false); toast.success('Message sent successfully'); }}
                className="flex-1 bg-blue-500 text-white px-4 py-2 rounded-lg hover:bg-blue-600 flex items-center justify-center"
              >
                <Send className="w-4 h-4 mr-2" />
                Send Message
              </button>
              <button 
                onClick={() => setShowMessageModal(false)}
                className="flex-1 bg-gray-500 text-white px-4 py-2 rounded-lg hover:bg-gray-600"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}