import { Card, CardContent, CardHeader, CardTitle } from "../../../ui/card";
import { Button } from "../../../ui/button";
import { Badge } from "../../../ui/badge";
import { Input } from "../../../ui/input";
import { Label } from "../../../ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "../../../ui/select";
import { TrendingUp, TrendingDown, Minus, Target, Zap, Star, AlertCircle, ArrowUp, ArrowDown, ChevronDown, ChevronUp, MapPin, Calculator, BarChart3, Download } from "lucide-react";
import { useState, useMemo, useEffect } from "react";
import axios from 'axios';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || '/api';

interface PriceIntelligenceRequest {
  locality: string;
  bedrooms: number;
  size: number;
  age: number;
  property_type: 'apartment' | 'villa' | 'plot' | 'commercial' | 'pg';
  builder?: string;
  bathrooms?: number;
  floor_number?: number;
  total_floors?: number;
  furnishing?: 'unfurnished' | 'semi_furnished' | 'fully_furnished';
  facing?: string;
  amenities?: string[];
}

interface PriceIntelligenceResponse {
  valuation: {
    estimate: number;
    confidence: string;
  };
  benchmarking: {
    market_min: number;
    market_median: number;
    market_max: number;
    percentile: number;
  };
  locality_trend: {
    direction: 'UP' | 'DOWN' | 'STABLE';
    last_3_months: number[];
  };
  pricing_guidance: {
    fast_sale: number;
    fair_price: number;
    premium: number;
    confidence_level: 'HIGH' | 'MEDIUM' | 'LOW';
  };
}

const analyzePriceIntelligence = async (data: PriceIntelligenceRequest): Promise<PriceIntelligenceResponse> => {
  const response = await axios.post(`${API_BASE_URL}/price-intelligence/analyze`, data, {
    headers: { 'Content-Type': 'application/json' },
    timeout: 30000, // 30 second timeout
    maxContentLength: 50 * 1024 * 1024, // 50MB max response
    maxBodyLength: 50 * 1024 * 1024 // 50MB max request
  });
  return response.data;
};

export default function PriceIntelligence() {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [result, setResult] = useState<PriceIntelligenceResponse | null>(null);
  const [hasAnalyzed, setHasAnalyzed] = useState(false);
  const [showAdvanced, setShowAdvanced] = useState(false);
  
  const [formData, setFormData] = useState<PriceIntelligenceRequest>({
    locality: "",
    bedrooms: 0,
    size: 0,
    age: 0,
    property_type: "apartment",
    builder: "",
    bathrooms: undefined,
    floor_number: undefined,
    total_floors: undefined,
    furnishing: "unfurnished",
    facing: undefined,
    amenities: []
  });

  const availableAmenities = [
    "Parking", "Gym", "Swimming Pool", "Security", "Power Backup",
    "Lift", "Garden", "Club House", "Play Area", "Intercom",
    "Gas Pipeline", "Water Supply", "Maintenance Staff"
  ];

  const toggleAmenity = (amenity: string) => {
    setFormData(prev => ({
      ...prev,
      amenities: prev.amenities?.includes(amenity)
        ? prev.amenities.filter(a => a !== amenity)
        : [...(prev.amenities || []), amenity]
    }));
  };

  // Auto-analyze when form data changes after initial analysis
  useEffect(() => {
    if (hasAnalyzed && formData.locality.trim() && formData.bedrooms > 0 && formData.size > 0) {
      const timeoutId = setTimeout(async () => {
        setLoading(true);
        setError(null);
        try {
          console.log('Auto-analyzing with data:', formData);
          const data = await analyzePriceIntelligence(formData);
          setResult(data);
        } catch (err: any) {
          const errorData = err.response?.data;
          const errorMsg = errorData?.error || errorData?.detail?.error || err.message || "Analysis failed";
          setError(errorMsg);
        } finally {
          setLoading(false);
          // Force garbage collection
          if (window.gc) {
            window.gc();
          }
        }
      }, 500);
      
      return () => clearTimeout(timeoutId);
    }
  }, [formData, hasAnalyzed]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError(null);
    
    try {
      console.log('Manual analyzing with data:', formData);
      const data = await analyzePriceIntelligence(formData);
      setResult(data);
      setHasAnalyzed(true);
    } catch (err: any) {
      console.error('API Error:', err.response?.data);
      const errorData = err.response?.data;
      const errorMsg = errorData?.error || errorData?.detail?.error || err.message || "Analysis failed";
      const errorCode = errorData?.code || errorData?.detail?.code;
      
      if (errorCode === "insufficient_comp_data") {
        setError(`Not enough market data for "${formData.locality}". Try: koramangala, indiranagar, whitefield, or electronic_city`);
      } else {
        setError(errorMsg);
      }
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount: number) => {
    if (amount >= 10000000) return `₹${(amount / 10000000).toFixed(2)} Cr`;
    if (amount >= 100000) return `₹${(amount / 100000).toFixed(2)} L`;
    return `₹${amount.toLocaleString()}`;
  };

  const getTrendIcon = (direction: string) => {
    if (direction === "UP") return <TrendingUp className="w-5 h-5 text-green-600" />;
    if (direction === "DOWN") return <TrendingDown className="w-5 h-5 text-red-600" />;
    return <Minus className="w-5 h-5 text-yellow-600" />;
  };

  const getTrendColor = (direction: string) => {
    if (direction === "UP") return "text-green-600 bg-green-50 border-green-200";
    if (direction === "DOWN") return "text-red-600 bg-red-50 border-red-200";
    return "text-yellow-600 bg-yellow-50 border-yellow-200";
  };

  const getConfidenceColor = (level: string) => {
    if (level === "HIGH") return "bg-green-100 text-green-800 border-green-200";
    if (level === "MEDIUM") return "bg-yellow-100 text-yellow-800 border-yellow-200";
    return "bg-red-100 text-red-800 border-red-200";
  };



  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(1)) + ' ' + sizes[i];
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const generatePDF = async () => {
    if (!result) return;

    // Convert favicon to base64
    const getBase64Image = async (url: string): Promise<string> => {
      try {
        const response = await fetch(url);
        const blob = await response.blob();
        return new Promise((resolve) => {
          const reader = new FileReader();
          reader.onload = () => resolve(reader.result as string);
          reader.readAsDataURL(blob);
        });
      } catch {
        return '';
      }
    };

    const logoBase64 = await getBase64Image('/favicon.png');
    const analysisMode = showAdvanced && (formData.bathrooms || formData.floor_number || formData.furnishing !== 'unfurnished' || formData.facing || (formData.amenities && formData.amenities.length > 0)) ? 'XGBoost Enhanced ML' : 'XGBoost Basic ML';
    const factorsUsed = analysisMode === 'Enhanced' ? 10 : 6;

    const reportData = {
      generatedAt: new Date().toISOString(),
      propertyDetails: {
        locality: formData.locality,
        propertyType: formData.property_type,
        bedrooms: formData.bedrooms,
        size: formData.size,
        age: formData.age,
        builder: formData.builder || 'Not specified',
        bathrooms: formData.bathrooms || 'Not specified',
        floorNumber: formData.floor_number || 'Not specified',
        totalFloors: formData.total_floors || 'Not specified',
        furnishing: formData.furnishing || 'Not specified',
        facing: formData.facing || 'Not specified',
        amenities: formData.amenities || []
      },
      analysisResults: result,
      analysisMode,
      factorsUsed
    };

    const getPropertyTypeDisplay = (type: string) => {
      const typeMap: Record<string, string> = {
        'apartment': '🏢 Apartment',
        'villa': '🏡 Villa',
        'plot': '🏞️ Plot',
        'commercial': '🏢 Commercial',
        'pg': '🏠 PG'
      };
      return typeMap[type] || type;
    };

    const getFurnishingDisplay = (furnishing: string) => {
      const furnishingMap: Record<string, string> = {
        'unfurnished': 'Unfurnished',
        'semi_furnished': 'Semi Furnished',
        'fully_furnished': 'Fully Furnished'
      };
      return furnishingMap[furnishing] || furnishing;
    };

    const getFacingDisplay = (facing: string) => {
      const facingMap: Record<string, string> = {
        'north': '🧭 North',
        'south': '🧭 South',
        'east': '🧭 East',
        'west': '🧭 West',
        'northeast': '🧭 North-East',
        'northwest': '🧭 North-West',
        'southeast': '🧭 South-East',
        'southwest': '🧭 South-West'
      };
      return facingMap[facing] || facing;
    };

    const getMarketRecommendation = () => {
      const confidence = result.valuation.confidence;
      const trend = result.locality_trend.direction;
      const percentile = result.benchmarking.percentile;
      
      if (confidence === 'HIGH' && trend === 'UP' && percentile >= 70) {
        return `Excellent market positioning with high confidence analysis. Your property is valued in the top ${100 - percentile}% of the market with strong upward trends. This is an optimal time for premium pricing strategy. The comprehensive ${analysisMode.toLowerCase()} analysis using ${factorsUsed} factors provides reliable valuation for confident decision-making.`;
      } else if (confidence === 'HIGH' && percentile >= 50) {
        return `Strong market position with reliable valuation confidence. Your property shows solid market standing at the ${percentile}th percentile. The ${analysisMode.toLowerCase()} analysis provides comprehensive insights for strategic pricing decisions. Current market trends suggest ${trend === 'UP' ? 'favorable' : trend === 'DOWN' ? 'cautious' : 'stable'} conditions for property transactions.`;
      } else if (confidence === 'MEDIUM') {
        return `Moderate confidence analysis indicates good market understanding with room for additional data refinement. Your property is positioned at the ${percentile}th percentile in the local market. Consider providing additional property details for enhanced accuracy. Current ${trend.toLowerCase()} market trends should be factored into pricing strategy.`;
      } else {
        return `Initial market assessment provides baseline valuation insights. Your property shows ${percentile}th percentile positioning in the local market. For more accurate valuation, consider providing comprehensive property details including amenities, furnishing status, and location specifics. Market trends indicate ${trend.toLowerCase()} conditions requiring careful pricing consideration.`;
      }
    };

    // Create HTML content for PDF
    const htmlContent = `<!DOCTYPE html>
<html>
<head>
  <meta charset="utf-8">
  <title>NAL Price Intelligence Report</title>
  <style>
    @media print {
      body { margin: 0; }
      .no-print { display: none; }
    }
    body { font-family: Arial, sans-serif; margin: 20px; color: #333; line-height: 1.4; }
    .header { position: relative; text-align: center; margin-bottom: 30px; border-bottom: 2px solid #2B256D; padding-bottom: 20px; }
    .header-logo { position: absolute; top: 0; right: 0; width: 60px; height: 60px; }
    .property-summary { background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%); padding: 20px; border-radius: 12px; margin: 20px 0; border: 1px solid #0ea5e9; }
    .valuation-highlight { text-align: center; background: linear-gradient(135deg, #dcfce7 0%, #bbf7d0 100%); padding: 30px; border-radius: 12px; margin: 20px 0; border: 2px solid #10b981; }
    .valuation-amount { font-size: 48px; font-weight: bold; color: #059669; margin: 15px 0; }
    .summary { display: flex; justify-content: space-around; margin: 20px 0; gap: 15px; }
    .summary-item { text-align: center; padding: 20px; border-radius: 12px; flex: 1; }
    .market-min { background-color: #fee2e2; color: #991b1b; border: 2px solid #fecaca; }
    .market-median { background-color: #dbeafe; color: #1e40af; border: 2px solid #93c5fd; }
    .market-max { background-color: #dcfce7; color: #166534; border: 2px solid #bbf7d0; }
    .trend-up { background-color: #dcfce7; color: #166534; border: 2px solid #bbf7d0; }
    .trend-down { background-color: #fee2e2; color: #991b1b; border: 2px solid #fecaca; }
    .trend-stable { background-color: #fef3c7; color: #92400e; border: 2px solid #fde68a; }
    .pricing-strategy { display: flex; justify-content: space-between; gap: 15px; margin: 20px 0; }
    .pricing-card { flex: 1; padding: 20px; border-radius: 12px; text-align: center; }
    .quick-sale { background: linear-gradient(135deg, #fed7aa 0%, #fdba74 100%); border: 2px solid #f97316; }
    .fair-price { background: linear-gradient(135deg, #bbf7d0 0%, #86efac 100%); border: 2px solid #10b981; }
    .premium-price { background: linear-gradient(135deg, #bfdbfe 0%, #93c5fd 100%); border: 2px solid #3b82f6; }
    .recommendation { background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%); padding: 25px; border-radius: 12px; margin-top: 25px; border: 2px solid #3b82f6; }
    .property-details { background: #f8fafc; padding: 20px; border-radius: 12px; margin: 20px 0; border: 1px solid #e2e8f0; }
    .details-grid { display: grid; grid-template-columns: repeat(2, 1fr); gap: 15px; margin-top: 15px; }
    .detail-item { background: white; padding: 12px; border-radius: 8px; border: 1px solid #e5e7eb; }
    .amenities-list { display: flex; flex-wrap: wrap; gap: 8px; margin-top: 10px; }
    .amenity-tag { background: #3b82f6; color: white; padding: 4px 12px; border-radius: 20px; font-size: 12px; }
    h1 { color: #2B256D; margin: 0 0 10px 0; font-size: 28px; }
    h2 { color: #374151; margin: 25px 0 15px 0; font-size: 20px; }
    h3 { color: #1f2937; margin: 15px 0 10px 0; font-size: 16px; }
    .confidence-badge { padding: 8px 16px; border-radius: 25px; font-size: 14px; font-weight: bold; text-transform: uppercase; display: inline-block; }
    .confidence-high { background-color: #dcfce7; color: #166534; border: 2px solid #bbf7d0; }
    .confidence-medium { background-color: #fef3c7; color: #92400e; border: 2px solid #fde68a; }
    .confidence-low { background-color: #fee2e2; color: #991b1b; border: 2px solid #fecaca; }
    .trend-indicator { font-size: 24px; font-weight: bold; }
    p { margin: 10px 0; }
    .print-btn { background: #2B256D; color: white; padding: 12px 24px; border: none; border-radius: 8px; cursor: pointer; margin: 20px; font-size: 14px; }
    .footer { text-align: center; margin-top: 40px; padding-top: 20px; border-top: 1px solid #e5e7eb; color: #6b7280; font-size: 12px; }
  </style>
</head>
<body>
  <button class="print-btn no-print" onclick="window.print()">Print/Save as PDF</button>
  
  <div class="header">
    ${logoBase64 ? `<img src="${logoBase64}" alt="NAL Logo" class="header-logo" />` : ''}
    <h1>NAL Price Intelligence Report</h1>
    <p><strong>Property Analysis & Market Insights</strong></p>
    <p>Generated on: ${new Date().toLocaleDateString('en-US', { year: 'numeric', month: 'long', day: 'numeric', hour: '2-digit', minute: '2-digit' })}</p>
  </div>

  <div class="property-summary">
    <h2>📍 Property Overview</h2>
    <div class="details-grid">
      <div class="detail-item"><strong>Locality:</strong> ${reportData.propertyDetails.locality}</div>
      <div class="detail-item"><strong>Property Type:</strong> ${getPropertyTypeDisplay(reportData.propertyDetails.propertyType)}</div>
      <div class="detail-item"><strong>Bedrooms:</strong> ${reportData.propertyDetails.bedrooms}</div>
      <div class="detail-item"><strong>Size:</strong> ${reportData.propertyDetails.size} sq ft</div>
      <div class="detail-item"><strong>Age:</strong> ${reportData.propertyDetails.age} years</div>
      <div class="detail-item"><strong>Builder:</strong> ${reportData.propertyDetails.builder}</div>
    </div>
    ${analysisMode === 'Enhanced' ? `
    <h3>🔍 Enhanced Analysis Details</h3>
    <div class="details-grid">
      <div class="detail-item"><strong>Bathrooms:</strong> ${reportData.propertyDetails.bathrooms}</div>
      <div class="detail-item"><strong>Floor:</strong> ${reportData.propertyDetails.floorNumber} of ${reportData.propertyDetails.totalFloors}</div>
      <div class="detail-item"><strong>Furnishing:</strong> ${getFurnishingDisplay(reportData.propertyDetails.furnishing)}</div>
      <div class="detail-item"><strong>Facing:</strong> ${getFacingDisplay(reportData.propertyDetails.facing)}</div>
    </div>
    ${reportData.propertyDetails.amenities.length > 0 ? `
    <div style="margin-top: 15px;">
      <strong>Amenities (${reportData.propertyDetails.amenities.length} selected):</strong>
      <div class="amenities-list">
        ${reportData.propertyDetails.amenities.map(amenity => `<span class="amenity-tag">${amenity}</span>`).join('')}
      </div>
    </div>` : ''}
    ` : ''}
    <div style="margin-top: 20px; text-align: center;">
      <span class="confidence-badge confidence-${reportData.analysisResults.valuation.confidence.toLowerCase()}">
        ${reportData.analysisMode} Analysis Mode • ${reportData.factorsUsed} Factors Used
      </span>
    </div>
  </div>

  <div class="valuation-highlight">
    <h2>💰 Property Valuation</h2>
    <div class="valuation-amount">${formatCurrency(reportData.analysisResults.valuation.estimate)}</div>
    <span class="confidence-badge confidence-${reportData.analysisResults.valuation.confidence.toLowerCase()}">
      ${reportData.analysisResults.valuation.confidence} Confidence
    </span>
    <p style="margin-top: 15px; font-size: 14px;">Based on comprehensive market analysis using ${reportData.factorsUsed} property factors</p>
  </div>

  <h2>📊 Market Benchmarking</h2>
  <div class="summary">
    <div class="summary-item market-min">
      <h3>₹${reportData.analysisResults.benchmarking.market_min.toLocaleString()}</h3>
      <p><strong>Market Minimum</strong></p>
      <p>per sq ft</p>
    </div>
    <div class="summary-item market-median">
      <h3>₹${reportData.analysisResults.benchmarking.market_median.toLocaleString()}</h3>
      <p><strong>Market Median</strong></p>
      <p>per sq ft</p>
    </div>
    <div class="summary-item market-max">
      <h3>₹${reportData.analysisResults.benchmarking.market_max.toLocaleString()}</h3>
      <p><strong>Market Maximum</strong></p>
      <p>per sq ft</p>
    </div>
  </div>
  
  <div style="text-align: center; margin: 20px 0; padding: 20px; background: #f8fafc; border-radius: 12px;">
    <h3>Market Position: ${reportData.analysisResults.benchmarking.percentile}th Percentile</h3>
    <p>Your property is valued higher than ${reportData.analysisResults.benchmarking.percentile}% of comparable properties in the area</p>
  </div>

  <h2>📈 Locality Trend Analysis</h2>
  <div class="summary-item trend-${reportData.analysisResults.locality_trend.direction.toLowerCase()}" style="margin: 20px auto; max-width: 400px;">
    <div class="trend-indicator">Market Trend: ${reportData.analysisResults.locality_trend.direction}</div>
    <p><strong>Last 3 Months Price Movement</strong></p>
    <div style="display: flex; justify-content: space-between; margin-top: 15px;">
      ${reportData.analysisResults.locality_trend.last_3_months.map((price, idx) => `
        <div style="text-align: center;">
          <div style="font-weight: bold;">Month ${idx + 1}</div>
          <div>₹${price.toLocaleString()}</div>
        </div>
      `).join('')}
    </div>
  </div>

  <h2>💡 Strategic Pricing Recommendations</h2>
  <div class="pricing-strategy">
    <div class="pricing-card quick-sale">
      <h3>⚡ Quick Sale</h3>
      <div style="font-size: 24px; font-weight: bold; margin: 10px 0;">${formatCurrency(reportData.analysisResults.pricing_guidance.fast_sale)}</div>
      <p><strong>Sell within 30 days</strong></p>
      <p>Competitive pricing for fast liquidity</p>
    </div>
    <div class="pricing-card fair-price">
      <h3>🎯 Fair Market Price</h3>
      <div style="font-size: 24px; font-weight: bold; margin: 10px 0;">${formatCurrency(reportData.analysisResults.pricing_guidance.fair_price)}</div>
      <p><strong>Market Equilibrium</strong></p>
      <p>Balanced pricing for optimal returns</p>
    </div>
    <div class="pricing-card premium-price">
      <h3>⭐ Premium Pricing</h3>
      <div style="font-size: 24px; font-weight: bold; margin: 10px 0;">${formatCurrency(reportData.analysisResults.pricing_guidance.premium)}</div>
      <p><strong>Hot Market Strategy</strong></p>
      <p>Maximum value in favorable conditions</p>
    </div>
  </div>

  <div class="recommendation">
    <h2>🎓 Professional Market Analysis</h2>
    <p><strong>Overall Confidence Level:</strong> 
      <span class="confidence-badge confidence-${reportData.analysisResults.pricing_guidance.confidence_level.toLowerCase()}">
        ${reportData.analysisResults.pricing_guidance.confidence_level}
      </span>
    </p>
    <p style="margin-top: 20px; font-size: 16px; line-height: 1.6;">${getMarketRecommendation()}</p>
  </div>

  <div class="footer">
    <p><strong>Disclaimer:</strong> This report is generated based on available market data and property information provided. 
    Actual market conditions may vary. For investment decisions, please consult with real estate professionals.</p>
    <p>© 2024 New Age Land. All rights reserved. | Generated by NAL Price Intelligence System</p>
  </div>
</body>
</html>`;

    // Create blob and download
    const blob = new Blob([htmlContent], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    
    // Open in new window for printing
    const printWindow = window.open(url, '_blank');
    if (printWindow) {
      printWindow.onload = () => {
        setTimeout(() => {
          printWindow.focus();
          printWindow.print();
        }, 500);
      };
    } else {
      // Fallback: direct download
      const a = document.createElement('a');
      a.href = url;
      a.download = `price-intelligence-report-${formData.locality.replace(/\s+/g, '-').toLowerCase()}-${new Date().toISOString().split('T')[0]}.html`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    }
  };

  return (
    <div className="flex-1 p-4 md:p-6 lg:p-8 space-y-4 md:space-y-6">
      <div className="text-center md:text-left">
        <h2 className="text-2xl md:text-3xl font-bold tracking-tight bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">Price Intelligence</h2>
        <p className="text-sm md:text-base text-gray-600 mt-2">Get comprehensive market analysis and pricing guidance</p>
      </div>

      <div className="grid grid-cols-1 xl:grid-cols-3 gap-6 lg:gap-8">
        {/* Form */}
        <Card className="xl:col-span-1 shadow-xl border-0 bg-white">
          <CardHeader className="bg-gradient-to-r from-blue-50 to-indigo-50 border-b">
            <CardTitle className="flex items-center gap-2 text-gray-900">
              <Target className="w-5 h-5 text-blue-600" />
              Property Details
            </CardTitle>
          </CardHeader>
          <CardContent className="p-6">
            <form onSubmit={handleSubmit} className="space-y-6">
              {/* Basic Details Section */}
              <div className="bg-gradient-to-r from-blue-50 to-indigo-50 p-4 md:p-6 rounded-xl border border-blue-200 space-y-4">
                <h3 className="font-semibold text-gray-800 flex items-center gap-2">
                  <MapPin className="w-4 h-4 text-blue-600" />
                  Basic Information
                </h3>
                <div>
                  <Label className="text-sm font-medium text-gray-700">Locality</Label>
                  <Input
                    type="text"
                    value={formData.locality}
                    onChange={(e) => setFormData({ ...formData, locality: e.target.value.toLowerCase() })}
                    placeholder="e.g., koramangala, indiranagar, whitefield"
                    className="mt-1 border-gray-300 focus:border-blue-500 focus:ring-blue-500"
                    required
                  />
                  <p className="text-xs text-blue-600 mt-1">Try: koramangala, indiranagar, whitefield, electronic_city</p>
                </div>

                <div>
                  <Label className="text-sm font-medium text-gray-700">Property Type</Label>
                  <Select
                    value={formData.property_type}
                    onValueChange={(value: any) => setFormData({ ...formData, property_type: value })}
                  >
                    <SelectTrigger className="mt-1 border-gray-300 focus:border-blue-500">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="apartment">Apartment</SelectItem>
                      <SelectItem value="villa">Villa</SelectItem>
                      <SelectItem value="plot">Plot</SelectItem>
                      <SelectItem value="commercial">Commercial</SelectItem>
                      <SelectItem value="pg">PG</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              {/* Property Specifications */}
              <div className="bg-gradient-to-r from-green-50 to-emerald-50 p-4 md:p-6 rounded-xl border border-green-200 space-y-4">
                <h3 className="font-semibold text-gray-800 flex items-center gap-2">
                  <Calculator className="w-4 h-4 text-green-600" />
                  Property Specifications
                </h3>
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <Label className="text-sm font-medium text-gray-700">Bedrooms</Label>
                    <Input
                      type="text"
                      inputMode="numeric"
                      value={formData.bedrooms || ''}
                      onChange={(e) => {
                        const val = e.target.value.replace(/[^0-9]/g, '');
                        setFormData({ ...formData, bedrooms: val ? parseInt(val) : 0 });
                      }}
                      placeholder="Enter bedrooms"
                      className="mt-1 border-gray-300 focus:border-green-500 focus:ring-green-500"
                      required
                    />
                  </div>
                  <div>
                    <Label className="text-sm font-medium text-gray-700">Age (years)</Label>
                    <Input
                      type="text"
                      inputMode="numeric"
                      value={formData.age || ''}
                      onChange={(e) => {
                        const val = e.target.value.replace(/[^0-9]/g, '');
                        setFormData({ ...formData, age: val ? parseInt(val) : 0 });
                      }}
                      placeholder="Enter age"
                      className="mt-1 border-gray-300 focus:border-green-500 focus:ring-green-500"
                      required
                    />
                  </div>
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-700">Size (sq ft)</Label>
                  <Input
                    type="text"
                    inputMode="numeric"
                    value={formData.size || ''}
                    onChange={(e) => {
                      const val = e.target.value.replace(/[^0-9]/g, '');
                      setFormData({ ...formData, size: val ? parseInt(val) : 0 });
                    }}
                    placeholder="Enter size in sq ft"
                    className="mt-1 border-gray-300 focus:border-green-500 focus:ring-green-500"
                    required
                  />
                </div>
                <div>
                  <Label className="text-sm font-medium text-gray-700">Builder (Optional)</Label>
                  <Input
                    type="text"
                    value={formData.builder || ""}
                    onChange={(e) => setFormData({ ...formData, builder: e.target.value })}
                    placeholder="e.g., Prestige, DLF, Godrej"
                    className="mt-1 border-gray-300 focus:border-green-500 focus:ring-green-500"
                  />
                </div>
              </div>

              {/* Advanced Options Toggle */}
              <div className="bg-gradient-to-r from-purple-50 to-indigo-50 p-4 rounded-xl border border-purple-200">
                <Button
                  type="button"
                  variant="ghost"
                  className="w-full justify-between hover:bg-purple-100 text-purple-800 transition-all duration-200"
                  onClick={() => setShowAdvanced(!showAdvanced)}
                >
                  <span className="text-sm font-medium flex items-center gap-2">
                    <Star className="w-4 h-4" />
                    Advanced Options (Optional)
                  </span>
                  {showAdvanced ? <ChevronUp className="w-4 h-4" /> : <ChevronDown className="w-4 h-4" />}
                </Button>
                <p className="text-xs text-purple-600 mt-2 text-center">Add details for more accurate valuation</p>
              </div>

              {/* Advanced Fields */}
              {showAdvanced && (
                <div className="bg-gradient-to-r from-purple-50 to-pink-50 p-4 md:p-6 rounded-xl border border-purple-200 space-y-4 animate-in slide-in-from-top-2 duration-300">
                  <h3 className="font-semibold text-gray-800 flex items-center gap-2">
                    <Star className="w-4 h-4 text-purple-600" />
                    Additional Details
                  </h3>
                  <div className="grid grid-cols-3 gap-3">
                    <div>
                      <Label className="text-xs font-medium text-gray-700">Bathrooms</Label>
                      <Select
                        value={formData.bathrooms?.toString() || ""}
                        onValueChange={(value) => setFormData({ ...formData, bathrooms: parseInt(value) })}
                      >
                        <SelectTrigger className="mt-1 h-8 text-sm">
                          <SelectValue placeholder="Select" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="1">1</SelectItem>
                          <SelectItem value="2">2</SelectItem>
                          <SelectItem value="3">3</SelectItem>
                          <SelectItem value="4">4</SelectItem>
                          <SelectItem value="5">5+</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label className="text-xs font-medium text-gray-700">Floor No.</Label>
                      <Select
                        value={formData.floor_number?.toString() || ""}
                        onValueChange={(value) => setFormData({ ...formData, floor_number: parseInt(value) })}
                      >
                        <SelectTrigger className="mt-1 h-8 text-sm">
                          <SelectValue placeholder="Select" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="0">Ground</SelectItem>
                          <SelectItem value="1">1st</SelectItem>
                          <SelectItem value="2">2nd</SelectItem>
                          <SelectItem value="3">3rd</SelectItem>
                          <SelectItem value="4">4th</SelectItem>
                          <SelectItem value="5">5th</SelectItem>
                          <SelectItem value="6">6th</SelectItem>
                          <SelectItem value="7">7th</SelectItem>
                          <SelectItem value="8">8th</SelectItem>
                          <SelectItem value="9">9th</SelectItem>
                          <SelectItem value="10">10th+</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label className="text-xs font-medium text-gray-700">Total Floors</Label>
                      <Select
                        value={formData.total_floors?.toString() || ""}
                        onValueChange={(value) => setFormData({ ...formData, total_floors: parseInt(value) })}
                      >
                        <SelectTrigger className="mt-1 h-8 text-sm">
                          <SelectValue placeholder="Select" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="1">1</SelectItem>
                          <SelectItem value="2">2</SelectItem>
                          <SelectItem value="3">3</SelectItem>
                          <SelectItem value="4">4</SelectItem>
                          <SelectItem value="5">5</SelectItem>
                          <SelectItem value="7">7</SelectItem>
                          <SelectItem value="10">10</SelectItem>
                          <SelectItem value="15">15</SelectItem>
                          <SelectItem value="20">20+</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                  </div>

                  <div className="grid grid-cols-2 gap-3">
                    <div>
                      <Label className="text-xs font-medium text-gray-700">Furnishing</Label>
                      <Select
                        value={formData.furnishing || "unfurnished"}
                        onValueChange={(value) => setFormData({ ...formData, furnishing: value })}
                      >
                        <SelectTrigger className="mt-1 h-8 text-sm">
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="unfurnished">Unfurnished</SelectItem>
                          <SelectItem value="semi_furnished">Semi Furnished</SelectItem>
                          <SelectItem value="fully_furnished">Fully Furnished</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label className="text-xs font-medium text-gray-700">Facing Direction</Label>
                      <Select
                        value={formData.facing || ""}
                        onValueChange={(value) => setFormData({ ...formData, facing: value })}
                      >
                        <SelectTrigger className="mt-1 h-8 text-sm">
                          <SelectValue placeholder="Select facing" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="north">🧭 North</SelectItem>
                          <SelectItem value="south">🧭 South</SelectItem>
                          <SelectItem value="east">🧭 East</SelectItem>
                          <SelectItem value="west">🧭 West</SelectItem>
                          <SelectItem value="northeast">🧭 North-East</SelectItem>
                          <SelectItem value="northwest">🧭 North-West</SelectItem>
                          <SelectItem value="southeast">🧭 South-East</SelectItem>
                          <SelectItem value="southwest">🧭 South-West</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                  </div>

                  <div>
                    <Label className="text-xs font-medium text-gray-700 mb-2 block">Amenities ({formData.amenities?.length || 0} selected)</Label>
                    <div className="flex flex-wrap gap-2 max-h-32 overflow-y-auto p-3 bg-gray-50 rounded-lg border">
                      {availableAmenities.map((amenity) => (
                        <button
                          key={amenity}
                          type="button"
                          onClick={() => toggleAmenity(amenity)}
                          className={`px-3 py-1 rounded-full text-xs font-medium transition-all ${
                            formData.amenities?.includes(amenity)
                              ? 'bg-blue-500 text-white shadow-md'
                              : 'bg-white text-gray-600 border border-gray-300 hover:border-blue-300 hover:bg-blue-50'
                          }`}
                        >
                          {amenity}
                        </button>
                      ))}
                    </div>
                  </div>
                </div>
              )}

              <Button 
                type="submit" 
                className="w-full bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white font-semibold py-4 rounded-xl shadow-xl transition-all duration-300 transform hover:scale-105 hover:shadow-2xl" 
                disabled={loading || !formData.locality || formData.bedrooms < 1 || formData.size < 100 || formData.age < 0}
              >
                {loading ? (
                  <div className="flex items-center gap-2">
                    <div className="animate-spin rounded-full h-4 w-4 border-2 border-white border-t-transparent"></div>
                    Analyzing...
                  </div>
                ) : (
                  <div className="flex items-center gap-2">
                    <TrendingUp className="w-4 h-4" />
                    Analyze Market
                  </div>
                )}
              </Button>
            </form>
          </CardContent>
        </Card>

        {/* Results */}
        <div className="xl:col-span-2 space-y-4 md:space-y-6">
          {error && (
            <Card className="border-red-200 bg-gradient-to-r from-red-50 to-pink-50 animate-in slide-in-from-top-2">
              <CardContent className="p-4 md:p-6">
                <div className="flex items-start gap-3">
                  <AlertCircle className="w-5 h-5 text-red-600 mt-0.5 flex-shrink-0" />
                  <div>
                    <h3 className="font-semibold text-red-900">Analysis Error</h3>
                    <p className="text-sm text-red-700 mt-1 leading-relaxed">{error}</p>
                  </div>
                </div>
              </CardContent>
            </Card>
          )}

          {result && (
            <>
              {/* Valuation - Hero Card */}
              <Card className="shadow-2xl border-0 bg-gradient-to-br from-emerald-50 to-green-50 animate-in slide-in-from-bottom-4">
                <CardContent className="p-6 md:p-8">
                  <div className="text-center">
                    <div className="flex items-center justify-center gap-2 mb-6">
                      <div className="p-2 bg-emerald-100 rounded-full">
                        <TrendingUp className="w-6 h-6 text-emerald-600" />
                      </div>
                      <h2 className="text-xl md:text-2xl font-bold text-gray-900">Property Valuation</h2>
                    </div>
                    <h3 className="text-4xl md:text-6xl font-bold mb-6 text-emerald-600 animate-pulse">
                      {formatCurrency(result.valuation.estimate)}
                    </h3>
                    <div className="flex items-center justify-center gap-4">
                      <Badge className={`${getConfidenceColor(result.valuation.confidence)} border-2 px-4 py-2 text-sm font-semibold`}>
                        <div className={`w-2 h-2 rounded-full mr-2 ${
                          result.valuation.confidence === 'HIGH' ? 'bg-green-500' :
                          result.valuation.confidence === 'MEDIUM' ? 'bg-yellow-500' : 'bg-red-500'
                        }`}></div>
                        {result.valuation.confidence} Confidence
                      </Badge>
                    </div>
                    <p className="text-gray-700 mt-4 text-sm">Based on XGBoost ML prediction & comprehensive market analysis</p>
                    <div className="flex items-center justify-center gap-2 mt-2">
                      <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                      <span className="text-xs text-green-600 font-medium">AI-Powered Analysis</span>
                    </div>
                    <div className="mt-4">
                      <Button 
                        onClick={generatePDF}
                        className="bg-white text-emerald-600 border-2 border-emerald-600 hover:bg-emerald-50 font-semibold px-6 py-2 rounded-lg transition-all duration-200"
                      >
                        <Download className="w-4 h-4 mr-2" />
                        Download Report
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>

              {/* Market Analysis Grid */}
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-4 md:gap-6">
                {/* Benchmarking */}
                <Card className="shadow-xl border-0 bg-gradient-to-br from-blue-50 to-indigo-50 animate-in slide-in-from-left-4">
                  <CardHeader className="bg-gradient-to-r from-blue-100 to-indigo-100 border-b">
                    <CardTitle className="flex items-center gap-2 text-gray-900">
                      <BarChart3 className="w-5 h-5 text-blue-600" />
                      Market Benchmarking
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="p-4 md:p-6 space-y-4">
                    <div className="grid grid-cols-3 gap-3">
                      <div className="text-center p-3 bg-red-50 rounded-lg border border-red-200">
                        <ArrowDown className="w-4 h-4 text-red-600 mx-auto mb-1" />
                        <p className="text-xs text-red-600 font-medium mb-1">Market Min</p>
                        <p className="text-sm font-bold text-red-800">₹{result.benchmarking.market_min.toLocaleString()}</p>
                        <p className="text-xs text-red-500">per sqft</p>
                      </div>
                      <div className="text-center p-3 bg-blue-50 rounded-lg border-2 border-blue-300">
                        <Minus className="w-4 h-4 text-blue-600 mx-auto mb-1" />
                        <p className="text-xs text-blue-600 font-medium mb-1">Market Median</p>
                        <p className="text-sm font-bold text-blue-900">₹{result.benchmarking.market_median.toLocaleString()}</p>
                        <p className="text-xs text-blue-600">per sqft</p>
                      </div>
                      <div className="text-center p-3 bg-green-50 rounded-lg border border-green-200">
                        <ArrowUp className="w-4 h-4 text-green-600 mx-auto mb-1" />
                        <p className="text-xs text-green-600 font-medium mb-1">Market Max</p>
                        <p className="text-sm font-bold text-green-800">₹{result.benchmarking.market_max.toLocaleString()}</p>
                        <p className="text-xs text-green-500">per sqft</p>
                      </div>
                    </div>
                  </CardContent>
                </Card>

                {/* Property Percentile */}
                <Card className="shadow-xl border-0 bg-gradient-to-br from-purple-50 to-pink-50 animate-in slide-in-from-right-4">
                  <CardHeader className="bg-gradient-to-r from-purple-100 to-pink-100 border-b">
                    <CardTitle className="flex items-center gap-2 text-gray-900">
                      <Target className="w-5 h-5 text-purple-600" />
                      Market Position
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="p-4 md:p-6">
                    <div className="text-center mb-4">
                      <div className="text-4xl font-bold text-gray-900 mb-2">{result.benchmarking.percentile}%</div>
                      <p className="text-sm text-gray-700 font-medium">Market Percentile</p>
                    </div>
                    <div className="relative">
                      <div className="h-4 bg-gradient-to-r from-red-200 via-yellow-200 to-green-200 rounded-full overflow-hidden">
                        <div 
                          className="h-full bg-gradient-to-r from-purple-500 to-purple-600 rounded-full transition-all duration-1000 ease-out"
                          style={{ width: `${result.benchmarking.percentile}%` }}
                        />
                      </div>
                      <div className="flex justify-between text-xs text-gray-500 mt-2">
                        <span>0%</span>
                        <span>50%</span>
                        <span>100%</span>
                      </div>
                    </div>
                    <p className="text-xs text-gray-700 mt-3 text-center">
                      Higher than {result.benchmarking.percentile}% of comparable properties
                    </p>
                  </CardContent>
                </Card>
              </div>



              {/* Pricing Guidance */}
              <Card className="shadow-xl border-0 bg-gradient-to-br from-indigo-50 to-purple-50 animate-in slide-in-from-bottom-4">
                <CardHeader className="bg-gradient-to-r from-indigo-100 to-purple-100 border-b">
                  <CardTitle className="flex items-center gap-2 text-gray-900">
                    <Zap className="w-5 h-5 text-indigo-600" />
                    Pricing Strategy
                  </CardTitle>
                </CardHeader>
                <CardContent className="p-4 md:p-6">
                  <div className="grid grid-cols-1 lg:grid-cols-3 gap-4 md:gap-6">
                    <div className="relative p-4 md:p-6 bg-gradient-to-br from-orange-50 to-red-50 rounded-xl border-2 border-orange-200 hover:shadow-xl transition-all duration-300 transform hover:scale-105">
                      <div className="absolute top-3 right-3">
                        <Zap className="w-6 h-6 text-orange-500" />
                      </div>
                      <div className="mb-4">
                        <h4 className="font-bold text-orange-900 text-base md:text-lg mb-1">Quick Sale</h4>
                        <p className="text-xs text-orange-600 font-medium">Sell within 30 days</p>
                      </div>
                      <p className="text-2xl md:text-3xl font-bold text-orange-900 mb-2">
                        {formatCurrency(result.pricing_guidance.fast_sale)}
                      </p>
                      <div className="w-full bg-orange-200 rounded-full h-2">
                        <div className="bg-orange-500 h-2 rounded-full transition-all duration-1000" style={{ width: `${Math.round((result.pricing_guidance.fast_sale / result.pricing_guidance.premium) * 100)}%` }}></div>
                      </div>
                    </div>

                    <div className="relative p-4 md:p-6 bg-gradient-to-br from-green-50 to-emerald-50 rounded-xl border-2 border-green-300 hover:shadow-xl transition-all duration-300 transform hover:scale-105">
                      <div className="absolute top-3 right-3">
                        <Target className="w-6 h-6 text-green-500" />
                      </div>
                      <div className="mb-4">
                        <h4 className="font-bold text-green-900 text-base md:text-lg mb-1">Fair Price</h4>
                        <p className="text-xs text-green-600 font-medium">Market equilibrium</p>
                      </div>
                      <p className="text-2xl md:text-3xl font-bold text-green-900 mb-2">
                        {formatCurrency(result.pricing_guidance.fair_price)}
                      </p>
                      <div className="w-full bg-green-200 rounded-full h-2">
                        <div className="bg-green-500 h-2 rounded-full transition-all duration-1000" style={{ width: `${Math.round((result.pricing_guidance.fair_price / result.pricing_guidance.premium) * 100)}%` }}></div>
                      </div>
                    </div>

                    <div className="relative p-4 md:p-6 bg-gradient-to-br from-blue-50 to-indigo-50 rounded-xl border-2 border-blue-200 hover:shadow-xl transition-all duration-300 transform hover:scale-105">
                      <div className="absolute top-3 right-3">
                        <Star className="w-6 h-6 text-blue-500" />
                      </div>
                      <div className="mb-4">
                        <h4 className="font-bold text-blue-900 text-base md:text-lg mb-1">Premium</h4>
                        <p className="text-xs text-blue-600 font-medium">Hot market pricing</p>
                      </div>
                      <p className="text-2xl md:text-3xl font-bold text-blue-900 mb-2">
                        {formatCurrency(result.pricing_guidance.premium)}
                      </p>
                      <div className="w-full bg-blue-200 rounded-full h-2">
                        <div className="bg-blue-500 h-2 rounded-full transition-all duration-1000" style={{ width: '100%' }}></div>
                      </div>
                    </div>
                  </div>

                  <div className="mt-6 p-4 bg-gradient-to-r from-gray-50 to-gray-100 rounded-xl border">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-2">
                        <AlertCircle className="w-5 h-5 text-gray-800" />
                        <span className="text-sm font-semibold text-gray-900">Analysis Confidence</span>
                      </div>
                      <div className="flex items-center gap-3">
                        <Badge className={`${getConfidenceColor(result.pricing_guidance.confidence_level)} border-2 px-4 py-2 text-sm font-bold`}>
                          <div className={`w-2 h-2 rounded-full mr-2 ${
                            result.pricing_guidance.confidence_level === 'HIGH' ? 'bg-green-500' :
                            result.pricing_guidance.confidence_level === 'MEDIUM' ? 'bg-yellow-500' : 'bg-red-500'
                          }`}></div>
                          {result.pricing_guidance.confidence_level}
                        </Badge>
                        <Button 
                          onClick={generatePDF}
                          size="sm"
                          className="bg-indigo-600 hover:bg-indigo-700 text-white font-medium px-4 py-2 rounded-lg transition-all duration-200"
                        >
                          <Download className="w-4 h-4 mr-2" />
                          Download Report
                        </Button>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </>
          )}

          {!result && !error && !loading && (
            <Card className="shadow-xl border-0 bg-gradient-to-br from-blue-50 via-indigo-50 to-purple-50 animate-in fade-in-0 duration-500">
              <CardContent className="p-8 md:p-12 text-center">
                <div className="bg-gradient-to-br from-blue-500 to-purple-600 w-16 h-16 md:w-20 md:h-20 rounded-full flex items-center justify-center mx-auto mb-6 animate-pulse">
                  <Target className="w-8 h-8 md:w-10 md:h-10 text-white" />
                </div>
                <h3 className="text-xl md:text-2xl font-bold text-gray-900 mb-3">Ready to Analyze Your Property</h3>
                <p className="text-sm md:text-base text-gray-600 max-w-md mx-auto leading-relaxed">
                  Fill in the property details in the form and click "Analyze Market" to get comprehensive pricing intelligence and market insights.
                </p>
                <div className="mt-6 flex flex-wrap items-center justify-center gap-3 md:gap-4 text-xs md:text-sm text-gray-500">
                  <div className="flex items-center gap-1">
                    <div className="w-2 h-2 bg-blue-500 rounded-full animate-pulse"></div>
                    <span>Market Analysis</span>
                  </div>
                  <div className="flex items-center gap-1">
                    <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                    <span>Price Guidance</span>
                  </div>
                  <div className="flex items-center gap-1">
                    <div className="w-2 h-2 bg-purple-500 rounded-full animate-pulse"></div>
                    <span>Trend Analysis</span>
                  </div>
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
    </div>
  );
}
