import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Badge } from "../components/ui/badge";
import { Button } from "../components/ui/button";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../components/ui/tabs";
import { Megaphone, Eye, MousePointer, Star, TrendingUp, Calendar, IndianRupee } from "lucide-react";
import { api } from "../utils/api";
import { mockProperties } from "../lib/mock-data";
import type { Property as PropertyRow } from "../types";
import { createPromotion } from "../utils/promotions";
import { usePromotion } from "../context/PromotionContext";

export default function Promote() {
  const [properties, setProperties] = useState<PropertyRow[]>([]);
  const [activePromotions, setActivePromotions] = useState<PropertyRow[]>([]);
  const [loading, setLoading] = useState(true);
  const { promoteProperty, updatePropertyPromotionStatus } = usePromotion();

  useEffect(() => {
    loadProperties();
  }, []);

  const loadProperties = async () => {
    try {
      setLoading(true);
      const list = await api.fetchProperties();
      
      // Update properties with promotion status
      const updatedList = list.map((p: PropertyRow) => updatePropertyPromotionStatus(p));
      
      const nonPromotedProperties = updatedList.filter((p: PropertyRow) => !p.isPromoted);
      const promotedProperties = updatedList.filter((p: PropertyRow) => p.isPromoted);
      setProperties(nonPromotedProperties);
      setActivePromotions(promotedProperties);
    } catch (error) {
      console.error('Failed to load properties:', error);
      // Fallback to mock data
      const fallback: PropertyRow[] = mockProperties.map((p: any) => {
        const property = {
          id: String(p.id),
          title: p.title,
          type: (p.type || p.propertyType || 'apartment').toString().toLowerCase(),
          bhk: p.bhk ?? undefined,
          address: p.address || p.location || '',
          city: p.city || '',
          state: p.state || '',
          pincode: p.pincode || '',
          landmark: p.landmark || '',
          area: Number(p.area ?? p.size ?? 0),
          price: Number(p.price ?? 0),
          listingIntent: p.listingIntent || (p.isUrgentSale ? 'urgent-sale' : 'sale'),
          status: p.status || 'active',
          amenities: p.features || [],
          customAmenities: [],
          virtualTour: undefined,
          sellerType: 'owner',
          biddingEnabled: Boolean(p.isBiddingEnabled),
          coordinates: undefined,
          views: Number(p.views || 0),
          inquiries: 0,
          createdAt: p.createdAt instanceof Date ? p.createdAt.toISOString() : String(p.createdAt || ''),
          updatedAt: p.updatedAt instanceof Date ? p.updatedAt.toISOString() : String(p.updatedAt || ''),
          isPromoted: false
        };
        return updatePropertyPromotionStatus(property);
      });
      
      const nonPromotedFallback = fallback.filter(p => !p.isPromoted);
      const promotedFallback = fallback.filter(p => p.isPromoted);
      setProperties(nonPromotedFallback);
      setActivePromotions(promotedFallback);
    } finally {
      setLoading(false);
    }
  };

  const handlePromoteNow = async (property: PropertyRow) => {
    try {
      // Create promotion
      const promotionData = {
        property_id: property.id,
        type: "featured" as const,
        platforms: ["website", "social"]
      };

      const result = await createPromotion(promotionData);
      
      if (result.promotion) {
        // Update promotion context
        promoteProperty(property.id);
        
        // Update property status locally
        const updatedProperty = updatePropertyPromotionStatus({
          ...property,
          promotionStartDate: result.promotion.start_date,
          promotionEndDate: result.promotion.end_date
        });
        
        // Move from properties to active promotions
        setProperties(prev => prev.filter(p => p.id !== property.id));
        setActivePromotions(prev => [...prev, updatedProperty]);
      }
    } catch (error) {
      console.error('Failed to promote property:', error);
    }
  };

  if (loading) return <div className="p-4">Loading properties…</div>;

  return (
    <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Promote Properties</h2>
      </div>
      
      <div className="space-y-6">
        {/* Stats Cards */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <Card className="bg-gradient-to-br from-yellow-50 to-yellow-100 border-yellow-200">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-yellow-700">
                Active Promotions
                <Star className="w-5 h-5 text-yellow-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-yellow-800">{activePromotions.length}</div>
              <p className="text-xs text-yellow-600 mt-1">properties promoted</p>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-green-50 to-green-100 border-green-200">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-green-700">
                Total Impressions
                <Eye className="w-5 h-5 text-green-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-green-800">15.2K</div>
              <p className="text-xs text-green-600 mt-1">this month</p>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-blue-700">
                Click-through Rate
                <MousePointer className="w-5 h-5 text-blue-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-blue-800">3.8%</div>
              <p className="text-xs text-blue-600 mt-1">+0.5% from last month</p>
            </CardContent>
          </Card>
        </div>

        {/* Tabs for Properties and Active Promotions */}
        <Tabs defaultValue="properties" className="w-full">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="properties">Properties ({properties.length})</TabsTrigger>
            <TabsTrigger value="active">Your Active Promotions ({activePromotions.length})</TabsTrigger>
          </TabsList>
          
          <TabsContent value="properties" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Promote Your Properties</CardTitle>
                <p className="text-sm text-muted-foreground">
                  Select properties to promote and increase their visibility
                </p>
              </CardHeader>
              <CardContent>
                {properties.length > 0 ? (
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                    {properties.map((property) => (
                      <Card key={property.id} className="overflow-hidden">
                        <div className="relative">
                          <img 
                            src="https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=200"
                            alt={property.title}
                            className="w-full h-40 object-cover"
                          />
                          <Badge className="absolute top-2 right-2 bg-green-100 text-green-800">
                            {property.status}
                          </Badge>
                        </div>
                        <CardContent className="p-4">
                          <h3 className="font-semibold text-lg mb-2">{property.title}</h3>
                          <p className="text-gray-600 text-sm mb-2">{property.city}, {property.state}</p>
                          <div className="flex justify-between items-center mb-3">
                            <span className="text-xl font-bold text-green-600">₹{property.price.toLocaleString()}</span>
                            <span className="text-sm text-gray-500">{property.views} views</span>
                          </div>
                          <Button 
                            onClick={() => handlePromoteNow(property)}
                            className="w-full bg-yellow-600 hover:bg-yellow-700"
                          >
                            <TrendingUp className="w-4 h-4 mr-2" />
                            Promote Now
                          </Button>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                ) : (
                  <div className="text-center py-8">
                    <Megaphone className="mx-auto w-12 h-12 text-gray-400 mb-4" />
                    <h3 className="text-lg font-medium text-gray-900 mb-2">
                      All properties are promoted
                    </h3>
                    <p className="text-gray-600">
                      You have no properties available for promotion
                    </p>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
          
          <TabsContent value="active" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle>Your Active Promotions</CardTitle>
                <p className="text-sm text-muted-foreground">
                  Monitor your promoted properties and their performance
                </p>
              </CardHeader>
              <CardContent>
                {activePromotions.length > 0 ? (
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                    {activePromotions.map((property) => (
                      <Card key={property.id} className="overflow-hidden border-yellow-200">
                        <div className="relative">
                          <img 
                            src="https://images.unsplash.com/photo-1586023492125-27b2c045efd7?ixlib=rb-4.0.3&auto=format&fit=crop&w=400&h=200"
                            alt={property.title}
                            className="w-full h-40 object-cover"
                          />
                          <Badge className="absolute top-2 right-2 bg-yellow-100 text-yellow-800">
                            <Star className="w-3 h-3 mr-1" />
                            Promoted
                          </Badge>
                        </div>
                        <CardContent className="p-4">
                          <h3 className="font-semibold text-lg mb-2">{property.title}</h3>
                          <p className="text-gray-600 text-sm mb-2">{property.city}, {property.state}</p>
                          <div className="flex justify-between items-center mb-3">
                            <span className="text-xl font-bold text-green-600">₹{property.price.toLocaleString()}</span>
                            <span className="text-sm text-gray-500">{property.views} views</span>
                          </div>
                          <div className="space-y-2">
                            <div className="flex justify-between text-sm">
                              <span className="text-gray-600">Promotion Status:</span>
                              <Badge className="bg-green-100 text-green-800">Active</Badge>
                            </div>
                            <div className="flex justify-between text-sm">
                              <span className="text-gray-600">Days Remaining:</span>
                              <span className="font-medium">12 days</span>
                            </div>
                          </div>
                        </CardContent>
                      </Card>
                    ))}
                  </div>
                ) : (
                  <div className="text-center py-8">
                    <Star className="mx-auto w-12 h-12 text-gray-400 mb-4" />
                    <h3 className="text-lg font-medium text-gray-900 mb-2">
                      No active promotions
                    </h3>
                    <p className="text-gray-600">
                      Start promoting your properties to increase visibility
                    </p>
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}
