import React from "react";
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
} from "../components/ui/card";
import { Button } from "../components/ui/button";
import { Badge } from "../components/ui/badge";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "../components/ui/select";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "../components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "../components/ui/alert-dialog";
import { Input } from "../components/ui/input";
import { Label } from "../components/ui/label";
import { Textarea } from "../components/ui/textarea";
import {
  Eye,
  Edit,
  Pause,
  Trash2,
  Plus,
  MapPin,
  Calendar,
  IndianRupee,
  Bed,
  Bath,
  Square,
  Play,
} from "lucide-react";
import { Link, useLocation } from "wouter";
import { useState, useEffect, useMemo } from "react";
import { api } from "../utils/api";
import type { Property as PropertyRow } from "../types";
import { mockProperties } from "../lib/mock-data";
import PropertyDetailsModal from "../components/PropertyDetailsModal";
import SellerPropertyCard from "../components/SellerPropertyCard";
import AddAssetModal from "../components/AddAssetModal";

export default function Properties() {
  const [, navigate] = useLocation();
  const [filters, setFilters] = useState({
    status: "all",
    listingIntent: "all",
    city: "",
    search: "",
  });
  const [selectedProperty, setSelectedProperty] = useState<any>(null);
  const [isViewDialogOpen, setIsViewDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [editForm, setEditForm] = useState({
    title: "",
    location: "",
    status: "",
    price: "",
    description: "",
  });
  const [isAddAssetModalOpen, setIsAddAssetModalOpen] = useState(false);
  const [properties, setProperties] = useState<PropertyRow[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    (async () => {
      try {
        setLoading(true);
        const list = await api.fetchProperties();
        console.log("Loaded properties:", list);
        setProperties(list);
      } catch (error) {
        console.error("Failed to load properties:", error);
        // Fallback to mock data if API is unreachable
        const fallback: PropertyRow[] = mockProperties.map((p: any) => ({
          id: String(p.id),
          title: p.title,
          type: (p.type || p.propertyType || "apartment")
            .toString()
            .toLowerCase(),
          bhk: p.bhk ?? undefined,
          address: p.address || p.location || "",
          city: p.city || "",
          state: p.state || "",
          pincode: p.pincode || "",
          landmark: p.landmark || "",
          area: Number(p.area ?? p.size ?? 0),
          price: Number(p.price ?? 0),
          listingIntent:
            p.listingIntent || (p.isUrgentSale ? "urgent-sale" : "sale"),
          status: p.status || "active",
          amenities: p.features || [],
          customAmenities: [],
          virtualTour: undefined,
          sellerType: "owner",
          biddingEnabled: Boolean(p.isBiddingEnabled),
          coordinates: undefined,
          views: Number(p.views || 0),
          inquiries: 0,
          createdAt:
            p.createdAt instanceof Date
              ? p.createdAt.toISOString()
              : String(p.createdAt || ""),
          updatedAt:
            p.updatedAt instanceof Date
              ? p.updatedAt.toISOString()
              : String(p.updatedAt || ""),
        }));
        setProperties(fallback);
      } finally {
        setLoading(false);
      }
    })();
  }, []);

  const getStatusColor = (status: string) => {
    switch (status) {
      case "active":
        return "bg-green-100 text-green-800";
      case "inactive":
        return "bg-orange-100 text-orange-800";
      case "under_review":
        return "bg-blue-100 text-blue-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getStatusLabel = (status: string) => {
    switch (status) {
      case "active":
        return "Active";
      case "inactive":
        return "Inactive";
      case "under_review":
        return "Under Review";
      default:
        return status;
    }
  };

  // Filter properties based on current filters
  const filteredProperties = useMemo(() => {
    return properties.filter((property: PropertyRow) => {
      // Check if property is an asset (handle both boolean and numeric values)
      const isAsset =
        property.isasset === true ||
        property.isasset === 1 ||
        property.isasset === true ||
        property.isasset === 1;

      // Hide ALL pending/under_review assets - they should only show after admin approval
      if (
        isAsset &&
        (property.status === "pending" || property.status === "under_review")
      ) {
        console.log(
          "🚫 HIDING PENDING ASSET:",
          property.id,
          "status:",
          property.status,
        );
        return false;
      }

      // For assets, only show if status is 'approved' or 'active'
      if (isAsset && !["approved", "active"].includes(property.status)) {
        console.log(
          "🚫 HIDING NON-APPROVED ASSET:",
          property.id,
          "status:",
          property.status,
        );
        return false;
      }

      const matchesStatus =
        filters.status === "all" || property.status === filters.status;
      const matchesIntent =
        filters.listingIntent === "all" ||
        property.listingIntent === filters.listingIntent;
      const matchesCity = !filters.city || property.city === filters.city;
      const matchesSearch =
        !filters.search ||
        property.title.toLowerCase().includes(filters.search.toLowerCase()) ||
        property.address.toLowerCase().includes(filters.search.toLowerCase()) ||
        property.city.toLowerCase().includes(filters.search.toLowerCase());

      return matchesStatus && matchesIntent && matchesCity && matchesSearch;
    });
  }, [properties, filters]);

  if (loading) return <div className="p-4">Loading properties…</div>;

  const handleView = (property: any) => {
    setSelectedProperty(property);
    setIsViewDialogOpen(true);
  };

  const handleEdit = (property: any) => {
    setSelectedProperty(property);
    setEditForm({
      title: property.title,
      location: property.location,
      status: property.status,
      price: property.price,
      description: property.description || "",
    });
    setIsEditDialogOpen(true);
  };

  const handlePause = (property: any) => {
    // Toggle between active and inactive for demo purposes
    const newStatus = property.status === "active" ? "inactive" : "active";

    // Update the properties state to trigger re-render
    setProperties((prevProperties) =>
      prevProperties.map((p) =>
        p.id === property.id ? { ...p, status: newStatus } : p,
      ),
    );

    // In real app, update backend here
    // No alert needed - state change is immediate and visible
  };

  const handleDelete = (property: any) => {
    setSelectedProperty(property);
    setIsDeleteDialogOpen(true);
  };

  const handleListForSale = async (property: PropertyRow) => {
    try {
      console.log("Listing asset for sale:", property.id);
      const updatedProperty = await api.listAssetForSale(property.id);

      // Update the property in the local state
      setProperties((prevProperties) =>
        prevProperties.map((p) => (p.id === property.id ? updatedProperty : p)),
      );

      console.log(`Asset ${property.id} listed for sale successfully`);
    } catch (error) {
      console.error("Error listing asset for sale:", error);
      alert("Failed to list asset for sale. Please try again.");
    }
  };

  const confirmDelete = () => {
    if (selectedProperty) {
      // Update the properties state to remove the property
      setProperties((prevProperties) =>
        prevProperties.filter((p) => p.id !== selectedProperty.id),
      );

      // In real app, delete property from backend here
      setIsDeleteDialogOpen(false);
      setSelectedProperty(null);
    }
  };

  return (
    <div className="flex-1 overflow-y-auto">
      {/* Header */}
      <div className="bg-white border-b border-gray-200 px-4 py-3">
        <h1 className="text-2xl font-semibold text-gray-900">My Properties</h1>
        <p className="text-sm text-gray-600">
          Manage and track your property listings
        </p>
      </div>
      <div className="p-0">
        <div className="flex flex-col lg:flex-row lg:items-center justify-between gap-4 mb-6">
          <div>
            <h2 className="text-2xl font-bold text-gray-900">
              Properties ({filteredProperties.length})
            </h2>
            <p className="text-gray-600 mt-1">Manage your property listings</p>
          </div>

          <div className="flex flex-col sm:flex-row gap-4 items-start sm:items-center">
            <div className="flex gap-2">
              <Input
                placeholder="Search properties..."
                value={filters.search}
                onChange={(e) =>
                  setFilters({ ...filters, search: e.target.value })
                }
                className="w-48"
              />
              <Select
                value={filters.status}
                onValueChange={(value) =>
                  setFilters({ ...filters, status: value })
                }
              >
                <SelectTrigger className="w-32">
                  <SelectValue placeholder="Status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Status</SelectItem>
                  <SelectItem value="active">Active</SelectItem>
                  <SelectItem value="inactive">Inactive</SelectItem>
                  <SelectItem value="under_review">Under Review</SelectItem>
                </SelectContent>
              </Select>
              <Select
                value={filters.listingIntent}
                onValueChange={(value) =>
                  setFilters({ ...filters, listingIntent: value })
                }
              >
                <SelectTrigger className="w-32">
                  <SelectValue placeholder="Intent" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Intent</SelectItem>
                  <SelectItem value="sale">Sale</SelectItem>
                  <SelectItem value="rent">Rent</SelectItem>
                  <SelectItem value="urgent-sale">Urgent Sale</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <Link href="/seller/dashboard/add-property">
              <Button>
                <Plus className="mr-2 h-4 w-4" />
                Add Property
              </Button>
            </Link>
            <Dialog
              open={isAddAssetModalOpen}
              onOpenChange={setIsAddAssetModalOpen}
            >
              <DialogTrigger asChild>
                <Button variant="outline">
                  <Plus className="mr-2 h-4 w-4" />
                  Add to Assets
                </Button>
              </DialogTrigger>
              <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
                <DialogHeader>
                  <DialogTitle>Add Property to Assets</DialogTitle>
                </DialogHeader>
                <AddAssetModal onClose={() => setIsAddAssetModalOpen(false)} />
              </DialogContent>
            </Dialog>
          </div>
        </div>

        {/* Properties Grid */}
        {filteredProperties.length > 0 ? (
          <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
            {filteredProperties.map((property) => (
              <SellerPropertyCard
                key={property.id}
                property={property}
                onEdit={handleEdit}
                onPause={handlePause}
                onDelete={handleDelete}
                onView={handleView}
                onListForSale={handleListForSale}
              />
            ))}
          </div>
        ) : (
          <div className="text-center py-12">
            <svg
              className="mx-auto w-16 h-16 text-gray-400 mb-4"
              fill="none"
              stroke="currentColor"
              viewBox="0 0 24 24"
            >
              <path
                strokeLinecap="round"
                strokeLinejoin="round"
                strokeWidth={1}
                d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"
              />
            </svg>
            <h3 className="text-lg font-medium text-gray-900 mb-2">
              No properties found
            </h3>
            <p className="text-gray-600 mb-4">
              Try adjusting your filters or search terms
            </p>
            <button
              onClick={() =>
                setFilters({
                  status: "all",
                  listingIntent: "all",
                  city: "",
                  search: "",
                })
              }
              className="text-blue-600 hover:underline"
            >
              Clear all filters
            </button>
          </div>
        )}

        {/* Property Details Modal */}
        <PropertyDetailsModal
          property={selectedProperty}
          isOpen={isViewDialogOpen}
          onClose={() => setIsViewDialogOpen(false)}
          onEdit={(id) => {
            const property = properties.find((p) => p.id === id);
            if (property) handleEdit(property);
          }}
          onDelete={(id) => {
            const property = properties.find((p) => p.id === id);
            if (property) handleDelete(property);
          }}
        />

        {/* Edit Property Dialog */}
        <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
          <DialogContent className="max-w-3xl">
            <DialogHeader>
              <DialogTitle>Edit Property</DialogTitle>
            </DialogHeader>
            {selectedProperty && (
              <form
                className="space-y-4"
                onSubmit={(e) => {
                  e.preventDefault();

                  // Update the properties state with new data
                  setProperties((prevProperties) =>
                    prevProperties.map((p) =>
                      p.id === selectedProperty.id
                        ? {
                            ...p,
                            title: editForm.title,
                            location: editForm.location,
                            status: editForm.status,
                            price: Number(editForm.price),
                            description: editForm.description,
                          }
                        : p,
                    ),
                  );

                  // In real app, submit updated data to backend here
                  setIsEditDialogOpen(false);
                }}
              >
                <div>
                  <Label htmlFor="title">Property Title</Label>
                  <Input
                    id="title"
                    value={editForm.title}
                    onChange={(e) =>
                      setEditForm({ ...editForm, title: e.target.value })
                    }
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="location">Location</Label>
                  <Input
                    id="location"
                    value={editForm.location}
                    onChange={(e) =>
                      setEditForm({ ...editForm, location: e.target.value })
                    }
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="status">Status</Label>
                  <Select
                    value={editForm.status}
                    onValueChange={(value) =>
                      setEditForm({ ...editForm, status: value })
                    }
                  >
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="active">Active</SelectItem>
                      <SelectItem value="inactive">Inactive</SelectItem>
                      <SelectItem value="under_review">Under Review</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div>
                  <Label htmlFor="price">Price</Label>
                  <Input
                    id="price"
                    type="number"
                    value={editForm.price}
                    onChange={(e) =>
                      setEditForm({ ...editForm, price: e.target.value })
                    }
                    required
                  />
                </div>
                <div>
                  <Label htmlFor="description">Description</Label>
                  <Textarea
                    id="description"
                    value={editForm.description}
                    onChange={(e) =>
                      setEditForm({ ...editForm, description: e.target.value })
                    }
                    rows={3}
                  />
                </div>
                <div className="flex justify-end gap-2">
                  <Button
                    type="button"
                    variant="outline"
                    onClick={() => setIsEditDialogOpen(false)}
                  >
                    Cancel
                  </Button>
                  <Button type="submit">Save</Button>
                </div>
              </form>
            )}
          </DialogContent>
        </Dialog>

        {/* Delete Confirmation Dialog */}
        <AlertDialog
          open={isDeleteDialogOpen}
          onOpenChange={setIsDeleteDialogOpen}
        >
          <AlertDialogTrigger />
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Confirm Delete</AlertDialogTitle>
              <AlertDialogDescription>
                Are you sure you want to delete the property "
                {selectedProperty?.title}"? This action cannot be undone.
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Cancel</AlertDialogCancel>
              <AlertDialogAction onClick={confirmDelete}>
                Delete
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
      </div>
    </div>
  );
}
