

import { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { User, Settings as SettingsIcon, Bell, Shield, Edit, Save, X } from "lucide-react";

export default function Settings() {
  const [profile, setProfile] = useState({
    full_name: '',
    email: '',
    phone_number: '',
    date_of_birth: '',
    gender: '',
    ownership_type: '',
    property_count: 0,
    property_types: [],
    years_of_experience: 0,
    total_portfolio_value_range: '',
    selling_method: '',
    selling_timeline: '',
    price_flexibility: false,
    preferred_communication: [],
    availability_hours: '',
    bio: '',
    profile_image_url: '',
    portfolio_summary: '',
    marketing_consent: false,
    status: '',
    verification_level: '',
    kyc_status: '',
    total_properties_listed: 0,
    total_properties_sold: 0,
    total_revenue_generated: 0,
    average_selling_time_days: 0
  });
  const [propertyStats, setPropertyStats] = useState({
    total_listed: 0,
    total_sold: 0
  });
  const [isEditing, setIsEditing] = useState(false);
  const [editedProfile, setEditedProfile] = useState({});
  const [saving, setSaving] = useState(false);

  useEffect(() => {
    fetchProfile();
    fetchPropertyStats();
  }, []);

  const fetchProfile = async () => {
    try {
      const token = localStorage.getItem('sellerToken');
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/seller-profile/me`, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      if (response.ok) {
        const data = await response.json();
        setProfile(data);
      }
    } catch (error) {
      console.error('Error fetching profile:', error);
    }
  };

  const handleEdit = () => {
    setEditedProfile({...profile});
    setIsEditing(true);
  };

  const handleCancel = () => {
    setEditedProfile({});
    setIsEditing(false);
  };

  const handleSave = async () => {
    setSaving(true);
    try {
      const token = localStorage.getItem('sellerToken');
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/seller-profile/me`, {
        method: 'PUT',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify(editedProfile)
      });
      
      if (response.ok) {
        const updatedProfile = await response.json();
        setProfile(updatedProfile);
        setIsEditing(false);
        setEditedProfile({});
      }
    } catch (error) {
      console.error('Error updating profile:', error);
    } finally {
      setSaving(false);
    }
  };

  const handleInputChange = (field, value) => {
    setEditedProfile(prev => ({ ...prev, [field]: value }));
  };

  const fetchPropertyStats = async () => {
    try {
      const token = localStorage.getItem('sellerToken');
      const userId = localStorage.getItem('sellerId');
      
      // Fetch all properties for this seller using listed_by_id filter
      const response = await fetch(`${import.meta.env.VITE_API_BASE || 'http://localhost:8090/api'}/properties?listed_by_id=${userId}&limit=1000`, {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      
      if (response.ok) {
        const data = await response.json();
        const properties = data.properties || [];
        const totalListed = properties.length;
        const totalSold = properties.filter(p => p.status === 'sold').length;
        
        setPropertyStats({
          total_listed: totalListed,
          total_sold: totalSold
        });
      }
    } catch (error) {
      console.error('Error fetching property stats:', error);
    }
  };
  return (
    <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
      <div className="space-y-6">
        <div>
          <h1 className="text-3xl font-bold text-foreground">Profile & Settings</h1>
          <p className="text-muted-foreground mt-1">Manage your account settings and preferences</p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <User className="w-5 h-5" />
                Personal Information
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div>
                  <label className="text-sm font-medium">Full Name</label>
                  {isEditing ? (
                    <Input 
                      value={editedProfile.full_name || ''} 
                      onChange={(e) => handleInputChange('full_name', e.target.value)}
                      className="mt-1"
                    />
                  ) : (
                    <p className="text-muted-foreground">{profile.full_name || 'Not provided'}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Email</label>
                  <p className="text-muted-foreground">{profile.email || localStorage.getItem('sellerEmail') || 'Not provided'}</p>
                </div>
                <div>
                  <label className="text-sm font-medium">Phone</label>
                  {isEditing ? (
                    <Input 
                      value={editedProfile.phone_number || ''} 
                      onChange={(e) => handleInputChange('phone_number', e.target.value)}
                      className="mt-1"
                    />
                  ) : (
                    <p className="text-muted-foreground">{profile.phone_number || 'Not provided'}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Date of Birth</label>
                  {isEditing ? (
                    <Input 
                      type="date"
                      value={editedProfile.date_of_birth || ''} 
                      onChange={(e) => handleInputChange('date_of_birth', e.target.value)}
                      className="mt-1"
                    />
                  ) : (
                    <p className="text-muted-foreground">{profile.date_of_birth || 'Not provided'}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Gender</label>
                  {isEditing ? (
                    <Select value={editedProfile.gender || ''} onValueChange={(value) => handleInputChange('gender', value)}>
                      <SelectTrigger className="mt-1">
                        <SelectValue placeholder="Select gender" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="male">Male</SelectItem>
                        <SelectItem value="female">Female</SelectItem>
                        <SelectItem value="other">Other</SelectItem>
                      </SelectContent>
                    </Select>
                  ) : (
                    <p className="text-muted-foreground">{profile.gender || 'Not provided'}</p>
                  )}
                </div>
                <div className="flex gap-2">
                  {isEditing ? (
                    <>
                      <Button onClick={handleSave} disabled={saving} className="flex items-center gap-1">
                        <Save className="w-4 h-4" />
                        {saving ? 'Saving...' : 'Save'}
                      </Button>
                      <Button variant="outline" onClick={handleCancel}>
                        <X className="w-4 h-4" />
                        Cancel
                      </Button>
                    </>
                  ) : (
                    <Button onClick={handleEdit} className="flex items-center gap-1">
                      <Edit className="w-4 h-4" />
                      Edit Profile
                    </Button>
                  )}
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <SettingsIcon className="w-5 h-5" />
                Property Portfolio
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div>
                  <label className="text-sm font-medium">Ownership Type</label>
                  {isEditing ? (
                    <Select value={editedProfile.ownership_type || ''} onValueChange={(value) => handleInputChange('ownership_type', value)}>
                      <SelectTrigger className="mt-1">
                        <SelectValue placeholder="Select ownership type" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="individual">Individual</SelectItem>
                        <SelectItem value="joint">Joint</SelectItem>
                        <SelectItem value="company">Company</SelectItem>
                        <SelectItem value="trust">Trust</SelectItem>
                      </SelectContent>
                    </Select>
                  ) : (
                    <p className="text-muted-foreground">{profile.ownership_type || 'Not provided'}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Property Count</label>
                  {isEditing ? (
                    <Input 
                      type="number"
                      value={editedProfile.property_count || ''} 
                      onChange={(e) => handleInputChange('property_count', parseInt(e.target.value) || 0)}
                      className="mt-1"
                    />
                  ) : (
                    <p className="text-muted-foreground">{profile.property_count || 0}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Property Types</label>
                  {isEditing ? (
                    <Input 
                      value={editedProfile.property_types?.join(', ') || ''} 
                      onChange={(e) => handleInputChange('property_types', e.target.value.split(', ').filter(Boolean))}
                      placeholder="residential, commercial, land"
                      className="mt-1"
                    />
                  ) : (
                    <p className="text-muted-foreground">{profile.property_types?.join(', ') || 'Not provided'}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Experience (Years)</label>
                  {isEditing ? (
                    <Input 
                      type="number"
                      value={editedProfile.years_of_experience || ''} 
                      onChange={(e) => handleInputChange('years_of_experience', parseInt(e.target.value) || 0)}
                      className="mt-1"
                    />
                  ) : (
                    <p className="text-muted-foreground">{profile.years_of_experience || 0}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Portfolio Value Range</label>
                  {isEditing ? (
                    <Select value={editedProfile.total_portfolio_value_range || ''} onValueChange={(value) => handleInputChange('total_portfolio_value_range', value)}>
                      <SelectTrigger className="mt-1">
                        <SelectValue placeholder="Select value range" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="under_50L">Under ₹50L</SelectItem>
                        <SelectItem value="50L_1Cr">₹50L - ₹1Cr</SelectItem>
                        <SelectItem value="1Cr_5Cr">₹1Cr - ₹5Cr</SelectItem>
                        <SelectItem value="above_5Cr">Above ₹5Cr</SelectItem>
                      </SelectContent>
                    </Select>
                  ) : (
                    <p className="text-muted-foreground">{profile.total_portfolio_value_range || 'Not provided'}</p>
                  )}
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Bell className="w-5 h-5" />
                Selling Preferences
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div>
                  <label className="text-sm font-medium">Selling Method</label>
                  {isEditing ? (
                    <Select value={editedProfile.selling_method || ''} onValueChange={(value) => handleInputChange('selling_method', value)}>
                      <SelectTrigger className="mt-1">
                        <SelectValue placeholder="Select selling method" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="direct">Direct</SelectItem>
                        <SelectItem value="agent">Agent</SelectItem>
                        <SelectItem value="auction">Auction</SelectItem>
                        <SelectItem value="flexible">Flexible</SelectItem>
                      </SelectContent>
                    </Select>
                  ) : (
                    <p className="text-muted-foreground">{profile.selling_method || 'Not provided'}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Selling Timeline</label>
                  {isEditing ? (
                    <Select value={editedProfile.selling_timeline || ''} onValueChange={(value) => handleInputChange('selling_timeline', value)}>
                      <SelectTrigger className="mt-1">
                        <SelectValue placeholder="Select timeline" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="immediate">Immediate</SelectItem>
                        <SelectItem value="3_months">3 Months</SelectItem>
                        <SelectItem value="6_months">6 Months</SelectItem>
                        <SelectItem value="1_year">1 Year</SelectItem>
                        <SelectItem value="flexible">Flexible</SelectItem>
                      </SelectContent>
                    </Select>
                  ) : (
                    <p className="text-muted-foreground">{profile.selling_timeline || 'Not provided'}</p>
                  )}
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Price Flexibility</span>
                  {isEditing ? (
                    <Checkbox 
                      checked={editedProfile.price_flexibility || false}
                      onCheckedChange={(checked) => handleInputChange('price_flexibility', checked)}
                    />
                  ) : (
                    <Badge variant={profile.price_flexibility ? "default" : "outline"}>
                      {profile.price_flexibility ? 'Yes' : 'No'}
                    </Badge>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Preferred Communication</label>
                  {isEditing ? (
                    <Input 
                      value={editedProfile.preferred_communication?.join(', ') || ''} 
                      onChange={(e) => handleInputChange('preferred_communication', e.target.value.split(', ').filter(Boolean))}
                      placeholder="email, phone, whatsapp"
                      className="mt-1"
                    />
                  ) : (
                    <p className="text-muted-foreground">{profile.preferred_communication?.join(', ') || 'Not provided'}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Availability Hours</label>
                  {isEditing ? (
                    <Input 
                      value={editedProfile.availability_hours || ''} 
                      onChange={(e) => handleInputChange('availability_hours', e.target.value)}
                      placeholder="9 AM - 6 PM"
                      className="mt-1"
                    />
                  ) : (
                    <p className="text-muted-foreground">{profile.availability_hours || 'Not provided'}</p>
                  )}
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Shield className="w-5 h-5" />
                Account Status & Verification
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Account Status</span>
                  <Badge variant={profile.status === 'active' ? 'default' : 'outline'}>
                    {profile.status || 'Unknown'}
                  </Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Verification Level</span>
                  <Badge variant="secondary">{profile.verification_level || 'Basic'}</Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">KYC Status</span>
                  <Badge variant={profile.kyc_status === 'verified' ? 'default' : 'outline'}>
                    {profile.kyc_status || 'Pending'}
                  </Badge>
                </div>
                <div className="flex items-center justify-between">
                  <span className="text-sm font-medium">Marketing Consent</span>
                  {isEditing ? (
                    <Checkbox 
                      checked={editedProfile.marketing_consent || false}
                      onCheckedChange={(checked) => handleInputChange('marketing_consent', checked)}
                    />
                  ) : (
                    <Badge variant={profile.marketing_consent ? 'default' : 'outline'}>
                      {profile.marketing_consent ? 'Yes' : 'No'}
                    </Badge>
                  )}
                </div>
              </div>
            </CardContent>
          </Card>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Card>
            <CardHeader>
              <CardTitle>Performance Metrics</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div>
                  <label className="text-sm font-medium">Properties Listed</label>
                  <p className="text-muted-foreground">{propertyStats.total_listed}</p>
                </div>
                <div>
                  <label className="text-sm font-medium">Properties Sold</label>
                  <p className="text-muted-foreground">{propertyStats.total_sold}</p>
                </div>
                <div>
                  <label className="text-sm font-medium">Total Revenue</label>
                  <p className="text-muted-foreground">₹{profile.total_revenue_generated || 0}</p>
                </div>
                <div>
                  <label className="text-sm font-medium">Avg. Selling Time</label>
                  <p className="text-muted-foreground">{profile.average_selling_time_days || 0} days</p>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card>
            <CardHeader>
              <CardTitle>Profile & Bio</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div>
                  <label className="text-sm font-medium">Bio</label>
                  {isEditing ? (
                    <Textarea 
                      value={editedProfile.bio || ''} 
                      onChange={(e) => handleInputChange('bio', e.target.value)}
                      className="mt-1"
                      rows={3}
                    />
                  ) : (
                    <p className="text-muted-foreground text-sm">{profile.bio || 'No bio provided'}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Portfolio Summary</label>
                  {isEditing ? (
                    <Textarea 
                      value={editedProfile.portfolio_summary || ''} 
                      onChange={(e) => handleInputChange('portfolio_summary', e.target.value)}
                      className="mt-1"
                      rows={3}
                    />
                  ) : (
                    <p className="text-muted-foreground text-sm">{profile.portfolio_summary || 'No summary provided'}</p>
                  )}
                </div>
                <div>
                  <label className="text-sm font-medium">Profile Image URL</label>
                  {isEditing ? (
                    <Input 
                      value={editedProfile.profile_image_url || ''} 
                      onChange={(e) => handleInputChange('profile_image_url', e.target.value)}
                      placeholder="https://example.com/image.jpg"
                      className="mt-1"
                    />
                  ) : (
                    <p className="text-muted-foreground text-sm">{profile.profile_image_url || 'Not uploaded'}</p>
                  )}
                </div>

              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}
