

import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Zap, Calendar, Users, Clock, CheckCircle, XCircle, MapPin, Phone, Video } from "lucide-react";
import { mockBookings, getPropertyById, getBuyerById } from "../lib/mock-data";
import { useState } from "react";

export default function UrgentVisits() {
  const [urgentVisits] = useState(() => 
    mockBookings.filter(b => b.type === "visit" && (b.status === "pending" || b.status === "confirmed"))
  );

  const isVideoCallTime = (visit: any) => {
    if (!visit.scheduledDate) return false;
    
    const now = new Date();
    const visitDate = new Date(visit.scheduledDate);
    const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const scheduleDate = new Date(visitDate.getFullYear(), visitDate.getMonth(), visitDate.getDate());
    
    // Only show on the scheduled date
    if (today.getTime() !== scheduleDate.getTime()) return false;
    
    const currentHour = now.getHours();
    const visitHour = visitDate.getHours();
    
    // Show video call button 1 hour before and during the scheduled time
    return currentHour >= (visitHour - 1) && currentHour <= (visitHour + 1);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case "pending":
        return "bg-yellow-100 text-yellow-800";
      case "confirmed":
        return "bg-green-100 text-green-800";
      case "cancelled":
        return "bg-red-100 text-red-800";
      default:
        return "bg-gray-100 text-gray-800";
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "pending":
        return <Clock className="w-4 h-4" />;
      case "confirmed":
        return <CheckCircle className="w-4 h-4" />;
      case "cancelled":
        return <XCircle className="w-4 h-4" />;
      default:
        return <Clock className="w-4 h-4" />;
    }
  };

  return (
    <div className="flex-1 space-y-4 p-4 md:p-8 pt-6">
      <div className="flex items-center justify-between space-y-2">
        <h2 className="text-3xl font-bold tracking-tight">Urgent / Flash Sale Visits</h2>
      </div>
      <div className="space-y-6">

        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <Card className="bg-gradient-to-br from-red-50 to-red-100 border-red-200">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-red-700">
                Flash Sale Visits
                <Zap className="w-5 h-5 text-red-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-red-800">4</div>
              <p className="text-xs text-red-600 mt-1">this week</p>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-orange-50 to-orange-100 border-orange-200">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-orange-700">
                Urgent Bookings
                <Calendar className="w-5 h-5 text-orange-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-orange-800">6</div>
              <p className="text-xs text-orange-600 mt-1">pending approval</p>
            </CardContent>
          </Card>

          <Card className="bg-gradient-to-br from-blue-50 to-blue-100 border-blue-200">
            <CardHeader className="pb-3">
              <CardTitle className="flex items-center justify-between text-sm font-medium text-blue-700">
                Total Visitors
                <Users className="w-5 h-5 text-blue-600" />
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="text-3xl font-bold text-blue-800">23</div>
              <p className="text-xs text-blue-600 mt-1">this month</p>
            </CardContent>
          </Card>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Recent Urgent Visits</CardTitle>
          </CardHeader>
          <CardContent>
            {urgentVisits.length === 0 ? (
              <div className="text-sm text-muted-foreground">
                No urgent visit requests found.
              </div>
            ) : (
              <div className="space-y-4">
                {urgentVisits.map((visit) => {
                  const property = getPropertyById(visit.propertyId);
                  const buyer = getBuyerById(visit.buyerId);
                  return (
                    <Card key={visit.id} className="border">
                      <CardContent className="p-4">
                        <div className="flex justify-between items-center mb-2">
                          <h3 className="text-lg font-semibold">{property?.title || "Unknown Property"}</h3>
                          <Badge className={getStatusColor(visit.status)} variant="outline">
                            {getStatusIcon(visit.status)}
                            <span className="ml-1">{visit.status.charAt(0).toUpperCase() + visit.status.slice(1)}</span>
                          </Badge>
                        </div>
                        <p className="text-sm text-muted-foreground mb-1">
                          <MapPin className="inline w-4 h-4 mr-1" />
                          {property?.location || "Unknown Location"}
                        </p>
                        <p className="text-sm mb-1">
                          <strong>Buyer:</strong> {buyer?.firstName} {buyer?.lastName}
                        </p>
                        <p className="text-sm mb-1">
                          <strong>Scheduled Date:</strong> {visit.scheduledDate.toLocaleString()}
                        </p>
                        {visit.notes && (
                          <p className="text-sm mb-1">
                            <strong>Notes:</strong> {visit.notes}
                          </p>
                        )}
                        <div className="flex gap-2 mt-2">
                          <Button size="sm" variant="outline">
                            <Phone className="w-4 h-4 mr-1" />
                            Contact Buyer
                          </Button>
                          {visit.status === 'confirmed' && isVideoCallTime(visit) && (
                            <Button 
                              size="sm" 
                              className="bg-blue-600 hover:bg-blue-700 text-white"
                              onClick={() => {
                                const roomId = `NAL_Property_${visit.propertyId}_Visit_${visit.id}`;
                                window.open(`https://meet.jit.si/${roomId}`, '_blank');
                              }}
                            >
                              <Video className="w-4 h-4 mr-1" />
                              Join Call
                            </Button>
                          )}
                        </div>
                      </CardContent>
                    </Card>
                  );
                })}
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
